#include <solomon/libs/cpp/selectors/selectors.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

using namespace NSolomon;

namespace {

template<typename T>
void TestAll(TVector<TStringBuf>& values, T&& tester) {
    for (auto value: values) {
        try {
            tester(value);
        } catch (...) {
            FAIL() << "FAIL on [" << value << "], message: " << CurrentExceptionMessage();
        }
    }
}

} // namespace

TEST(TSelectorsTest, ParseEmpty) {
    TVector<TStringBuf> values = {
        TStringBuf(""),
        TStringBuf(" "),
        TStringBuf("  "),
        TStringBuf("\t\t\t"),
        TStringBuf("\n\n\n"),
    };

    TestAll(values, [](TStringBuf value) {
        ASSERT_THROW(ParseSelector(value), TInvalidSelectorsFormat);
    });
}

TEST(TSelectorsTest, ParseExact) {
    TVector<TStringBuf> values = {
        TStringBuf("host=solomon-fetcher-sas-00"),
        TStringBuf("host = solomon-fetcher-sas-00"),
        TStringBuf(" host = solomon-fetcher-sas-00 "),
        TStringBuf("    host    =   solomon-fetcher-sas-00    "),
        TStringBuf("'host' = 'solomon-fetcher-sas-00'"),
        TStringBuf(" 'host' = 'solomon-fetcher-sas-00' "),
        TStringBuf("\"host\" = \"solomon-fetcher-sas-00\""),
        TStringBuf(" \"host\" = \"solomon-fetcher-sas-00\" "),
        TStringBuf("'host' = \"solomon-fetcher-sas-00\""),
        TStringBuf("host==solomon-fetcher-sas-00"),
        TStringBuf(" host == solomon-fetcher-sas-00 "),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_TRUE(s.IsExact());

        EXPECT_EQ(EMatcherType::EXACT, s.Type());
        EXPECT_EQ("solomon-fetcher-sas-00", s.Pattern());
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-00"));

        EXPECT_EQ("'host' == 'solomon-fetcher-sas-00'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseNotExact) {
    TVector<TStringBuf> values = {
        TStringBuf("host!=solomon-fetcher-sas-00"),
        TStringBuf("host != solomon-fetcher-sas-00"),
        TStringBuf(" host != solomon-fetcher-sas-00 "),
        TStringBuf("    host    !=   solomon-fetcher-sas-00    "),
        TStringBuf("'host' != 'solomon-fetcher-sas-00'"),
        TStringBuf(" 'host' != 'solomon-fetcher-sas-00' "),
        TStringBuf("\"host\" != \"solomon-fetcher-sas-00\""),
        TStringBuf(" \"host\" != \"solomon-fetcher-sas-00\" "),
        TStringBuf("'host' != \"solomon-fetcher-sas-00\""),
        TStringBuf("host!==solomon-fetcher-sas-00"),
        TStringBuf(" host !== solomon-fetcher-sas-00 "),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_TRUE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::EXACT, s.Type());
        EXPECT_EQ("solomon-fetcher-sas-00", s.Pattern());
        EXPECT_FALSE(s.Match("solomon-fetcher-sas-00"));
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-xx"));

        EXPECT_EQ("'host' !== 'solomon-fetcher-sas-00'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseGlob) {
    TVector<TStringBuf> values = {
        TStringBuf("host=solomon-fetcher-*"),
        TStringBuf("host = solomon-fetcher-*"),
        TStringBuf(" host = solomon-fetcher-* "),
        TStringBuf("    host    =   solomon-fetcher-*    "),
        TStringBuf("'host' = 'solomon-fetcher-*'"),
        TStringBuf(" 'host' = 'solomon-fetcher-*' "),
        TStringBuf("\"host\" = \"solomon-fetcher-*\""),
        TStringBuf(" \"host\" = \"solomon-fetcher-*\" "),
        TStringBuf("'host' = \"solomon-fetcher-*\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::GLOB, s.Type());
        EXPECT_EQ("solomon-fetcher-*", s.Pattern());
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-00"));
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-01"));
        EXPECT_TRUE(s.Match("solomon-fetcher-vla-02"));

        EXPECT_EQ("'host' = 'solomon-fetcher-*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseNotGlob) {
    TVector<TStringBuf> values = {
        TStringBuf("host!=solomon-fetcher-*"),
        TStringBuf("host != solomon-fetcher-*"),
        TStringBuf(" host != solomon-fetcher-* "),
        TStringBuf("    host    !=   solomon-fetcher-*    "),
        TStringBuf("'host' != 'solomon-fetcher-*'"),
        TStringBuf(" 'host' != 'solomon-fetcher-*' "),
        TStringBuf("\"host\" != \"solomon-fetcher-*\""),
        TStringBuf(" \"host\" != \"solomon-fetcher-*\" "),
        TStringBuf("'host' != \"solomon-fetcher-*\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_TRUE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::GLOB, s.Type());
        EXPECT_EQ("solomon-fetcher-*", s.Pattern());
        EXPECT_FALSE(s.Match("solomon-fetcher-sas-00"));
        EXPECT_FALSE(s.Match("solomon-fetcher-sas-01"));
        EXPECT_FALSE(s.Match("solomon-fetcher-vla-02"));
        EXPECT_TRUE(s.Match("solomon-storage-vla-00"));

        EXPECT_EQ("'host' != 'solomon-fetcher-*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseMulti) {
    TVector<TStringBuf> values = {
        TStringBuf("host=solomon-fetcher-*|solomon-stockpile-sas-00|-"),
        TStringBuf("host = solomon-fetcher-*|solomon-stockpile-sas-00|-"),
        TStringBuf(" host = solomon-fetcher-*|solomon-stockpile-sas-00|- "),
        TStringBuf("    host    =   solomon-fetcher-*|solomon-stockpile-sas-00|-    "),
        TStringBuf("'host' = 'solomon-fetcher-*|solomon-stockpile-sas-00|-'"),
        TStringBuf(" 'host' = 'solomon-fetcher-*|solomon-stockpile-sas-00|-' "),
        TStringBuf("\"host\" = \"solomon-fetcher-*|solomon-stockpile-sas-00|-\""),
        TStringBuf(" \"host\" = \"solomon-fetcher-*|solomon-stockpile-sas-00|-\" "),
        TStringBuf("'host' = \"solomon-fetcher-*|solomon-stockpile-sas-00|-\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::MULTI, s.Type());
        EXPECT_EQ("solomon-fetcher-*|solomon-stockpile-sas-00|-", s.Pattern());
        EXPECT_EQ("'host' = 'solomon-fetcher-*|solomon-stockpile-sas-00|-'", ToString(s));

        auto* mm = s.MultiMatcherPtr();
        EXPECT_TRUE(mm);
        EXPECT_EQ(EMatcherType::MULTI, mm->Type());
        EXPECT_EQ(3u, mm->Size());

        {
            auto* m = mm->Get(0);
            ASSERT_TRUE(m);
            EXPECT_EQ(EMatcherType::GLOB, m->Type());
            EXPECT_EQ("solomon-fetcher-*", m->Pattern());
            EXPECT_TRUE(m->Match("solomon-fetcher-sas-00"));
            EXPECT_FALSE(m->Match("solomon-coremon-sas-00"));
        }
        {
            auto* m = mm->Get(1);
            ASSERT_TRUE(m);
            EXPECT_EQ(EMatcherType::EXACT, m->Type());
            EXPECT_EQ("solomon-stockpile-sas-00", m->Pattern());
            EXPECT_TRUE(m->Match("solomon-stockpile-sas-00"));
            EXPECT_FALSE(m->Match("solomon-stockpile-sas-01"));
        }
        {
            auto* m = mm->Get(2);
            ASSERT_TRUE(m);
            EXPECT_EQ(EMatcherType::ABSENT, m->Type());
            EXPECT_EQ("-", m->Pattern());
        }
    });
}

TEST(TSelectorsTest, ParseRegex) {
    TVector<TStringBuf> values = {
        TStringBuf("host=~solomon-[a-z]+-sas-[0-9]+"),
        TStringBuf("host =~ solomon-[a-z]+-sas-[0-9]+"),
        TStringBuf(" host =~ solomon-[a-z]+-sas-[0-9]+ "),
        TStringBuf("    host    =~   solomon-[a-z]+-sas-[0-9]+    "),
        TStringBuf("'host' =~ 'solomon-[a-z]+-sas-[0-9]+'"),
        TStringBuf(" 'host' =~ 'solomon-[a-z]+-sas-[0-9]+' "),
        TStringBuf("\"host\" =~ \"solomon-[a-z]+-sas-[0-9]+\""),
        TStringBuf(" \"host\" =~ \"solomon-[a-z]+-sas-[0-9]+\" "),
        TStringBuf("'host' =~ \"solomon-[a-z]+-sas-[0-9]+\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::REGEX, s.Type());
        EXPECT_EQ("solomon-[a-z]+-sas-[0-9]+", s.Pattern());
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-00"));
        EXPECT_TRUE(s.Match("solomon-storage-sas-01"));
        EXPECT_FALSE(s.Match("solomon-fetcher-vla-00"));

        EXPECT_EQ("'host' =~ 'solomon-[a-z]+-sas-[0-9]+'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseRegexAsExact) {
    TVector<TStringBuf> values = {
            TStringBuf("host=~^solomon-fetcher-sas-000$"),
            TStringBuf("host =~ solomon-fetcher-sas-000"),
            TStringBuf(" host =~ ^solomon-fetcher-sas-000$ "),
            TStringBuf("    host    =~   solomon-fetcher-sas-000    "),
            TStringBuf("'host' =~ '^solomon-fetcher-sas-000$'"),
            TStringBuf(" 'host' =~ 'solomon-fetcher-sas-000' "),
            TStringBuf("\"host\" =~ \"^solomon-fetcher-sas-000$\""),
            TStringBuf(" \"host\" =~ \"solomon-fetcher-sas-000\" "),
            TStringBuf("'host' =~ \"^solomon-fetcher-sas-000$\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_TRUE(s.IsExact());

        EXPECT_EQ(EMatcherType::EXACT, s.Type());
        EXPECT_EQ("solomon-fetcher-sas-000", s.Pattern());
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-000"));
        EXPECT_FALSE(s.Match("solomon-fetcher-sas-001"));

        EXPECT_EQ("'host' == 'solomon-fetcher-sas-000'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseRegexAsGlob) {
    TVector<TStringBuf> values = {
            TStringBuf("host=~^.*fetcher.*$"),
            TStringBuf("host =~ .*fetcher.*"),
            TStringBuf(" host =~ ^.*fetcher.*$ "),
            TStringBuf("    host    =~   .*fetcher.*    "),
            TStringBuf("'host' =~ '^.*fetcher.*$'"),
            TStringBuf(" 'host' =~ '.*fetcher.*' "),
            TStringBuf("\"host\" =~ \"^.*fetcher.*$\""),
            TStringBuf(" \"host\" =~ \".*fetcher.*\" "),
            TStringBuf("'host' =~ \"^.*fetcher.*$\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::GLOB, s.Type());
        EXPECT_EQ("*fetcher*", s.Pattern());
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-000"));
        EXPECT_FALSE(s.Match("solomon-stockpile-sas-000"));

        EXPECT_EQ("'host' = '*fetcher*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseRegexAsAny) {
    TVector<TStringBuf> values = {
            TStringBuf("host=~^.*$"),
            TStringBuf("host =~ .*"),
            TStringBuf(" host =~ ^.*$ "),
            TStringBuf("    host    =~   .*    "),
            TStringBuf("'host' =~ '^.*$'"),
            TStringBuf(" 'host' =~ '.*' "),
            TStringBuf("\"host\" =~ \"^.*$\""),
            TStringBuf(" \"host\" =~ \".*\" "),
            TStringBuf("'host' =~ \"^.*$\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::ANY, s.Type());
        EXPECT_EQ("*", s.Pattern());
        EXPECT_TRUE(s.Match("solomon-fetcher-sas-000"));
        EXPECT_TRUE(s.Match("abcdefg"));

        EXPECT_EQ("'host' = '*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseRegexWithNamedGroup) {
    TVector<TStringBuf> values = {
            TStringBuf("signal=~.*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*"),
            TStringBuf("signal =~ .*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*"),
            TStringBuf(" signal =~ .*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.* "),
            TStringBuf("    signal    =~   .*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*    "),
            TStringBuf("'signal' =~ '.*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*'"),
            TStringBuf(" 'signal' =~ '.*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*' "),
            TStringBuf("\"signal\" =~ \".*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*\""),
            TStringBuf(" \"signal\" =~ \".*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*\" "),
            TStringBuf("'signal' =~ \".*stream_nginxtskv-get-responses-(?P<status>\\d+)_summ.*\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("signal", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::REGEX, s.Type());
        EXPECT_EQ(".*stream_nginxtskv-get-responses-(\\d+)_summ.*", s.Pattern());
        EXPECT_TRUE(s.Match("stream_nginxtskv-get-responses-00_summ"));
        EXPECT_TRUE(s.Match("stream_nginxtskv-get-responses-01_summ"));
        EXPECT_FALSE(s.Match("stream_nginxtskv-get-responses-XX_summ"));

        EXPECT_EQ("'signal' =~ '.*stream_nginxtskv-get-responses-(\\d+)_summ.*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseNotRegex) {
    TVector<TStringBuf> values = {
        TStringBuf("host!~solomon-[a-z]+-sas-[0-9]+"),
        TStringBuf("host !~ solomon-[a-z]+-sas-[0-9]+"),
        TStringBuf(" host !~ solomon-[a-z]+-sas-[0-9]+ "),
        TStringBuf("    host    !~   solomon-[a-z]+-sas-[0-9]+    "),
        TStringBuf("'host' !~ 'solomon-[a-z]+-sas-[0-9]+'"),
        TStringBuf(" 'host' !~ 'solomon-[a-z]+-sas-[0-9]+' "),
        TStringBuf("\"host\" !~ \"solomon-[a-z]+-sas-[0-9]+\""),
        TStringBuf(" \"host\" !~ \"solomon-[a-z]+-sas-[0-9]+\" "),
        TStringBuf("'host' !~ \"solomon-[a-z]+-sas-[0-9]+\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_TRUE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::REGEX, s.Type());
        EXPECT_EQ("solomon-[a-z]+-sas-[0-9]+", s.Pattern());
        EXPECT_FALSE(s.Match("solomon-fetcher-sas-00"));
        EXPECT_FALSE(s.Match("solomon-storage-sas-01"));
        EXPECT_TRUE(s.Match("solomon-fetcher-vla-00"));

        EXPECT_EQ("'host' !~ 'solomon-[a-z]+-sas-[0-9]+'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseAny) {
    TVector<TStringBuf> values = {
        TStringBuf("host=*"),
        TStringBuf("host = *"),
        TStringBuf(" host = * "),
        TStringBuf("    host    =   *    "),
        TStringBuf("'host' = '*'"),
        TStringBuf(" 'host' = '*' "),
        TStringBuf("\"host\" = \"*\""),
        TStringBuf(" \"host\" = \"*\" "),
        TStringBuf("'host' = \"*\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::ANY, s.Type());
        EXPECT_EQ("*", s.Pattern());

        EXPECT_EQ("'host' = '*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseNotAny) {
    TVector<TStringBuf> values = {
        TStringBuf("host!=*"),
        TStringBuf("host != *"),
        TStringBuf(" host != * "),
        TStringBuf("    host    !=   *    "),
        TStringBuf("'host' != '*'"),
        TStringBuf(" 'host' != '*' "),
        TStringBuf("\"host\" != \"*\""),
        TStringBuf(" \"host\" != \"*\" "),
        TStringBuf("'host' != \"*\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_TRUE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::ANY, s.Type());
        EXPECT_EQ("*", s.Pattern());

        EXPECT_EQ("'host' != '*'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseAbsent) {
    TVector<TStringBuf> values = {
        TStringBuf("host=-"),
        TStringBuf("host = -"),
        TStringBuf(" host = - "),
        TStringBuf("    host    =   -    "),
        TStringBuf("'host' = '-'"),
        TStringBuf(" 'host' = '-' "),
        TStringBuf("\"host\" = \"-\""),
        TStringBuf(" \"host\" = \"-\" "),
        TStringBuf("'host' = \"-\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::ABSENT, s.Type());
        EXPECT_EQ("-", s.Pattern());

        EXPECT_EQ("'host' = '-'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseNotAbsent) {
    TVector<TStringBuf> values = {
        TStringBuf("host!=-"),
        TStringBuf("host != -"),
        TStringBuf(" host != - "),
        TStringBuf("    host    !=   -    "),
        TStringBuf("'host' != '-'"),
        TStringBuf(" 'host' != '-' "),
        TStringBuf("\"host\" != \"-\""),
        TStringBuf(" \"host\" != \"-\" "),
        TStringBuf("'host' != \"-\""),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelector(value);
        EXPECT_EQ("host", s.Key());
        EXPECT_TRUE(s.Matcher());
        EXPECT_TRUE(s.Negative());
        EXPECT_FALSE(s.IsExact());

        EXPECT_EQ(EMatcherType::ABSENT, s.Type());
        EXPECT_EQ("-", s.Pattern());

        EXPECT_EQ("'host' != '-'", ToString(s));
    });
}

TEST(TSelectorsTest, ParseSelectorsEmpty) {
    TVector<TStringBuf> values = {
        TStringBuf(""),
        TStringBuf(" "),
        TStringBuf("  "),
        TStringBuf("{}"),
        TStringBuf(" {} "),
        TStringBuf(" { } "),
        TStringBuf("\t{\t}\t"),
        TStringBuf("\n{\n}\n"),
    };

    TestAll(values, [](TStringBuf value) {
        auto s = ParseSelectors(value);
        EXPECT_TRUE(s.empty());
    });
}

TEST(TSelectorsTest, ParseSelectorsOne) {
    TVector<TStringBuf> values = {
        TStringBuf("{host=solomon-fetcher-sas-00}"),
        TStringBuf("{ host=solomon-fetcher-sas-00 }"),
        TStringBuf(" { host=solomon-fetcher-sas-00 } "),
        TStringBuf(" { host = solomon-fetcher-sas-00 } "),
        TStringBuf("{'host'='solomon-fetcher-sas-00'}"),
        TStringBuf("{ \"host\"=\"solomon-fetcher-sas-00\" }"),
        TStringBuf(" { 'host'=\"solomon-fetcher-sas-00\" } "),
        TStringBuf(" { \"host\" = 'solomon-fetcher-sas-00' } "),
    };

    TestAll(values, [](TStringBuf value) {
        auto selectors = ParseSelectors(value);
        EXPECT_EQ(1u, selectors.size());

        TSelector& s = selectors[0];
        EXPECT_EQ("host", s.Key());
        EXPECT_EQ("solomon-fetcher-sas-00", s.Pattern());
        EXPECT_EQ(EMatcherType::EXACT, s.Type());
        EXPECT_FALSE(s.Negative());
        EXPECT_TRUE(s.IsExact());

        EXPECT_EQ("{'host' == 'solomon-fetcher-sas-00'}", ToString(selectors));
    });
}

TEST(TSelectorsTest, ParseSelectorsMany) {
    TVector<TStringBuf> values = {
        TStringBuf("{host=solomon-fetcher-sas-00,sensor=~memory_(total|used)}"),
        TStringBuf("{ host=solomon-fetcher-sas-00, sensor=~memory_(total|used) }"),
        TStringBuf(" { host = solomon-fetcher-sas-00 , sensor =~ memory_(total|used) } "),
        TStringBuf(" { 'host' = 'solomon-fetcher-sas-00' , 'sensor' =~ 'memory_(total|used)' } "),
        TStringBuf(" { \"host\" = \"solomon-fetcher-sas-00\" , \"sensor\" =~ \"memory_(total|used)\" } "),
    };

    TestAll(values, [](TStringBuf value) {
        auto selectors = ParseSelectors(value);
        EXPECT_EQ(2u, selectors.size());

        {
            TSelector& s = selectors[0];
            EXPECT_EQ("host", s.Key());
            EXPECT_EQ(EMatcherType::EXACT, s.Type());
            EXPECT_EQ("solomon-fetcher-sas-00", s.Pattern());
            EXPECT_FALSE(s.Negative());
            EXPECT_TRUE(s.IsExact());
        }
        {
            TSelector& s = selectors[1];
            EXPECT_EQ("sensor", s.Key());
            EXPECT_EQ(EMatcherType::REGEX, s.Type());
            EXPECT_EQ("memory_(total|used)", s.Pattern());
            EXPECT_FALSE(s.Negative());
            EXPECT_FALSE(s.IsExact());
        }

        EXPECT_EQ("{'host' == 'solomon-fetcher-sas-00', 'sensor' =~ 'memory_(total|used)'}", ToString(selectors));
    });
}

TEST(TSelectorsTest, ParseOneKeyTwoPatternSelectors) {
    TStringBuf value = "{host!=solomon-fetcher-sas-00, host=solomon-fetcher-sas-*}";

    auto selectors = ParseSelectors(value);
    EXPECT_EQ(2u, selectors.size());

    {
        TSelector& s = selectors[0];
        EXPECT_EQ("host", s.Key());
        EXPECT_EQ("solomon-fetcher-sas-00", s.Pattern());
        EXPECT_TRUE(s.Negative());
        EXPECT_FALSE(s.IsExact());
    }
    {
        TSelector& s = selectors[1];
        EXPECT_EQ("host", s.Key());
        EXPECT_EQ(EMatcherType::GLOB, s.Type());
        EXPECT_EQ("solomon-fetcher-sas-*", s.Pattern());
        EXPECT_FALSE(s.Negative());
        EXPECT_FALSE(s.IsExact());
    }

    EXPECT_EQ("{'host' !== 'solomon-fetcher-sas-00', 'host' = 'solomon-fetcher-sas-*'}", ToString(selectors));
}

TEST(TSelectorsTest, Remove) {
    auto selectors = ParseSelectors("{host=solomon-fetcher-sas-00, sensor=memory_total}");
    ASSERT_EQ(selectors.size(), 2u);

    // remove first
    {
        auto s = selectors;
        ASSERT_TRUE(s.Remove("host"));
        ASSERT_EQ(s.size(), 1u);
        ASSERT_EQ(ToString(s), "{'sensor' == 'memory_total'}");
    }

    // remove last
    {
        auto s = selectors;
        ASSERT_TRUE(s.Remove("sensor"));
        ASSERT_EQ(s.size(), 1u);
        ASSERT_EQ(ToString(s), "{'host' == 'solomon-fetcher-sas-00'}");
    }

    // remove missing
    ASSERT_FALSE(selectors.Remove("missedKey"));
}

TEST(TSelectorsTest, Default) {
    TSelectors selectors;
    ASSERT_TRUE(selectors.empty());
    ASSERT_EQ(selectors.size(), 0u);
}

TEST(TSelectorsTest, Copy) {
    auto selectors = ParseSelectors("{host=solomon-fetcher-sas-00, sensor=memory_total}");
    ASSERT_EQ(selectors.size(), 2u);

    TSelectors copy{selectors};
    copy.Add("cluster", "prod");
    ASSERT_EQ(copy.size(), 3u);
    ASSERT_EQ(ToString(copy), "{'host' == 'solomon-fetcher-sas-00', 'sensor' == 'memory_total', 'cluster' == 'prod'}");

    ASSERT_EQ(selectors.size(), 2u);
    ASSERT_EQ(ToString(selectors), "{'host' == 'solomon-fetcher-sas-00', 'sensor' == 'memory_total'}");
}

TEST(TSelectorsTest, Move) {
    auto selectors = ParseSelectors("{host=solomon-fetcher-sas-00, sensor=memory_total}");
    ASSERT_EQ(selectors.size(), 2u);

    TSelectors copy{std::move(selectors)};
    ASSERT_EQ(copy.size(), 2u);
    ASSERT_EQ(ToString(copy), "{'host' == 'solomon-fetcher-sas-00', 'sensor' == 'memory_total'}");

    ASSERT_EQ(selectors.size(), 0u);  // NOLINT(bugprone-use-after-move)
    ASSERT_EQ(ToString(selectors), "{}");  // NOLINT(bugprone-use-after-move)
}
