function traceplot() {
  const axisHeight = 10;

  const margin = {
    top: 10 + axisHeight,
    right: 20,
    bottom: 10,
    left: 10
  };

  const childColor = "#C6C6D8";
  const parentColor = "#EEE8AA";
  const selectedColor = "#DAA520";
  const generalColor = "#DFDFDF";
  const unfinishedColor = "#F08080"

  let rowHeight = 20;

  function formatDur(x) {
    if (x === 0) {
      return '0';
    }
    if (x < 1000) {
      return `${x}\u03bcs`;
    }
    if (x < 1e6) {
      return `${(x / 1e3).toFixed(1)}ms`;
    }
    return `${(x / 1e6).toFixed(1)}s`;
  }

  const inactiveColor = function (d) {
    return d.end === undefined ? unfinishedColor : generalColor;
  };

  function wrap(width, padding) {
    return function () {
      let self = d3.select(this),
        textLength = self.node().getComputedTextLength(),
        text = self.text();
      while (textLength > (width - 2 * padding) && text.length > 0) {
        text = text.slice(0, -1);
        self.text(text + '\u2026');
        textLength = self.node().getComputedTextLength();
      }
    }
  }

  function traceplot(selection) {
    let data = selection.datum();


    const maxDur = d3.max(data, d => Math.max(d.end == undefined ? 0 : d.end));

    function delta(x, y) {
      return (y === undefined) ? `${formatDur(maxDur - x)}+` : formatDur(y - x);
    }

    function getTooltipContent(d) {
      return `${d.descr}<br/><b>Begin:</b> ${formatDur(d.begin)}<br/><b>End:</b> ${d.end === undefined ? "Never" : formatDur(d.end)}<br/><b>Duration</b>: ${delta(d.begin, d.end)}`;
    }

    const lookup = {};
    let idx = 0;
    for (let d of data) {
      lookup[d.id] = idx;
      idx += 1;
      d.children = [];
      d.safeEnd = d.end === undefined ? Math.min(maxDur, d.begin + maxDur / 20) : d.end;
    }

    for (let d of data) {
      const parent = lookup[d.parent];
      if (parent !== undefined) {
        data[parent].children.push(d.id);
      }
    }

    let visited = {}
    let s = []

    function dfs(v, s) {
      visited[v] = true;
      for (const c of data[lookup[v]].children) {
        if (!visited[c]) {
          dfs(c, s);
        }
      }
      s.push(v);
    }

    for (const v of Object.keys(lookup)) {
      if (!visited[v]) {
        dfs(v, s);
      }
    }

    for (let i = s.length - 1; i >= 0; i--) {
      const cur = lookup[s[i]];
      const par = lookup[data[cur].parent]
      if (par === undefined) {
        data[cur].level = 0;
      } else {
        data[cur].level = data[par].level + 1;
      }
    }

    const levels = d3.max(data, d => d.level) + 1;

    data.sort((a, b) => a.begin - b.begin)

    let lanesByLevel = []
    for (let i = 0; i < levels; i++) {
      lanesByLevel[i] = []
    }

    for (const d of data) {
      const levelLanes = lanesByLevel[d.level];
      let put = false;
      for (const lane of levelLanes) {
        let last = lane[lane.length - 1];
        if (last.safeEnd <= d.begin) {
          lane.push(d)
          put = true;
          break;
        }
      }
      if (!put) {
        levelLanes.push([d])
      }
    }

    let totalLines = 0;
    for (const levelLanes of lanesByLevel) {
      for (const lane of levelLanes) {
        for (let d of lane) {
          d.line = totalLines;
        }
        totalLines += 1;
      }
    }

    let width = selection.node().clientWidth;
    let height = rowHeight * totalLines + margin.top + margin.bottom;

    const svg = selection
      .append("svg")
      .attr("width", width)
      .attr("height", height);

    const rect = svg
      .append("g")
      .attr("transform", `translate(${margin.left} ${margin.top})`)
      .append("rect")
      .attr("width", width - margin.left - margin.right)
      .attr("height", height - margin.top - margin.bottom)
      .attr("fill", "none")
      .attr("pointer-events", "all");

    const groups = svg
      .append("g")
      .attr("transform", `translate(${margin.left} ${margin.top})`)
      .selectAll("g")
      .data(data)
      .enter()
      .append("g");

    const tooltip = d3
      .select(document.createElement("div"))
      .attr("class", "d3-trace-tooltip");

    const line = svg.append("line")
      .attr("y1", margin.top)
      .attr("y2", height - margin.bottom)
      .attr("stroke", "rgba(0,0,0,0.2)")
      .style("pointer-events", "none");

    let x = d3.scaleLinear()
      .domain([d3.min(data, d => d.begin), d3.max(data, d => d.safeEnd)])
      .range([0, width - margin.left - margin.right]);

    let y = d3.scaleBand()
      .domain(d3.range(totalLines))
      .range([0, height - margin.top - margin.bottom])
      .padding(0.1);

    let axisTop = d3.axisTop(x)
      .tickPadding(2)
      .tickFormat(formatDur)

    svg
      .append("g")
      .attr("transform", `translate(${margin.left} ${margin.top})`)
      .call(axisTop)

    groups.attr("transform", (d) => `translate(0 ${y(d.line)})`)

    const getRect = function (d) {
      const el = d3.select(this);
      const sx = x(d.begin);
      const w = x(d.safeEnd) - x(d.begin);

      el.style("cursor", "pointer").attr("id", `span-${d.id}`)

      el
        .append("rect")
        .attr("x", sx)
        .attr("y", 0)
        .attr("height", y.bandwidth())
        .attr("width", w)
        .attr("fill", inactiveColor)

      el
        .append("text")
        .text(d.descr)
        .each(wrap(w, 2.5))
        .attr("x", sx + 2.5)
        .attr("y", 12 + 2.5)
        .attr("width", w)
        .attr("fill", "black")
        .attr("class", "d3-trace-text")
    }

    let tooltipFixed = false;

    groups
      .each(getRect)
      .on("mouseover", function (d) {
        d3.select(this).select("rect").attr("fill", selectedColor)
        d3.select(`#span-${d.parent}`).select("rect").attr("fill", parentColor)
        for (let c of d.children) {
          d3.select(`#span-${c}`).select("rect").attr("fill", childColor)
        }

        tooltip.style("opacity", 1)

        if (!tooltipFixed) {
          tooltip.html(getTooltipContent(d));
        }
      })
      .on("mouseleave", function (d) {
        d3.select(this).select("rect").attr("fill", inactiveColor)
        d3.select(`#span-${d.parent}`).select("rect").attr("fill", inactiveColor)
        for (let c of d.children) {
          d3.select(`#span-${c}`).select("rect").attr("fill", inactiveColor)
        }

        if (!tooltipFixed) {
          tooltip.style("opacity", 0)
        }
      })
      .on("click", function () {
        if (tooltipFixed) {
          return;
        }
        tooltipFixed = true;
        tooltip.append("br")
        tooltip
          .append("a")
          .text("click here to unpin")
          .attr("class", "d3-trace-link")
          .on("click", function () {
            tooltipFixed = false;
            tooltip.style("opacity", 0)
          })
      })

    selection.append(() => tooltip.node())

    function overline() {

      const node = svg.node();
      let [x, y] = d3.mouse(node);
      const nodeCoords = node.getBoundingClientRect()

      line.attr("transform", `translate(${x} 0)`);
      const rightSide = x > nodeCoords.width / 2;
      const bottomSide = y > nodeCoords.height - 100;
      x += margin.left + nodeCoords.x;
      y += margin.top + nodeCoords.y;
      if (bottomSide) {
        y -= 100;
      }
      if (rightSide) {
        x -= tooltip.node().clientWidth + 20;
      }

      if (!tooltipFixed) {
        tooltip
          .style("left", x + "px")
          .style("top", y + "px")
      }
    }

    svg.on("mousemove", overline)
    rect.on("mousemove", overline)
  }

  traceplot.rowHeight = function(_) {
    if (!arguments.length) { return rowHeight }
    rowHeight = _;
    return traceplot;
  }

  return traceplot;
}
