#include "assets.h"

#include <library/cpp/resource/resource.h>
#include <library/cpp/mime/types/mime.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/http/http_proxy.h>
#include <library/cpp/string_utils/url/url.h>

using namespace NActors;

namespace NSolomon::NSelfMon {
namespace {

const TString ResourcePrefix = "resfs/file";

class TAssetsHandler: public TActor<TAssetsHandler> {
public:
    TAssetsHandler()
        : TActor<TAssetsHandler>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NHttp::TEvHttpProxy::TEvHttpIncomingRequest, OnRequest)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(const NHttp::TEvHttpProxy::TEvHttpIncomingRequest::TPtr& ev) {
        auto& req = ev->Get()->Request;
        TStringBuf url, query, fragment;
        SeparateUrlFromQueryAndFragment(req->URL, url, query, fragment);

        TString content;
        auto key = ResourcePrefix + url;
        if (!NResource::FindExact(key, &content)) {
            Reply(ev->Sender, req->CreateResponseNotFound(TStringBuilder{} << "asset " << url << " was not found"));
            return;
        }

        TStringBuf contentType{"text/plain"};
        if (auto* mimeType = mimetypeByExt(key.c_str())) {
            contentType = mimeType;
        }

        TStringBuf version = req->Version;
        if (version != "1.0" && version != "1.1") {
            version = "1.1";
        }

        auto resp = MakeIntrusive<NHttp::THttpOutgoingResponse>(req, "HTTP", version, "200", "OK");
        resp->Set("Cache-Control", "max-age=31536000");
        resp->Set<&NHttp::THttpResponse::ContentType>(contentType);
        resp->Set<&NHttp::THttpResponse::ContentLength>(ToString(content.size()));
        resp->Set<&NHttp::THttpResponse::Body>(content);

        Reply(ev->Sender, std::move(resp));
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

    void Reply(TActorId replyTo, NHttp::THttpOutgoingResponsePtr resp) {
        Send(replyTo, new NHttp::TEvHttpProxy::TEvHttpOutgoingResponse{std::move(resp)});
    }
};

} // namespace

std::unique_ptr<NActors::IActor> AssetsHandler() {
    return std::make_unique<TAssetsHandler>();
}

} // namespace NSolomon::NSelfMon
