#include "page.h"

#include <solomon/libs/cpp/http/server/core/handler.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/view/layout.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/http/http_proxy.h>

#include <utility>

using namespace NSolomon::NHttp;

namespace NSolomon::NSelfMon {
namespace {

constexpr TStringBuf FOUND_HTTP_TEMPLATE =
        "HTTP/1.1 302 Found\r\nCache-Control: no-cache,no-store,max-age=0,must-revalidate\r\nContent-Length: 0\r\nLocation: "sv;

class THtmlPage: public NHttp::TRequestHandler<THtmlPage> {
public:
    THtmlPage(
            NActors::TActorId backend,
            NActors::TActorId authenticationActor,
            NAuth::IInternalAuthorizerPtr internalAuthorizer,
            TString path,
            TString title,
            TRootLinks rootLinks)
        : TRequestHandler<THtmlPage>(TDuration::Seconds(10), authenticationActor ? EAuthPolicy::AuthRequired : EAuthPolicy::NoAuth, authenticationActor, std::move(internalAuthorizer))
        , Backend_{backend}
        , Layout_{std::move(path), std::move(title), std::move(rootLinks)}
    {
    }

    STATEFN(Handle) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPageDataResp, OnPageData)
            hFunc(TEvRedirectResp, OnRedirect)
        }
    }

    void AfterInit() {
        Send(Backend_, new TEvPageDataReq{RequestPtr()});
    }

    void OnPageData(const TEvPageDataResp::TPtr& ev) {
        try {
            TStringStream content;
            TRenderContext ctx{Request()};
            Layout_.Render(ctx, ev->Get()->Page, &content);
            ReplyAndDie(Request().CreateResponse("200", "OK", "text/html", content.Str()));
        } catch (...) {
            ReplyAndDie(Request().CreateResponse("500", "Internal server error", "text/html", CurrentExceptionMessage()));
        }
    }

    void OnRedirect(const TEvRedirectResp::TPtr& ev) {
        ReplyAndDie(Request().CreateResponseString(TString::Join(FOUND_HTTP_TEMPLATE, ev->Get()->Location, "\r\n\r\n"sv)));
    }

private:
    NActors::TActorId Backend_;
    TLayout Layout_;
};

class TPageHandler: public NActors::TActor<TPageHandler> {
public:
    TPageHandler(
            TString path,
            TString title,
            NActors::TActorId backend,
            NActors::TActorId authenticationActor,
            NAuth::IInternalAuthorizerPtr internalAuthorizer,
            bool hiddenFromRootLinks)
        : NActors::TActor<TPageHandler>(&TThis::StateFunc)
        , Backend_{backend}
        , Auth_{authenticationActor}
        , InternalAuthorizer_{std::move(internalAuthorizer)}
        , Path_{path}
        , Title_{title}
    {
        if (!hiddenFromRootLinks) {
            AddRootLink(std::move(path), std::move(title));
        }
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(::NHttp::TEvHttpProxy::TEvHttpIncomingRequest, OnRequest)
            hFunc(TEvAddRootLink, OnAddRootLink);
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(const ::NHttp::TEvHttpProxy::TEvHttpIncomingRequest::TPtr& ev) {
        // TODO: support non html pages
        auto page = Register(new THtmlPage{Backend_, Auth_, InternalAuthorizer_, Path_, Title_, RootLinks_});
        NActors::TActivationContext::Send(ev->Forward(page));
    }

    void OnAddRootLink(const TEvAddRootLink::TPtr& ev) {
        AddRootLink(std::move(ev->Get()->Path), std::move(ev->Get()->Title));
    }

    void OnPoison(const NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

    void AddRootLink(TString path, TString title) {
        if (title.empty()) {
            RootLinks_.emplace(path, path);
        } else {
            RootLinks_.emplace(std::move(path), std::move(title));
        }
    }

private:
    NActors::TActorId Backend_;
    NActors::TActorId Auth_; // optional
    NAuth::IInternalAuthorizerPtr InternalAuthorizer_;
    TString Path_;
    TString Title_;
    TRootLinks RootLinks_;
};

} // namespace

std::unique_ptr<NActors::IActor> PageHandler(
        TString path,
        TString title,
        NActors::TActorId backend,
        NActors::TActorId authenticationActor,
        const NAuth::IInternalAuthorizerPtr& internalAuthorizer,
        bool hiddenFromRootLinks)
{
    return std::make_unique<TPageHandler>(std::move(path), std::move(title), backend, authenticationActor, internalAuthorizer, hiddenFromRootLinks);
}

} // namespace NSolomon::NSelfMon
