#pragma once

#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/events/slots.h>
#include <solomon/libs/cpp/selfmon/model/page.pb.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/http/http.h>
#include <library/cpp/string_utils/url/url.h>

namespace NSolomon::NSelfMon {

/**
 * The main entrypoint to communicate with SelfMon service.
 */
inline NActors::TActorId ServiceId() noexcept {
    return NActors::TActorId{0, TStringBuf{"SelfMon\0"}};
}

/**
 * IDs of all SelfMon events
 */
struct TEvents: public TEventSlot<EEventSpace::Libs, ELibSlot::SelfMon> {
    enum {
        RegisterPage = SpaceBegin,
        AddRootLink,
        PageDataReq,
        PageDataResp,
        RedirectResp,
        End,
    };

    static_assert(End < SpaceEnd, "too many event types");
};

struct TEvRegisterPage: public NActors::TEventLocal<TEvRegisterPage, TEvents::RegisterPage> {
    TString Path;
    TString Title;
    NActors::TActorId Backend;
    bool HiddenFromRootLinks;

    TEvRegisterPage(TString path, TString title, NActors::TActorId backend, bool hiddenFromRootLinks) noexcept
        : Path{std::move(path)}
        , Title{std::move(title)}
        , Backend{backend}
        , HiddenFromRootLinks{hiddenFromRootLinks}
    {
    }
};

struct TEvAddRootLink: public NActors::TEventLocal<TEvAddRootLink, TEvents::AddRootLink> {
    TString Path;
    TString Title;

    explicit TEvAddRootLink(TString path, TString title) noexcept
        : Path{std::move(path)}
        , Title{std::move(title)}
    {
    }
};

struct TEvPageDataReq: public NActors::TEventLocal<TEvPageDataReq, TEvents::PageDataReq> {
    ::NHttp::THttpIncomingRequestPtr HttpReq;
    TStringBuf Url;
    TStringBuf Query;
    TStringBuf Fragment;

    TEvPageDataReq(const TEvPageDataReq& other)
        : HttpReq{other.HttpReq}
        , Url{other.Url}
        , Query{other.Query}
        , Fragment{other.Fragment}
    {
    }

    explicit TEvPageDataReq(::NHttp::THttpIncomingRequestPtr httpReq) noexcept
        : HttpReq{std::move(httpReq)}
    {
        SeparateUrlFromQueryAndFragment(HttpReq->URL, Url, Query, Fragment);
    }

    TStringBuf Param(TStringBuf name) const {
        // linear search, because we will never have more than 10 params and it is
        // not needed to allocate memory to build hash map for find just one of them

        TStringBuf query = Query;
        while (!query.empty()) {
            auto val = query.NextTok('&');
            if (val.empty()) {
                continue; // && case
            }

            auto key = val.NextTok('=');
            if (key == name) {
                return val;
            }
        }
        return {};
    }
};

struct TEvPageDataResp: public NActors::TEventLocal<TEvPageDataResp, TEvents::PageDataResp> {
    ::yandex::monitoring::selfmon::Page Page;

    explicit TEvPageDataResp(::yandex::monitoring::selfmon::Page page) noexcept
        : Page{std::move(page)}
    {
    }
};

struct TEvRedirectResp: public NActors::TEventLocal<TEvRedirectResp, TEvents::RedirectResp> {
    TString Location;

    explicit TEvRedirectResp(TString location) noexcept
        : Location{std::move(location)}
    {
    }
};

/**
 * Register given page in selfmon service.
 *
 * @param actorSystem  reference to actor system
 * @param path         page path (must be unique)
 * @param title        page title
 * @param pageActor    actor of the page, which can handle TEvPageDataReq
 * @return id of the selfmon page
 */
NActors::TActorId RegisterPage(
        NActors::TActorSystem& actorSystem,
        TString path, TString title,
        std::unique_ptr<NActors::IActor> pageActor,
        bool hiddenFromRootLinks = false);

/**
 * Register given actor id as a page in selfmon service.
 *
 * @param actorSystem  reference to actor system
 * @param path         page path (must be unique)
 * @param title        page title
 * @param pageId       id of the page actor, which can handle TEvPageDataReq
 */
void RegisterPage(
        NActors::TActorSystem& actorSystem,
        TString path, TString title,
        NActors::TActorId pageId,
        bool hiddenFromRootLinks = false);

} // namespace NSolomon::NSelfMon
