#include "logger_page.h"

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace yandex::monitoring::selfmon;
using namespace NActors::NLog;

namespace NSolomon::NSelfMon {
namespace {

Style PriorityToStyle(EPrio priority) {
    switch (priority) {
        case EPrio::Emerg:
        case EPrio::Alert:
        case EPrio::Crit:
            return Style::Danger;

        case EPrio::Error:
        case EPrio::Warn:
            return Style::Warning;

        case EPrio::Notice:
        case EPrio::Info:
            return Style::Info;

        case EPrio::Debug:
        case EPrio::Trace:
            return Style::Success;
    }
}

std::optional<EPriority> ParseLogLevel(TStringBuf levelName) {
    if (levelName.empty()) {
        return std::nullopt;
    }
    if (levelName == "EMERG") return EPriority::PRI_EMERG;
    if (levelName == "ALERT") return EPriority::PRI_ALERT;
    if (levelName == "CRIT") return EPriority::PRI_CRIT;
    if (levelName == "ERROR") return EPriority::PRI_ERROR;
    if (levelName == "WARN") return EPriority::PRI_WARN;
    if (levelName == "NOTICE") return EPriority::PRI_NOTICE;
    if (levelName == "INFO") return EPriority::PRI_INFO;
    if (levelName == "DEBUG") return EPriority::PRI_DEBUG;
    if (levelName == "TRACE") return EPriority::PRI_TRACE;

    return std::nullopt;
}

std::optional<ui32> ParseUint32(TStringBuf str) {
    ui32 value;
    if (TryFromString(str, value)) {
        return value;
    }
    return std::nullopt;
}

void FillListGroup(TStringBuf componentName, ui8 currentLevel, TStringBuf parameter, ListGroup* list) {
    list->set_active(currentLevel == Max<ui8>() ? -1 : currentLevel);

    auto* items = list->mutable_items();
    for (int level = PRI_EMERG; level <= PRI_TRACE; ++level) {
        auto priority = static_cast<EPrio>(level);
        const char* levelName = PriorityToString(priority);

        auto* item = items->Add();
        item->set_style(PriorityToStyle(priority));

        auto* ref = item->mutable_reference();
        ref->set_page("/logger");
        ref->set_args(TStringBuilder{} << "component=" << componentName << '&' << parameter << '=' << levelName);
        ref->set_title(levelName);
    }
}

void RenderSamplingRateForm(TStringBuf componentName, ui32 value, Form* form) {
    form->set_layout(FormLayout::Horizontal);
    form->set_method(FormMethod::Get);
    if (auto* item = form->add_items()) {
        auto* input = item->mutable_input();
        input->set_type(InputType::Hidden);
        input->set_name("component");
        input->set_value(TString{componentName});
    }
    if (auto* item = form->add_items()) {
        auto* input = item->mutable_input();
        input->set_name("samplingRate");
        input->set_type(InputType::IntNumber);
        input->set_value(ToString(value));
    }
    if (auto* submit = form->add_submit()) {
        submit->set_title("Save");
    }
}

class TLoggerPage: public NActors::TActor<TLoggerPage> {
public:
    TLoggerPage()
        : NActors::TActor<TLoggerPage>(&TThis::StateFunc)
        , Settings_{NActors::TActorContext::ActorSystem()->LoggerSettings()}
    {
        Y_VERIFY(Settings_);
    }

    STFUNC(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            HFunc(TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(const TEvPageDataReq::TPtr& ev, const NActors::TActorContext& ctx) {
        TStringBuf componentName = ev->Get()->Param("component");
        auto component = ParseComponent(componentName);
        auto level = ParseLogLevel(ev->Get()->Param("level"));
        auto samplingLevel = ParseLogLevel(ev->Get()->Param("samplingLevel"));
        auto samplingRate = ParseUint32(ev->Get()->Param("samplingRate"));

        if (component.has_value()) {
            TString explanation;
            if (level.has_value()) {
                LOG_EMERG_S(ctx, ELogComponent::Logger,
                        "set level " << PriorityToString(EPrio(*level)) << " for " << componentName);
                Settings_->SetLevel(*level, *component, explanation);
            }
            if (samplingLevel.has_value()) {
                LOG_EMERG_S(ctx, ELogComponent::Logger,
                        "set sampling level " << PriorityToString(EPrio(*samplingLevel)) << " for " << componentName);
                Settings_->SetSamplingLevel(*samplingLevel, *component, explanation);
            }
            if (samplingRate.has_value()) {
                LOG_EMERG_S(ctx, ELogComponent::Logger,
                        "set sampling rate " << *samplingRate << " for " << componentName);
                Settings_->SetSamplingRate(*samplingRate, *component, explanation);
            }
        }

        Page page;
        if (component.has_value() && *component != InvalidComponent) {
            RenderComponent(*component, componentName, &page);
        } else {
            RenderComponentsList(&page);
        }
        Send(ev->Sender, new TEvPageDataResp{std::move(page)});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

    std::optional<EComponent> ParseComponent(TStringBuf name) {
        if (name == "all") {
            return InvalidComponent;
        }

        if (!name.empty()) {
            auto component = Settings_->FindComponent(name);
            if (component != InvalidComponent) {
                return component;
            }
        }
        return std::nullopt;
    }

    void RenderComponent(EComponent component, TStringBuf componentName, Page* page) {
        page->set_title(componentName + TString{" Logger Settings"});

        auto componentSettings = Settings_->GetComponentSettings(component);
        auto* grid = page->mutable_grid();

        if (auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading()) {
            h->set_level(4);
            h->set_content("Current Settings");
        }

        if (auto* obj = grid->add_rows()->add_columns()->mutable_component()->mutable_object()) {
            if (auto* f = obj->add_fields()) {
                f->set_name("Level");
                f->mutable_value()->set_string(PriorityToString(EPrio(componentSettings.Raw.X.Level)));
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("Sampling Level");
                f->mutable_value()->set_string(PriorityToString(EPrio(componentSettings.Raw.X.SamplingLevel)));
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("Sampling Rate");
                f->mutable_value()->set_uint32(componentSettings.Raw.X.SamplingRate);
            }
        }

        if (auto* row = grid->add_rows()){
            if (auto* col = row->add_columns()) {
                col->set_width(2);
                if (auto* h = col->mutable_component()->mutable_heading()) {
                    h->set_level(4);
                    h->set_content("Change Level");
                }
            }
            if (auto* col = row->add_columns()) {
                col->set_width(2);
                if (auto* h = col->mutable_component()->mutable_heading()) {
                    h->set_level(4);
                    h->set_content("Change Sampling Level");
                }
            }
            if (auto* col = row->add_columns()) {
                col->set_width(2);
                if (auto* h = col->mutable_component()->mutable_heading()) {
                    h->set_level(4);
                    h->set_content("Change Sampling Rate");
                }
            }
        }

        if (auto* row = grid->add_rows()) {
            if (auto* col = row->add_columns()) {
                col->set_width(2);
                if (auto* list = col->mutable_component()->mutable_list_group()) {
                    FillListGroup(componentName, componentSettings.Raw.X.Level, "level", list);
                }
            }
            if (auto* col = row->add_columns()) {
                col->set_width(2);
                if (auto* list = col->mutable_component()->mutable_list_group()) {
                    FillListGroup(componentName, componentSettings.Raw.X.SamplingLevel, "samplingLevel", list);
                }
            }
            if (auto* col = row->add_columns()) {
                col->set_width(3);
                RenderSamplingRateForm(
                        componentName,
                        componentSettings.Raw.X.SamplingRate,
                        col->mutable_component()->mutable_form());
            }
        }

        if (auto* row = grid->add_rows()) {
            auto* ref = row->add_columns()->mutable_component()->mutable_value()->mutable_reference();
            ref->set_title("Back");
            ref->set_page("/logger");
        }
    }

    void RenderComponentsList(Page* page) {
        page->set_title("Logger Settings");

        auto* grid = page->mutable_grid();
        if (auto* row = grid->add_rows()) {
            if (auto* col = row->add_columns()) {
                col->set_width(8);
                auto* h = col->mutable_component()->mutable_heading();
                h->set_level(4);
                h->set_content("Current Settings Per Component");
            }
            if (auto* col = row->add_columns()) {
                col->set_width(4);
                auto* h = col->mutable_component()->mutable_heading();
                h->set_level(4);
                h->set_content("Change Level For All Components");
            }
        }

        if (auto* row = grid->add_rows()) {
            if (auto* col = row->add_columns()) {
                col->set_width(8);
                RenderComponentsTable(col->mutable_component()->mutable_table());
            }
            if (auto* col = row->add_columns()) {
                col->set_width(4);
                auto* rightGrid = col->mutable_grid();
                auto commonSettings = CommonSettings();

                if (auto* r = rightGrid->add_rows()) {
                    auto* c = r->add_columns();
                    c->set_width(4);
                    FillListGroup(
                            "all", commonSettings.Raw.X.Level, "level",
                            c->mutable_component()->mutable_list_group());
                }

                if (auto* r = rightGrid->add_rows()) {
                    auto* h = r->add_columns()->mutable_component()->mutable_heading();
                    h->set_level(4);
                    h->set_content("Change Sampling Level For All Components");
                }

                if (auto* r = rightGrid->add_rows()) {
                    auto* c = r->add_columns();
                    c->set_width(4);
                    FillListGroup(
                            "all", commonSettings.Raw.X.SamplingLevel, "samplingLevel",
                            c->mutable_component()->mutable_list_group());
                }

                if (auto* r = rightGrid->add_rows()) {
                    auto* h = r->add_columns()->mutable_component()->mutable_heading();
                    h->set_level(4);
                    h->set_content("Change Sampling Rate For All Components");
                }

                if (auto* r = rightGrid->add_rows()) {
                    RenderSamplingRateForm(
                            "all", commonSettings.Raw.X.SamplingRate,
                            r->add_columns()->mutable_component()->mutable_form());
                }
            }
        }
    }

    void RenderComponentsTable(Table* table) {
        table->set_numbered(true);

        auto* componentCol = table->add_columns();
        componentCol->set_title("Component");
        auto* componentValues = componentCol->mutable_reference();

        auto* levelCol = table->add_columns();
        levelCol->set_title("Level");
        auto* levelValues = levelCol->mutable_string();

        auto* samplingLevelCol = table->add_columns();
        samplingLevelCol->set_title("Sampling Level");
        auto* samplingLevelValues = samplingLevelCol->mutable_string();

        auto* samplingRateCol = table->add_columns();
        samplingRateCol->set_title("Sampling Rate");
        auto* samplingRateValues = samplingRateCol->mutable_uint32();

        for (auto component = Settings_->MinVal; component < Settings_->MaxVal; ++component) {
            const char* componentName = Settings_->ComponentName(component);
            if (componentName == nullptr || *componentName == '\0') {
                continue;
            }

            {
                auto* ref = componentValues->add_values();
                ref->set_title(componentName);
                ref->set_args(TString{"component="} + componentName);
                ref->set_page("/logger");
            }

            auto componentSettings = Settings_->GetComponentSettings(component);
            levelValues->add_values(PriorityToString(EPrio(componentSettings.Raw.X.Level)));
            samplingLevelValues->add_values(PriorityToString(EPrio(componentSettings.Raw.X.SamplingLevel)));
            samplingRateValues->add_values(componentSettings.Raw.X.SamplingRate);
        }
    }

    TComponentSettings CommonSettings() {
        std::optional<ui8> commonLevel;
        std::optional<ui8> commonSamplingLevel;
        std::optional<ui32> commonSamplingRate;
        for (auto component = Settings_->MinVal; component < Settings_->MaxVal; ++component) {
            const char* componentName = Settings_->ComponentName(component);
            if (componentName == nullptr || *componentName == '\0') {
                continue;
            }

            auto componentSettings = Settings_->GetComponentSettings(component);
            if (!commonLevel.has_value()) {
                commonLevel = componentSettings.Raw.X.Level;
            } else if (*commonLevel != componentSettings.Raw.X.Level) {
                commonLevel = Max<ui8>();
            }

            if (!commonSamplingLevel.has_value()) {
                commonSamplingLevel = componentSettings.Raw.X.SamplingLevel;
            } else if (*commonSamplingLevel != componentSettings.Raw.X.SamplingLevel) {
                commonSamplingLevel = Max<ui8>();
            }

            if (!commonSamplingRate.has_value()) {
                commonSamplingRate = componentSettings.Raw.X.SamplingRate;
            } else if (*commonSamplingRate != componentSettings.Raw.X.SamplingRate) {
                commonSamplingRate = 0;
            }
        }

        return TComponentSettings{
            commonLevel.value_or(Max<ui8>()),
            commonSamplingLevel.value_or(Max<ui8>()),
            commonSamplingRate.value_or(0)};
    }

private:
    TSettings* Settings_;
};

} // namespace

std::unique_ptr<NActors::IActor> LoggerPage() {
    return std::make_unique<TLoggerPage>();
}

} // namespace NSolomon::NSelfMon
