#include "memory_page.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/backtrace/printer.h>
#include <solomon/libs/cpp/string_map/string_map.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/html/escape/escape.h>

#include <util/generic/algorithm.h>
#include <util/generic/string.h>
#include <util/stream/format.h>

#include <contrib/libs/tcmalloc/tcmalloc/malloc_extension.h>

using namespace yandex::monitoring::selfmon;

namespace NSolomon::NSelfMon {
namespace {

constexpr auto ProfileTtl = TDuration::Hours(2);

template <typename T>
std::optional<T> ParseParam(const TQuickCgiParam& params, TStringBuf name) {
    if (TStringBuf valueStr = params.Get(name)) {
        T value{};
        if (TryFromString(valueStr, value)) {
            return value;
        }
    }
    return std::nullopt;
}

struct TMemoryProfile {
    tcmalloc::Profile Profile;
    size_t Samples;
    TInstant CreatedAt;

    explicit TMemoryProfile(tcmalloc::Profile profile) noexcept
        : Profile{std::move(profile)}
        , Samples{0}
        , CreatedAt{TInstant::Now()}
    {
        Profile.Iterate([this](const tcmalloc::Profile::Sample&) {
            Samples++;
        });
    }
};

TStringBuf ProfileTypeStr(tcmalloc::ProfileType type) {
    switch (type) {
        case tcmalloc::ProfileType::kHeap: return "Heap";
        case tcmalloc::ProfileType::kFragmentation: return "Fragmentation";
        case tcmalloc::ProfileType::kPeakHeap: return "PeakHeap";
        case tcmalloc::ProfileType::kAllocations: return "Allocations";
        default:
            return "Unknown";
    }
}

class TTcMallocPage: public NActors::TActorBootstrapped<TTcMallocPage> {
public:
    void Bootstrap() {
        Become(&TThis::StateFunc);
        Schedule(TDuration::Minutes(1), new NActors::TEvents::TEvWakeup);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
            sFunc(NActors::TEvents::TEvWakeup, OnWakeup)
        }
    }

private:
    void OnRequest(TEvPageDataReq::TPtr ev) {
        if (ev->Get()->HttpReq->Method == "POST") {
            TQuickCgiParam params{ev->Get()->HttpReq->Body};
            ProcessParamsActions(params);
            ProcessProfileActions(params);
        }

        auto activeTab = ev->Get()->Param("tab");

        Page page;
        page.set_title("Allocator: tcmalloc");

        auto* grid = page.mutable_grid();
        if (auto* r = grid->add_rows()) {
            auto* component = r->add_columns()->mutable_component();
            RenderTabs(activeTab, component->mutable_tabs());
        }

        if (auto* r = grid->add_rows()) {
            if (activeTab.empty() || activeTab == "params") {
                auto* col = r->add_columns();
                col->set_width(4);
                RenderParams(col->mutable_component());
            } else if (activeTab == "stats") {
                RenderStats(r->add_columns()->mutable_component());
            } else if (activeTab == "profile") {
                if (auto* left = r->add_columns()) {
                    left->set_width(4);
                    auto* leftGrid = left->mutable_grid();
                    leftGrid->add_rows()->add_columns()->mutable_component()->mutable_value()->set_string(TStringBuilder{}
                            << "Note that all profiles older than " << ProfileTtl.Hours()
                            << " hours will be automatically deleted.");
                    CurrentProfileId_ = ev->Get()->Param("profileId");
                    RenderProfilesTable(leftGrid->add_rows()->add_columns()->mutable_component());
                    RenderProfileActions(leftGrid->add_rows()->add_columns()->mutable_component());
                }
                if (auto* right = r->add_columns()) {
                    right->set_width(8);
                    if (!ev->Get()->Param("profileId").empty()) {
                        RenderProfile(*ev->Get(), right);
                    } else if (!ev->Get()->Param("profileId1").empty() && !ev->Get()->Param("profileId2").empty()) {
                        RenderDiffProfileIds(*ev->Get(), right);
                    } else {
                        auto* c = right->mutable_component();
                        c->mutable_value()->set_string("select profile from the left table");
                    }
                }
            }
        }
        Send(ev->Sender, new TEvPageDataResp{std::move(page)});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

    void OnWakeup() {
        Schedule(TDuration::Minutes(1), new NActors::TEvents::TEvWakeup);

        auto now = NActors::TActivationContext::Now();
        EraseIf(Profiles_, [now](const TMemoryProfile& p) {
            return (now - p.CreatedAt) >= ProfileTtl;
        });
    }

    void RenderTabs(TStringBuf activeTab, Tabs* tabs) {
        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(activeTab.empty() || activeTab == "params");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Params");
                ref->set_page("/memory");
                ref->set_args("tab=params");
            }
        }
        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(activeTab == "stats");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Stats");
                ref->set_page("/memory");
                ref->set_args("tab=stats");
            }
        }
        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(activeTab == "profile");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Profile");
                ref->set_page("/memory");
                ref->set_args("tab=profile");
            }
        }
    }

    void RenderParams(Component* c) {
        auto* f = c->mutable_form();
        f->set_layout(FormLayout::Vertical);
        f->set_action("/memory");
        f->set_method(FormMethod::Post);

        auto memLimit = tcmalloc::MallocExtension::GetMemoryLimit();
        if (auto* item = f->add_items()) {
            item->set_label("Memory Limit");
            item->set_help(
                    "Make a best effort attempt to prevent more than limit bytes of memory "
                    "from being allocated by the system. Note: limit=SIZE_T_MAX implies no limit.");
            auto* input = item->mutable_input();
            input->set_type(InputType::IntNumber);
            input->set_name("memoryLimit");
            input->set_value(ToString(memLimit.limit));
        }
        if (auto* item = f->add_items()) {
            item->set_label("Memory Limit Hard");
            item->set_help("If hard is set, crash if returning memory is unable to get below the limit.");
            auto* input = item->mutable_check();
            input->set_name("memoryLimitHard");
            input->set_value(memLimit.hard);
        }
        if (auto* item = f->add_items()) {
            item->set_label("Profile Sampling Rate");
            item->set_help(
                    "Sets the sampling rate for heap profiles. TCMalloc samples approximately "
                    "every rate bytes allocated.");
            auto* input = item->mutable_input();
            input->set_type(InputType::IntNumber);
            input->set_name("profileSamplingRate");
            input->set_value(ToString(tcmalloc::MallocExtension::GetProfileSamplingRate()));
        }
        if (auto* item = f->add_items()) {
            item->set_label("Guarded Sampling Rate");
            item->set_help(
                    "Sets the guarded sampling rate for sampled allocations. Guarded samples "
                    "provide probabilistic protections against buffer underflow, overflow, and "
                    "use-after-free.");
            auto* input = item->mutable_input();
            input->set_type(InputType::IntNumber);
            input->set_name("guardedSamplingRate");
            input->set_value(ToString(tcmalloc::MallocExtension::GetGuardedSamplingRate()));
        }
        if (auto* item = f->add_items()) {
            item->set_label("Max Per Cpu Cache Size");
            item->set_help("Sets the maximum cache size per CPU cache. This is a per-core limit.");
            auto* input = item->mutable_input();
            input->set_type(InputType::IntNumber);
            input->set_name("maxPerCpuCacheSize");
            input->set_value(ToString(tcmalloc::MallocExtension::GetMaxPerCpuCacheSize()));
        }
        if (auto* item = f->add_items()) {
            item->set_label("Max Total Thread Cache Bytes");
            item->set_help("Sets the maximum thread cache size. This is a whole-process limit.");
            auto* input = item->mutable_input();
            input->set_type(InputType::IntNumber);
            input->set_name("maxTotalThreadCacheBytes");
            input->set_value(ToString(tcmalloc::MallocExtension::GetMaxTotalThreadCacheBytes()));
        }
        if (auto* item = f->add_items()) {
            item->set_label("Background Release Rate");
            item->set_help("Specifies the release rate from the page heap.");
            auto* input = item->mutable_input();
            input->set_type(InputType::IntNumber);
            input->set_name("backgroundReleaseRate");
            input->set_value(ToString(size_t(tcmalloc::MallocExtension::GetBackgroundReleaseRate())));
        }

        auto* submit = f->add_submit();
        submit->set_title("Save");
    }

    void ProcessParamsActions(const TQuickCgiParam& params) {
        if (auto memLimit = ParseParam<ui64>(params, "memoryLimit")) {
            auto memLimitHard = ParseParam<bool>(params, "memoryLimitHard");
            tcmalloc::MallocExtension::MemoryLimit x;
            x.limit = *memLimit;
            x.hard = memLimitHard.value_or(false);
            tcmalloc::MallocExtension::SetMemoryLimit(x);
        }
        if (auto samplingRate = ParseParam<i64>(params, "profileSamplingRate")) {
            tcmalloc::MallocExtension::SetProfileSamplingRate(*samplingRate);
        }
        if (auto samplingRate = ParseParam<i64>(params, "guardedSamplingRate")) {
            tcmalloc::MallocExtension::SetGuardedSamplingRate(*samplingRate);
        }
        if (auto cacheSize = ParseParam<i32>(params, "maxPerCpuCacheSize")) {
            tcmalloc::MallocExtension::SetMaxPerCpuCacheSize(*cacheSize);
        }
        if (auto cacheSize = ParseParam<i64>(params, "maxTotalThreadCacheBytes")) {
            tcmalloc::MallocExtension::SetMaxTotalThreadCacheBytes(*cacheSize);
        }
        if (auto releaseRate = ParseParam<ui64>(params, "backgroundReleaseRate")) {
            tcmalloc::MallocExtension::SetBackgroundReleaseRate(tcmalloc::MallocExtension::BytesPerSecond(*releaseRate));
        }
    }

    void RenderStats(Component* c) {
        auto* code = c->mutable_code();
        code->set_content(TString{tcmalloc::MallocExtension::GetStats()});
    }

    void RenderProfilesTable(Component* c) {
        auto* t = c->mutable_table();
        t->set_numbered(true);

        auto* refColumn = t->add_columns();
        refColumn->set_title("Profile");
        auto* refValues = refColumn->mutable_reference();

        auto* typeColumn = t->add_columns();
        typeColumn->set_title("Type");
        auto* typeValues = typeColumn->mutable_string();

        auto* ageColumn = t->add_columns();
        ageColumn->set_title("Age");
        auto* ageValues = ageColumn->mutable_duration();

        auto* samplesColumn = t->add_columns();
        samplesColumn->set_title("Samples");
        auto* samplesValues = samplesColumn->mutable_uint64();

        auto* diffColumn = t->add_columns();
        diffColumn->set_title("Diff");
        auto* diffValues = diffColumn->mutable_reference();

        size_t id = 0;
        auto now = NActors::TActivationContext::Now();
        for (const auto& p: Profiles_) {
            auto idStr = ToString(id++);

            if (auto* ref = refValues->add_values()) {
                ref->set_title("p" + idStr);
                ref->set_page("/memory");
                ref->set_args("tab=profile&profileId=" + idStr);
            }

            typeValues->add_values()->assign(ProfileTypeStr(p.Profile.Type()));
            ageValues->add_values((now - p.CreatedAt).GetValue());
            samplesValues->add_values(p.Samples);

            if (auto* diff = diffValues->add_values()) {
                diff->set_page("/memory");
                if (!CurrentProfileId_.empty() && idStr != CurrentProfileId_) {
                    diff->set_title("diff with");
                    diff->set_args("tab=profile&profileId1=" + CurrentProfileId_ + "&profileId2=" + idStr);
                } else {
                    diff->set_title("-");
                    diff->set_args("tab=profile");
                }
            }
        }
    }

    void RenderProfileActions(Component* c) {
        auto* f = c->mutable_form();
        f->set_layout(FormLayout::Horizontal);
        f->set_action("/memory?tab=profile");
        f->set_method(FormMethod::Post);

        if (auto* s = f->add_submit()) {
            s->set_name("action");
            s->set_value("heap");
            s->set_title("Heap");
            s->set_style(Style::Primary);
        }
        if (auto* s = f->add_submit()) {
            s->set_name("action");
            s->set_value("fragmentation");
            s->set_title("Fragmentation");
            s->set_style(Style::Warning);
        }
        if (auto* s = f->add_submit()) {
            s->set_name("action");
            s->set_value("peakHeap");
            s->set_title("Peak Heap");
            s->set_style(Style::Danger);
        }
        if (auto* s = f->add_submit()) {
            s->set_name("action");
            s->set_value("allocations");
            s->set_title("Allocations");
            s->set_style(Style::Dark);
        }
    }

    void ProcessProfileActions(const TQuickCgiParam& params) {
        auto action = params.Get("action");
        std::optional<tcmalloc::ProfileType> profileType;
        if (action == "heap") {
            profileType = tcmalloc::ProfileType::kHeap;
        } else if (action == "fragmentation") {
            profileType = tcmalloc::ProfileType::kFragmentation;
        } else if (action == "peakHeap") {
            profileType = tcmalloc::ProfileType::kPeakHeap;
        } else if (action == "allocations") {
            profileType = tcmalloc::ProfileType::kAllocations;
        }

        if (profileType.has_value()) {
            Profiles_.emplace_back(tcmalloc::MallocExtension::SnapshotCurrent(*profileType));
        }
    }

    void RenderFlameGraph(yandex::monitoring::selfmon::FlameGraph* root, std::shared_ptr<TVector<TString>> stackTrace, ui64 sampleSum) {
        root->set_value(root->value() + sampleSum);
        for (int i = stackTrace->size() - 1; i >= 0; --i) {
            auto* children = root->mutable_children()->mutable_graph();
            auto iter = children->find(stackTrace->at(i));
            if (iter == children->end()) {
                yandex::monitoring::selfmon::FlameGraph fg;
                fg.set_value(0);
                iter = children->insert({stackTrace->at(i), fg}).first;
            }
            auto* child = &iter->second;
            child->set_value(child->value() + sampleSum);
            root = child;
        }
    }

    TString BackTraceToString(const tcmalloc::Profile::Sample& s, IBacktracePrinter* backtracePrinter, std::shared_ptr<TVector<TString>> stackTrace = nullptr) {
        TStringStream ss;
        for (int i = 0; i < s.depth; ++i) {
            ss << '#' << (i + 1) << ' ';
            TStringStream frame;
            backtracePrinter->Print(s.stack[i], &frame);
            TString name = NHtml::EscapeAttributeValue(frame.Str());
            ss << name;
            if (stackTrace != nullptr) {
                name.pop_back();
                stackTrace->push_back(std::move(name));
            }
        }
        return ss.Str();
    }

    Grid* AddGrid(Grid_Column* col, const TString& str) {
        auto* g = col->mutable_grid();
        g->add_rows()->add_columns()->mutable_component()->mutable_value()->set_string(str);
        return g;
    }

    FlameGraph* AddFlameGraph(Grid* grid) {
        auto* collapsibleFlameGraph = grid->add_rows()->add_columns()->mutable_component()->mutable_collapsible();
        collapsibleFlameGraph->set_title("Flame Graph");
        collapsibleFlameGraph->set_expanded(false);
        collapsibleFlameGraph->set_id("graph");
        auto* flameGraph = collapsibleFlameGraph->mutable_content()->mutable_flame_graph();
        flameGraph->set_value(0);
        return flameGraph;
    }

    Table* AddTable(Grid* grid) {
        auto* c = grid->add_rows()->add_columns()->mutable_component();
        auto* t = c->mutable_table();
        t->set_numbered(true);
        return t;
    }

    Table_Column* AddColumn(Table* t, const TString& title) {
        auto* column = t->add_columns();
        column->set_title(title);
        return column;
    }

    void RenderProfile(const TEvPageDataReq& req, Grid_Column* col) {
        auto* backtracePrinter = BacktracePrinter();

        auto profileIdStr = req.Param("profileId");
        size_t profileId;
        if (!TryFromString(profileIdStr, profileId)) {
            auto* c = col->mutable_component();
            c->mutable_value()->set_string("invalid profile id " + TString{profileIdStr});
            return;
        }
        if (profileId >= Profiles_.size()) {
            auto* c = col->mutable_component();
            c->mutable_value()->set_string("invalid profile id " + TString{profileIdStr});
            return;
        }
        auto* g = AddGrid(col, "Samples of profile p" + TString{profileIdStr});
        auto* flameGraph = AddFlameGraph(g);
        auto* t = AddTable(g);

        auto* sumValues =  AddColumn(t, "Sum")->mutable_data_size();
        auto* countValues = AddColumn(t, "Count")->mutable_int64();
        auto* reqSizeValues = AddColumn(t, "Req Size")->mutable_data_size();
        auto* reqAlignValues = AddColumn(t, "Req Align")->mutable_uint64();
        auto* allocSizeValues = AddColumn(t, "Alloc Size")->mutable_data_size();
        auto* stackValues = AddColumn(t, "Stack")->mutable_components();

        const auto& profile = Profiles_[profileId];
        profile.Profile.Iterate([&](const tcmalloc::Profile::Sample& s) {
            sumValues->add_values(s.sum);
            countValues->add_values(s.count);
            reqSizeValues->add_values(s.requested_size);
            reqAlignValues->add_values(s.requested_alignment);
            allocSizeValues->add_values(s.allocated_size);
            auto stackTrace = std::make_shared<TVector<TString>>();
            auto backtraceStr = BackTraceToString(s, backtracePrinter, stackTrace);
            RenderFlameGraph(flameGraph, stackTrace, s.sum);

            auto* code = stackValues->add_values()->mutable_code();
            code->set_lang("plaintext");
            code->set_content(backtraceStr);
        });
    }

    void RenderDiffProfileIds(const TEvPageDataReq& req, Grid_Column* col) {
        auto* backtracePrinter = BacktracePrinter();

        auto profileIdStr1 = req.Param("profileId1");
        auto profileIdStr2 = req.Param("profileId2");
        size_t profileId1, profileId2;
        if (!TryFromString(profileIdStr1, profileId1) || !TryFromString(profileIdStr2, profileId2)) {
            auto* c = col->mutable_component();
            c->mutable_value()->set_string("invalid profile ids");
            return;
        }
        if (profileId1 >= Profiles_.size() || profileId2 >= Profiles_.size()) {
            auto* c = col->mutable_component();
            c->mutable_value()->set_string("invalid profile ids");
            return;
        }
        auto* g = AddGrid(col, TStringBuilder{}
                << "Difference between profiles: "
                << profileIdStr1
                << " vs "
                << profileIdStr2);
        auto* flameGraph = AddFlameGraph(g);
        auto* t = AddTable(g);

        auto* sumValues = AddColumn(t, "Sum")->mutable_string();
        auto* countValues = AddColumn(t, "Count")->mutable_string();
        auto* reqSizeValues = AddColumn(t, "Req Size")->mutable_string();
        auto* reqAlignValues = AddColumn(t, "Req Align")->mutable_string();
        auto* allocSizeValues = AddColumn(t, "Alloc Size")->mutable_string();
        auto* stackValues = AddColumn(t, "Stack")->mutable_components();

        const auto& profile1 = Profiles_[profileId1];
        TStringMap<tcmalloc::Profile::Sample> samples;
        profile1.Profile.Iterate([&](const tcmalloc::Profile::Sample &s) {
            auto& sample = samples[BackTraceToString(s, backtracePrinter)];
            sample.sum = s.sum;
            sample.allocated_size = s.allocated_size;
            sample.requested_alignment = s.requested_alignment;
            sample.requested_size = s.requested_size;
            sample.count = s.count;
        });
        const auto& profile2 = Profiles_[profileId2];
        profile2.Profile.Iterate([&](const tcmalloc::Profile::Sample &s) {
            auto stackTrace = std::make_shared<TVector<TString>>();
            auto backtraceStr = BackTraceToString(s, backtracePrinter, stackTrace);
            tcmalloc::Profile::Sample s1;
            if (auto iter = samples.find(backtraceStr); iter != samples.end()) {
                s1 = iter->second;
            } else {
                s1.sum = 0;
                s1.allocated_size = 0;
                s1.requested_size = 0;
                s1.requested_alignment = 0;
                s1.count = 0;
            }
            auto diffStr = [](i64 first, i64 second) {
                TStringBuilder sb;
                sb << HumanReadableSize(first, SF_BYTES) << " -> " << HumanReadableSize(second, SF_BYTES)
                   << "(" << HumanReadableSize(second - first, SF_BYTES) << ")";
                return sb;
            };
            sumValues->add_values(diffStr(s1.sum, s.sum));
            countValues->add_values(TStringBuilder{} << s1.count << " -> " << s.count << "(" << s.count - s1.count << ")");
            reqSizeValues->add_values(diffStr(s1.requested_size, s.requested_size));
            reqAlignValues->add_values(diffStr(s1.requested_alignment, s.requested_alignment));
            allocSizeValues->add_values(diffStr(s1.allocated_size, s.allocated_size));

            if (s.sum > s1.sum) {
                RenderFlameGraph(flameGraph, std::move(stackTrace), s.sum - s1.sum);
            }

            auto *collapsible = stackValues->add_values()->mutable_collapsible();
            collapsible->set_title("stack");
            collapsible->set_expanded(false);
            collapsible->set_id("stack" + ToString(sumValues->values_size()));
            auto *code = collapsible->mutable_content()->mutable_code();
            code->set_lang("plaintext");
            code->set_content(backtraceStr);
        });
    }

private:
    std::vector<TMemoryProfile> Profiles_;
    TString CurrentProfileId_;
};

} // namespace

std::unique_ptr<NActors::IActor> MemoryPage() {
    tcmalloc::MallocExtension::ActivateGuardedSampling();
    return std::make_unique<TTcMallocPage>();
}

} // namespace NSolomon::NSelfMon
