#include "minidumps_page.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/service/name.h>
#include <solomon/libs/cpp/minidump/minidump_path.h>
#include <solomon/libs/cpp/minidump/metrics/minidump.h>
#include <solomon/libs/cpp/minidump/model/stack_trace.pb.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/html/escape/escape.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <util/folder/path.h>
#include <util/system/fstat.h>

#include <fstream>

using namespace yandex::monitoring::selfmon;

namespace NSolomon::NSelfMon {
namespace {

class TMinidumpsPage: public NActors::TActorBootstrapped<TMinidumpsPage> {
public:
    TMinidumpsPage(std::shared_ptr<NMonitoring::IMetricRegistry> registry)
        : MinidumpPath_(*GetMinidumpPath())
        , Registry_{std::move(registry)}
    {
    }

    void Bootstrap() {
        auto* intGauge = Registry_->IntGauge(NMonitoring::MakeLabels({{"sensor", "minidump.count"}}));
        Register(CreateMinidumpCounter(intGauge, MinidumpPath_).release());
        Become(&TMinidumpsPage::StateFunc);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(const NSolomon::NSelfMon::TEvPageDataReq::TPtr& ev) {
        TStringBuf minidumpId = ev->Get()->Param("id");
        Page page;
        if (minidumpId.empty()) {
            RenderMinidumpsPage(page);
        } else {
            RenderStackTrace(minidumpId, page);
        }
        Send(ev->Sender, new TEvPageDataResp{std::move(page)});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

    void RenderMinidumpsPage(Page& page) {
        page.set_title("Minidumps");
        try {
            TVector<TString> list;
            MinidumpPath_.ListNames(list);
            RenderMinidumpsList(page, list);
        } catch (yexception& e) {
            auto* grid = page.mutable_grid();
            grid->clear_rows();
            auto* code = grid->add_rows()->add_columns()->mutable_component()->mutable_code();
            code->set_content(e.what());
        }
    }

    void RenderMinidumpsList(Page& page, const TVector<TString>& fileNames) {
        auto* grid = page.mutable_grid();
        auto* table = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* minidumpIdColumn = table->add_columns();
        minidumpIdColumn->set_title("Minidump id");
        auto* minidumpIdValue = minidumpIdColumn->mutable_reference();

        auto* minidumpDateColumn = table->add_columns();
        minidumpDateColumn->set_title("Date");
        auto* minidumpDateValue = minidumpDateColumn->mutable_duration();

        auto* minidumpCrashedFunctionColumn = table->add_columns();
        minidumpCrashedFunctionColumn->set_title("Crashed function");
        auto* minidumpCrashedFunctionValue = minidumpCrashedFunctionColumn->mutable_string();

        auto now = TInstant::Now();
        for (auto fileName: fileNames) {
            if (fileName.EndsWith("dmp")) {
                auto* ref = minidumpIdValue->add_values();
                ref->set_title(fileName);
                ref->set_args(TString{"id="} + fileName);
                ref->set_page("/minidumps");

                TFileStat fileStat(MinidumpPath_.Child(fileName).GetPath());
                auto ctime = TInstant::Seconds(fileStat.CTime);
                minidumpDateValue->add_values((now - ctime).GetValue());

                minidumpCrashedFunctionValue->add_values(GetCrashedFunctionName(fileName));
            }
        }
    }

    void SetStackTraceToPage(yandex::monitoring::selfmon::Grid* grid, yandex::monitoring::minidump::StackTrace& stackTrace) {
        auto* ref = grid->add_rows()->add_columns()->mutable_component()->mutable_value()->mutable_reference();
        ref->set_title("Create ticket");
        ref->set_page("https://st.yandex-team.ru/createTicket");
        TStringBuilder args;
        auto* collapsibleList = grid->add_rows()->add_columns()->mutable_component()->mutable_collapsible_list();
        for (int threadIndex = 0; threadIndex < stackTrace.threads_size(); ++threadIndex) {
            auto* thread = stackTrace.mutable_threads(threadIndex);
            auto* collapsible = collapsibleList->add_collapsibles();
            TStringBuilder title;
            title << "Thread-" << thread->index();
            if (thread->is_crashed()) {
                title << " (crashed)";
            }
            if (!thread->exception_message().empty()) {
                title << " Exception message: " << thread->exception_message();
            }
            TStringBuilder content;
            for (int frameIndex = 0; frameIndex < thread->frames_size(); ++frameIndex) {
                auto* frame = thread->mutable_frames(frameIndex);
                content << frameIndex << " " << frame->function_name() << " "
                       << frame->path() << ":" << frame->line() << ":" << frame->column() << "\n";
            }
            if (thread->is_crashed()) {
                TStringBuilder summary;
                summary << TServiceInfo::Name() << " crashed in " <<
                        (thread->frames_size() > 0 ? thread->frames(0).function_name() : " Unknown");
                args << "summary=" << CGIEscapeRet(summary);
                args << "&description=%25%25%0a" << title << "%0A" <<  CGIEscapeRet(content) << "%25%25&type=1";
            }
            collapsible->set_title(title);
            collapsible->set_expanded(thread->is_crashed());
            collapsible->set_id("thread" + ToString(threadIndex));
            auto* code = collapsible->mutable_content()->mutable_code();
            code->set_lang("cpp");
            code->set_content(NHtml::EscapeAttributeValue(content));
        }
        args << "&queue=SOLOMON";
        ref->set_args(args);
    }

    void RenderStackTrace(TStringBuf minidumpId, Page& page) {
        page.set_title(minidumpId + TString{" stack"});

        auto* grid = page.mutable_grid();

        auto stackFile = GetStackFile(minidumpId);
        if (stackFile) {
            yandex::monitoring::minidump::StackTrace stackTrace;
            if(stackTrace.ParseFromIstream(stackFile.get())) {
                SetStackTraceToPage(grid, stackTrace);
            } else {
                auto* code = grid->add_rows()->add_columns()->mutable_component()->mutable_code();
                code->set_content("Failed to parse stack trace");
            }
        } else {
            auto* code = grid->add_rows()->add_columns()->mutable_component()->mutable_code();
            code->set_content("Unknown");
        }

        auto* ref = grid->add_rows()->add_columns()->mutable_component()->mutable_value()->mutable_reference();
        ref->set_title("Back");
        ref->set_page("/minidumps");
    }

    TString GetCrashedFunctionName(TString& dmpfileName) {
        auto stackFile = GetStackFile(dmpfileName);
        if (!stackFile) {
            return "Unknown";
        }

        yandex::monitoring::minidump::StackTrace stackTrace;
        if (stackTrace.ParseFromIstream(stackFile.get()) && stackTrace.threads_size() > 0) {
            auto& thread = stackTrace.threads(0);
            if (thread.is_crashed() && thread.frames_size() > 0) {
                return NHtml::EscapeAttributeValue(thread.frames(0).function_name());
            }
        }
        return "Unknown";
    }

    std::unique_ptr<std::ifstream> GetStackFile(const TStringBuf& dmpPath) {
        TString stackFileName = dmpPath.substr(0, dmpPath.find('.')) + TString{".stack"};
        TFsPath stackPath = MinidumpPath_.Child(stackFileName);
        if (stackPath.Exists()) {
            return std::make_unique<std::ifstream>(stackPath.GetPath().Data(), std::ios::binary);
        }
        return nullptr;
    }

    TFsPath MinidumpPath_;
    std::shared_ptr<NMonitoring::IMetricRegistry> Registry_;
};

} // namespace

std::unique_ptr<NActors::IActor> MinidumpsPage(std::shared_ptr<NMonitoring::IMetricRegistry> registry) {
    return std::make_unique<TMinidumpsPage>(std::move(registry));
}

} // namespace NSolomon::NSelfMon
