#include "service.h"
#include "logger_page.h"
#include "memory_page.h"
#include "status_page.h"
#include "version_page.h"
#include "threads_page.h"
#include "minidumps_page.h"

#include <solomon/libs/cpp/actors/metrics/self_ping.h>
#include <solomon/libs/cpp/actors/poison/poisoner.h>
#include <solomon/libs/cpp/selfmon/handler/assets.h>
#include <solomon/libs/cpp/selfmon/handler/page.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/view/prefix.h>
#include <solomon/libs/cpp/minidump/minidump_path.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/http/http_proxy.h>

namespace NSolomon::NSelfMon {
namespace {

class TSelfMonService: public NActors::TActorBootstrapped<TSelfMonService> {
    struct TPage {
        TString Title;
        NActors::TActorId Handler;
        bool HiddenFromRootLinks{false};
    };

public:
    TSelfMonService(
            NActors::TActorId httpProxy,
            NActors::TActorId authenticator,
            NAuth::IInternalAuthorizerPtr internalAuthorizer,
            std::shared_ptr<NMonitoring::IMetricRegistry> registry) noexcept
        : HttpProxy_{httpProxy}
        , Authenticator_{authenticator}
        , InternalAuthorizer_{std::move(internalAuthorizer)}
        , Registry_{std::move(registry)}
    {
    }

    void Bootstrap() {
        Become(&TThis::StateFunc);

        RegisterPage("", "Status", StatusPage());
        RegisterPage("/logger", "Logger", LoggerPage());
        RegisterPage("/version", "Version", VersionPage());
        RegisterPage("/memory", "Memory", MemoryPage());
        RegisterPage("/threads", "Threads", ThreadsPage());
        if (!GetMinidumpPath()->empty()) {
            RegisterPage("/minidumps", "Minidumps", MinidumpsPage(Registry_));
        }

        Register(CreateUptimeActor(Registry_->Counter(NMonitoring::MakeLabels({{"sensor", "uptimeMillis"}}))).release());
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvRegisterPage, OnRegisterPage)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void RegisterPage(TString path, TString title, std::unique_ptr<NActors::IActor> pageActor) {
        auto pageId = Register(pageActor.release());
        Send(SelfId(), new TEvRegisterPage{std::move(path), std::move(title), pageId, false});
    }

    void OnRegisterPage(const TEvRegisterPage::TPtr& ev) {
        auto& path = ev->Get()->Path;
        auto& title = ev->Get()->Title;
        auto& backend = ev->Get()->Backend;
        bool hidden = ev->Get()->HiddenFromRootLinks;
        Y_VERIFY(path.Empty() || path.StartsWith('/'), "page path (%s) do not starts with /", path.c_str());

        auto handlerId = Register(PageHandler(path, title, backend, Authenticator_, InternalAuthorizer_, hidden).release());
        bool isNew = Pages_.emplace(path, TPage{title, handlerId, hidden}).second;
        Y_VERIFY(isNew, "duplicated page path registration: %s", path.c_str());

        for (const auto& it: Pages_) {
            if (it.first != path) {
                if (!hidden) {
                    // let all other pages to know about new page
                    Send(it.second.Handler, new TEvAddRootLink{path, title});
                }
                if (!it.second.HiddenFromRootLinks) {
                    // let new page to know about all other pages
                    Send(handlerId, new TEvAddRootLink{it.first, it.second.Title});
                }
            }
        }

        Send(HttpProxy_, new NHttp::TEvHttpProxy::TEvRegisterHandler{BasePrefix + path, handlerId});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        std::set<NActors::TActorId> handlers;
        for (const auto& it: Pages_) {
            handlers.insert(it.second.Handler);
        }

        // poisoner will reply with TEvPoisonTaken
        auto poisoner = Register(CreatePoisoner(handlers).release());
        NActors::TActivationContext::Send(ev->Forward(poisoner));

        // so don't need to wait all pages destruction here
        PassAway();
    }

private:
    NActors::TActorId HttpProxy_;
    NActors::TActorId Authenticator_; // optional
    NAuth::IInternalAuthorizerPtr InternalAuthorizer_;
    std::map<TString, TPage> Pages_;
    std::shared_ptr<NMonitoring::IMetricRegistry> Registry_;
};

} // namespace

void InitService(
        std::shared_ptr<NMonitoring::IMetricRegistry> registry,
        NActors::TActorSystem& actorSystem,
        NActors::TActorId httpProxy,
        ui32 executorPool,
        NActors::TActorId authenticator,
        NAuth::IInternalAuthorizerPtr internalAuthorizer)
{
    auto assetsHandler = actorSystem.Register(AssetsHandler().release(), NActors::TMailboxType::HTSwap, executorPool);
    actorSystem.Send(httpProxy, new NHttp::TEvHttpProxy::TEvRegisterHandler{"/assets/", assetsHandler});

    auto selfMonService = actorSystem.Register(
            new TSelfMonService(httpProxy, authenticator, std::move(internalAuthorizer), std::move(registry)),
            NActors::TMailboxType::HTSwap,
            executorPool);
    actorSystem.RegisterLocalService(ServiceId(), selfMonService);
}

} // namespace NSolomon::NSelfMon
