#include "status_page.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/process_stats/process_stats.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using ::yandex::monitoring::selfmon::Page;
using ::yandex::monitoring::selfmon::Object;

namespace NSolomon::NSelfMon {
namespace {

void AddFieldDataSize(TString name, double value, Object* object) {
    auto* field = object->add_fields();
    field->set_name(std::move(name));
    auto* fieldValue = field->mutable_value();
    fieldValue->set_data_size(value);
}

void AddField(TString name, ui32 value, Object* object) {
    auto* field = object->add_fields();
    field->set_name(std::move(name));
    auto* fieldValue = field->mutable_value();
    fieldValue->set_uint32(value);
}

class TStatusPage: public NActors::TActor<TStatusPage> {
public:
    TStatusPage()
        : NActors::TActor<TStatusPage>(&TThis::StateFunc)
        , StatsProvider_{CreateProcessStatProvider()}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(const TEvPageDataReq::TPtr& ev) {
        ::yandex::monitoring::selfmon::Page page;
        page.set_title("Process Status");

        auto* component = page.mutable_component();
        auto* object = component->mutable_object();

        auto result = StatsProvider_->GetSelfStats();
        if (result.Fail()) {
            auto* field = object->add_fields();
            field->set_name("Error");
            field->mutable_value()->set_string(result.Error().MessageString());
        } else {
            auto& processStats = result.Value();
            AddField("CPU: User Time", processStats.CpuUser, object);
            AddField("CPU: System Time", processStats.CpuSystem, object);
            AddFieldDataSize("Memory: RSS", static_cast<double>(processStats.MemRss), object);
            AddFieldDataSize("Memory: Swap", static_cast<double>(processStats.MemSwap), object);
            AddFieldDataSize("Memory: Lib", static_cast<double>(processStats.MemLib), object);
            AddFieldDataSize("Memory: Major Page Faults", static_cast<double>(processStats.MajorPageFaults), object);
            AddField("Thread Count", (ui32) processStats.ThreadCount, object);
        }

        Send(ev->Sender, new TEvPageDataResp{std::move(page)});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    IProcessStatProviderPtr StatsProvider_;
};

} // namespace

std::unique_ptr<NActors::IActor> StatusPage() {
    return std::make_unique<TStatusPage>();
}

} // namespace NSolomon::NSelfMon
