#include "threads_page.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/process_stats/threads_stats.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/hash_set.h>

using namespace yandex::monitoring::selfmon;
using namespace NActors;

namespace NSolomon::NSelfMon {
namespace {

template<typename T>
struct TRateValue {
    T PrevValue{0};
    TInstant PrevTime = TInstant::Now();
    double RateChange{0};

    void SetValue(T newValue) {
        TInstant newTime = TInstant::Now();
        RateChange = static_cast<double>(newValue - PrevValue) / (newTime - PrevTime).Seconds();
        PrevValue = newValue;
        PrevTime = newTime;
    }
};

struct TThreadStatsRate {
    i64 Pid{};
    TString Comm{};
    char State{};
    i64 Ppid{};
    i64 Pgrp{};
    i64 Session{};
    i64 TtyNr{};
    i64 Tpgid{};
    ui64 Flags{};
    TRateValue<ui64> Minflt{};
    TRateValue<ui64> Cminflt{};
    TRateValue<ui64> Majflt{};
    TRateValue<ui64> Cmajflt{};
    ui64 Utime{};
    ui64 Stime{};
    i64 Cutime{};
    i64 Cstime{};
    i64 Priority{};
    i64 Nice{};
    i64 NumThreads{};
    i64 Itrealvalue{};
    ui64 Starttime{};
    ui64 Vsize{};
    TRateValue<i64> Rss{};
    ui64 Rsslim{};
    ui64 Startcode{};
    ui64 Endcode{};
    ui64 Startstack{};
    ui64 Kstkesp{};
    ui64 Kstkeip{};
    ui64 Signal{};
    ui64 Blocked{};
    ui64 Sigignore{};
    ui64 Sigcatch{};
    ui64 Wchan{};
    TRateValue<ui64> Nswap{};
    TRateValue<ui64> Cnswap{};
    i64 ExitSignal{};
    i64 Processor{};
    ui64 RtPriority{};
    ui64 Policy{};
    TRateValue<ui64> DelayacctBlkioTicks{};
    TRateValue<ui64> GuestTime{};
    TRateValue<i64> CguestTime{};
    ui64 StartData{};
    ui64 EndData{};
    ui64 StartBrk{};
    ui64 ArgStart{};
    ui64 ArgEnd{};
    ui64 EnvStart{};
    ui64 EnvEnd{};
    i64 ExitCode{};

    TThreadStatsRate() = default;

    void Change(const TThreadStats& thread) {
        Pid = thread.Pid;
        Comm = thread.Comm;
        State = thread.State;
        Ppid = thread.Ppid;
        Pgrp = thread.Pgrp;
        Session = thread.Session;
        TtyNr = thread.TtyNr;
        Tpgid = thread.Tpgid;
        Flags = thread.Flags;
        Minflt.SetValue(thread.Minflt);
        Cminflt.SetValue(thread.Cminflt);
        Majflt.SetValue(thread.Majflt);
        Cmajflt.SetValue(thread.Cmajflt);
        Utime = thread.Utime;
        Stime = thread.Stime;
        Cutime = thread.Cutime;
        Cstime = thread.Cstime;
        Priority = thread.Priority;
        Nice = thread.Nice;
        NumThreads = thread.NumThreads;
        Itrealvalue = thread.Itrealvalue;
        Starttime = thread.Starttime;
        Vsize = thread.Vsize;
        Rss.SetValue(thread.Rss);
        Rsslim = thread.Rsslim;
        Startcode = thread.Startcode;
        Endcode = thread.Endcode;
        Startstack = thread.Startstack;
        Kstkesp = thread.Kstkesp;
        Kstkeip = thread.Kstkeip;
        Signal = thread.Signal;
        Blocked = thread.Blocked;
        Sigignore = thread.Sigignore;
        Sigcatch = thread.Sigcatch;
        Wchan = thread.Wchan;
        Nswap.SetValue(thread.Nswap);
        Cnswap.SetValue(thread.Cnswap);
        ExitSignal = thread.ExitSignal;
        Processor = thread.Processor;
        RtPriority = thread.RtPriority;
        Policy = thread.Policy;
        DelayacctBlkioTicks.SetValue(thread.DelayacctBlkioTicks);
        GuestTime.SetValue(thread.GuestTime);
        CguestTime.SetValue(thread.CguestTime);
        StartData = thread.StartData;
        EndData = thread.EndData;
        StartBrk = thread.StartBrk;
        ArgStart = thread.ArgStart;
        ArgEnd = thread.ArgEnd;
        EnvStart = thread.EnvStart;
        EnvEnd = thread.EnvEnd;
        ExitCode = thread.ExitCode;
    }
};

Int64Array* AddColumnI64(TString name, Table* table) {
    auto* column = table->add_columns();
    column->set_title(std::move(name));
    return column->mutable_int64();
}

Uint64Array* AddColumnUi64(TString name, Table* table) {
    auto* column = table->add_columns();
    column->set_title(std::move(name));
    return column->mutable_uint64();
}

Float64Array* AddColumnDouble(TString name, Table* table) {
    auto* column = table->add_columns();
    column->set_title(std::move(name));
    return column->mutable_float64();
}

Uint64Array* AddColumnDuration(TString name, Table* table) {
    auto* column = table->add_columns();
    column->set_title(std::move(name));
    return column->mutable_duration();
}

Float64Array* AddColumnDataSize(TString name, Table* table) {
    auto* column = table->add_columns();
    column->set_title(std::move(name));
    return column->mutable_data_size();
}

class TThreadsPage: public NActors::TActorBootstrapped<TThreadsPage> {
public:
    TThreadsPage()
        : StatsProvider_(CreateThreadStatProvider())
    {
    }

    void Bootstrap() {
        Become(&TThis::StateFunc);
        Schedule(ThreadsInfoCollectionInterval_, new NActors::TEvents::TEvWakeup);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
            sFunc(NActors::TEvents::TEvWakeup, OnCollectThreadsInfo);
        }
    }

private:
    void OnRequest(const NSolomon::NSelfMon::TEvPageDataReq::TPtr& ev) {
        Page page;
        page.set_title("Threads");
        auto* grid = page.mutable_grid();
        if (!ErrorMessage_.empty()) {
            auto* field = grid->add_rows()->add_columns()->mutable_component()->mutable_object()->add_fields();
            field->set_name("Error");
            field->mutable_value()->set_string(ErrorMessage_);
        }
        auto* table = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* threadPidValue = AddColumnI64("Pid", table);

        auto* threadCommColumn = table->add_columns();
        threadCommColumn->set_title("Comm");
        auto* threadCommValue = threadCommColumn->mutable_string();

        auto* threadStateColumn = table->add_columns();
        threadStateColumn->set_title("State");
        auto* threadStateValue = threadStateColumn->mutable_string();

        auto* ppid = AddColumnI64("Ppid", table);
        auto* pgrp = AddColumnI64("Pgrp", table);
        auto* session = AddColumnI64("Session", table);
        auto* ttyNr = AddColumnI64("Tty_nr", table);
        auto* tpgid = AddColumnI64("Tpgid", table);
        auto* flags = AddColumnUi64("Flags", table);
        auto* minflt = AddColumnDouble("Minflt", table);
        auto* cminflt = AddColumnDouble("Cminflt", table);
        auto* majflt = AddColumnDouble("Majflt", table);
        auto* cmajflt = AddColumnDouble("Cmajflt", table);
        auto* utime = AddColumnDuration("Utime", table);
        auto* stime = AddColumnDuration("Stime", table);
        auto* cutime = AddColumnDuration("Cutime", table);
        auto* cstime = AddColumnDuration("Cstime", table);
        auto* priority = AddColumnI64("Priority", table);
        auto* nice = AddColumnI64("Nice", table);
        auto* numThreads = AddColumnI64("Num_threads", table);
        auto* itrealvalue = AddColumnDuration("Itrealvalue", table);
        auto* starttime = AddColumnDuration("Starttime", table);
        auto* vsize = AddColumnDataSize("Vsize", table);
        auto* rss = AddColumnDataSize("Rss", table);
        auto* rsslim = AddColumnDataSize("Rsslim", table);
        auto* startcode = AddColumnUi64("Startcode", table);
        auto* endcode = AddColumnUi64("Endcode", table);
        auto* startstack = AddColumnUi64("Startstack", table);
        auto* kstkesp = AddColumnUi64("Kstkesp", table);
        auto* kstkeip = AddColumnUi64("Kstkeip", table);
        auto* signal = AddColumnUi64("Signal", table);
        auto* blocked = AddColumnUi64("Blocked", table);
        auto* sigignore = AddColumnUi64("Sigignore", table);
        auto* sigcatch = AddColumnUi64("Sigcatch", table);
        auto* wchan = AddColumnUi64("Wchan", table);
        auto* nswap = AddColumnDouble("Nswap", table);
        auto* cnswap = AddColumnDouble("Cnswap", table);
        auto* exitSignal = AddColumnI64("Exit_signal", table);
        auto* processor = AddColumnI64("Processor", table);
        auto* rtPriority = AddColumnUi64("Rt_priority", table);
        auto* policy = AddColumnUi64("Policy", table);
        auto* delayacctBlkioTicks = AddColumnUi64("Delayacct_blkio_ticks", table);
        auto* guestTime = AddColumnDouble("Guest_time", table);
        auto* cguestTime = AddColumnDouble("Cguest_time", table);
        auto* startData = AddColumnUi64("Start_data", table);
        auto* endData = AddColumnUi64("End_data", table);
        auto* startBrk = AddColumnUi64("Start_brk", table);
        auto* argStart = AddColumnUi64("Arg_start", table);
        auto* argEnd = AddColumnUi64("Arg_end", table);
        auto* envStart = AddColumnUi64("Env_start", table);
        auto* envEnd = AddColumnUi64("Env_end", table);
        auto* exitCode = AddColumnI64("Exit_code", table);

        for (auto& [_, threadStat]: ThreadsStats_) {
            threadPidValue->add_values(threadStat.Pid);
            threadCommValue->add_values(threadStat.Comm);
            threadStateValue->add_values(ToString(threadStat.State));
            ppid->add_values(threadStat.Ppid);
            pgrp->add_values(threadStat.Pgrp);
            session->add_values(threadStat.Session);
            ttyNr->add_values(threadStat.TtyNr);
            tpgid->add_values(threadStat.Tpgid);
            flags->add_values(threadStat.Flags);
            minflt->add_values(threadStat.Minflt.RateChange);
            cminflt->add_values(threadStat.Cminflt.RateChange);
            majflt->add_values(threadStat.Majflt.RateChange);
            cmajflt->add_values(threadStat.Cmajflt.RateChange);
            utime->add_values(threadStat.Utime);
            stime->add_values(threadStat.Stime);
            cutime->add_values(threadStat.Cutime);
            cstime->add_values(threadStat.Cstime);
            priority->add_values(threadStat.Priority);
            nice->add_values(threadStat.Nice);
            numThreads->add_values(threadStat.NumThreads);
            itrealvalue->add_values(threadStat.Itrealvalue);
            starttime->add_values(threadStat.Starttime);
            vsize->add_values(static_cast<double>(threadStat.Vsize));
            rss->add_values(threadStat.Rss.RateChange);
            rsslim->add_values(static_cast<double>(threadStat.Rsslim));
            startcode->add_values(threadStat.Startcode);
            endcode->add_values(threadStat.Endcode);
            startstack->add_values(threadStat.Startstack);
            kstkesp->add_values(threadStat.Kstkesp);
            kstkeip->add_values(threadStat.Kstkeip);
            signal->add_values(threadStat.Signal);
            blocked->add_values(threadStat.Blocked);
            sigignore->add_values(threadStat.Sigignore);
            sigcatch->add_values(threadStat.Sigcatch);
            wchan->add_values(threadStat.Wchan);
            nswap->add_values(threadStat.Nswap.RateChange);
            cnswap->add_values(threadStat.Cnswap.RateChange);
            exitSignal->add_values(threadStat.ExitSignal);
            processor->add_values(threadStat.Processor);
            rtPriority->add_values(threadStat.RtPriority);
            policy->add_values(threadStat.Policy);
            delayacctBlkioTicks->add_values(threadStat.DelayacctBlkioTicks.PrevValue);
            guestTime->add_values(threadStat.GuestTime.RateChange);
            cguestTime->add_values(threadStat.CguestTime.RateChange);
            startData->add_values(threadStat.StartData);
            endData->add_values(threadStat.EndData);
            startBrk->add_values(threadStat.StartBrk);
            argStart->add_values(threadStat.ArgStart);
            argEnd->add_values(threadStat.ArgEnd);
            envStart->add_values(threadStat.EnvStart);
            envEnd->add_values(threadStat.EnvEnd);
            exitCode->add_values(threadStat.ExitCode);
        }

        Send(ev->Sender, new TEvPageDataResp{std::move(page)});
    }

    void OnCollectThreadsInfo() {
        Schedule(ThreadsInfoCollectionInterval_, new NActors::TEvents::TEvWakeup);

        auto now = TActivationContext::Now();
        if (TActivationContext::Now() - ErrorTimeHappened_  > TDuration::Hours(2)) {
            ErrorMessage_.clear();
        }
        auto threadIds = StatsProvider_->GetSelfThreads();
        if (threadIds.Success()) {
            THashSet<ui64> threadsSet;
            TVector<ui64> trash;
            for (const auto& threadId: threadIds.Value()) {
                threadsSet.insert(threadId);
            }
            for (const auto& [id, _]: ThreadsStats_) {
                if (threadsSet.find(id) == threadsSet.end()) {
                    trash.push_back(id);
                }
            }
            for (const auto& trashId: trash) {
                ThreadsStats_.erase(trashId);
            }
            for (const auto& threadId: threadIds.Value()) {
                auto threadInfo = StatsProvider_->GetThreadStats(threadId);
                if (threadInfo.Success()) {
                    ThreadsStats_[threadId].Change(threadInfo.Value());
                } else {
                    ErrorTimeHappened_ = now;
                    ErrorMessage_ = threadInfo.Error().MessageString();
                }
            }
        } else {
            ErrorTimeHappened_ = now;
            ErrorMessage_ = threadIds.Error().MessageString();
        }
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    TDuration ThreadsInfoCollectionInterval_ = TDuration::Seconds(5);
    THashMap<ui64, TThreadStatsRate> ThreadsStats_;
    TString ErrorMessage_;
    TInstant ErrorTimeHappened_;
    std::unique_ptr<IThreadStatProvider> StatsProvider_;
};

} // namespace

std::unique_ptr<NActors::IActor> ThreadsPage() {
    return std::make_unique<TThreadsPage>();
}

} // namespace NSolomon::NSelfMon
