#include "render.h"

#include <solomon/libs/cpp/selfmon/model/component.pb.h>
#include <solomon/libs/cpp/selfmon/view/prefix.h>
#include <library/cpp/html/escape/escape.h>
#include <library/cpp/json/json_writer.h>

using namespace yandex::monitoring::selfmon;

namespace NSolomon::NSelfMon {

template <>
void Render<Heading>(const TRenderContext&, const Heading& h, IOutputStream* os) {
    *os << "<h" << h.level() << " class='pt-2 pb-2'>" << h.content() << "</h" << h.level() << '>';
}

template <>
void Render<Code>(const TRenderContext&, const Code& c, IOutputStream* os) {
    *os << "<pre class='border rounded'><code class='hljs";
    if (!c.lang().empty()) {
        *os << " language-" << c.lang();
    }
    *os << "'>" << c.content() << "</code></pre>";
}

template <>
void Render<Tabs>(const TRenderContext&, const Tabs& t, IOutputStream* os) {
    *os << "<ul class='nav nav-tabs mb-3'>";
    for (const auto& tab: t.tabs()) {
        const auto& ref = tab.reference();
        *os << "<li class='nav-item'><a class='nav-link";
        if (tab.active()) {
            *os << " active";
        } else if (tab.disabled()) {
            *os << " disable";
        }
        *os << "' href='" << BasePrefix << ref.page();
        if (!ref.args().empty()) {
            *os << '?' << ref.args();
        }
        *os << "'>" << ref.title() << "</a></li>";
    }
    *os << "</ul>";
}

NJson::TJsonValue ConvertToJsonValue(const TString& nodeName, const FlameGraph& fg) {
    NJson::TJsonValue v;
    v["value"] = fg.value();
    v["name"] = nodeName;
    if (!fg.has_children()) {
        return v;
    }
    for (auto& child: fg.children().graph()) {
        v["children"].AppendValue(ConvertToJsonValue(child.first, child.second));
    }
    return v;
};

template <>
void Render<FlameGraph>(const TRenderContext&, const FlameGraph& fg, IOutputStream* os) {
    *os << "<div id=\"chart\"></div>";
    *os << "<script type=text/javascript src='https://yastatic.net/d3/4.5.0/d3.min.js'></script>"
           "<script type=text/javascript src='/assets/d3-flamegraph.min.js'></script>";
    auto value = ConvertToJsonValue("root", fg);
    *os << "<script type=text/javascript>"
           "var chart = flamegraph();\n"
           "var data = ";
    NJson::WriteJson(os, &value);
    *os << ";\n"
           "d3.select(\"#chart\").datum(data).call(chart);\n"
           "</script>";
}

static NJson::TJsonValue ConvertToJsonValue(const TracePlot& tp) {
    NJson::TJsonArray array;
    for (auto& span: tp.spans()) {
        NJson::TJsonValue v;
        v["id"] = span.id();
        v["parent"] = span.parent();
        v["begin"] = span.begin();
        if (span.finished()) {
            v["end"] = span.end();
        }
        v["descr"] = span.description();
        array.AppendValue(v);
    }
    return array;
};

template <>
void Render<TracePlot>(const TRenderContext&, const TracePlot& tp, IOutputStream* os) {
    *os << "<div id=\"chart\"></div>"
           "<script type=text/javascript src='https://yastatic.net/d3/4.5.0/d3.min.js'></script>"
           "<script type=text/javascript src='/assets/d3-traceplot.js'></script>";
    auto value = ConvertToJsonValue(tp);
    *os << "<script type=text/javascript>"
           "var chart = traceplot().rowHeight(20);\n"
           "var data = ";
    NJson::WriteJson(os, &value);
    *os << ";\n"
           "d3.select(\"#chart\").datum(data).call(chart);\n"
           "</script>";
}

template <>
void Render<Component>(const TRenderContext& ctx, const Component& component, IOutputStream* os) {
    switch (component.type_case()) {
        case Component::kValue:
            Render<Value>(ctx, component.value(), os);
            break;
        case Component::kObject:
            Render<Object>(ctx, component.object(), os);
            break;
        case Component::kList:
            Render<List>(ctx, component.list(), os);
            break;
        case Component::kListGroup:
            Render<ListGroup>(ctx, component.list_group(), os);
            break;
        case Component::kTable:
            Render<Table>(ctx, component.table(), os);
            break;
        case Component::kHeading:
            Render<Heading>(ctx, component.heading(), os);
            break;
        case Component::kForm:
            Render<Form>(ctx, component.form(), os);
            break;
        case Component::kCode:
            Render<Code>(ctx, component.code(), os);
            break;
        case Component::kTabs:
            Render<Tabs>(ctx, component.tabs(), os);
            break;
        case Component::kCollapsible:
            Render<Collapsible>(ctx, component.collapsible(), os);
            break;
        case Component::kCollapsibleList:
            Render<CollapsibleList>(ctx, component.collapsible_list(), os);
            break;
        case Component::kFlameGraph:
            Render<FlameGraph>(ctx, component.flame_graph(), os);
            break;
        case Component::kTracePlot:
            Render<TracePlot>(ctx, component.trace_plot(), os);
            break;
        case Component::TYPE_NOT_SET:
            *os << "Component is empty";
            break;
    }
}

} // namespace NSolomon::NSelfMon
