#include "render.h"
#include "style.h"

#include <solomon/libs/cpp/selfmon/model/form.pb.h>
#include <solomon/libs/cpp/selfmon/view/prefix.h>
#include <library/cpp/html/escape/escape.h>

using namespace yandex::monitoring::selfmon;

namespace NSolomon::NSelfMon {
namespace {

TStringBuf InputTypeStr(InputType type) {
    switch (type) {
        case Text: return "text";
        case Email: return "email";
        case IntNumber: return "number";
        case FloatNumber: return "number";
        case Password: return "password";
        case Hidden: return "hidden";

        case InputType_INT_MIN_SENTINEL_DO_NOT_USE_:
        case InputType_INT_MAX_SENTINEL_DO_NOT_USE_:
            Y_FAIL();
    }
}

TStringBuf GetName(const FormItem& item) {
    switch (item.type_case()) {
        case FormItem::kInput:
            return item.input().name();
        case FormItem::kTextarea:
            return item.textarea().name();
        case FormItem::kSelect:
            return item.select().name();
        case FormItem::kCheck:
            return item.check().name();
        case FormItem::TYPE_NOT_SET:
            return "empty form item";
    }
}

void RenderInput(const FormInput& input, IOutputStream* os) {
    *os << "<input type='" << InputTypeStr(input.type()) << '\'';

    if (input.type() == InputType::IntNumber) {
        *os << " step='1'";
    } else if (input.type() == InputType::FloatNumber) {
        *os << " step='any'";
    }

    *os << " class='form-control' id='" << input.name()
        << "' name='" << input.name()
        << "' value=\"" << NHtml::EscapeAttributeValue(input.value());

    if (const auto& p = input.placeholder()) {
        *os << "\" placeholder=\"" << NHtml::EscapeAttributeValue(p);
    }

    *os << "\"/>";
}

void RenderTextarea(const FormTextarea& textarea, IOutputStream* os) {
    *os << "<textarea class='form-control' id='" << textarea.name()
        << "' name='" << textarea.name() << '\'';
    if (auto rows = textarea.rows()) {
        *os << " rows=" << rows;
    }
    if (auto cols = textarea.cols()) {
        *os << " cols=" << cols;
    }
    if (const auto& p = textarea.placeholder()) {
        *os << " placeholder=\"" << NHtml::EscapeAttributeValue(p) << '"';
    }
    *os << '>' << textarea.value() << "</textarea>";
}

void RenderSelect(const FormSelect& select, IOutputStream* os) {
    *os << "<select class='form-select' id='" << select.name() << "' name='" << select.name() << "'>";
    for (const auto& o: select.options()) {
        *os << "<option value='" << o.value() << (o.selected() ? "' selected>" : "'>");
        *os << (o.title() ? o.title(): o.value());
        *os << "</option>";
    }
    *os << "</select>";
}

void RenderCheck(const FormItem& item, IOutputStream* os) {
    const auto& check = item.check();
    *os << "<input class='form-check-input' type='checkbox' ";
    *os << "id='" << check.name() << "' name='" << check.name();
    *os << (check.value() ? "' checked>" : "'>");
    if (const auto& l = item.label()) {
        *os << "<label class='form-check-label' for='" << check.name() << "'>" << l << "</label>";
    }
}

void RenderSubmit(const FormSubmit& s, IOutputStream* os) {
    auto style = (s.style() == Style::None) ? Style::Primary : s.style();
    *os << "<button type='submit' class='btn btn-outline-" << StyleClass(style) << '\'';
    if (!s.name().empty()) {
        *os << " name='" << s.name() << '\'';
    }
    if (!s.value().empty()) {
        *os << " value='" << s.value() << '\'';
    }
    *os << '>' << s.title() << "</button>";
}

void RenderVerticalForm(const Form& f, IOutputStream* os) {
    auto withLabel = [os](const FormItem& item, auto&& fn) {
        *os << "<div class='mb-3'>";
        if (const auto& l = item.label()) {
            *os << "<label for='" << GetName(item) << "' class='form-label'>" << l << "</label>";
        }
        fn();
        if (item.help()) {
            *os << "<div class='form-text'>" << item.help() << "</div>";
        }
        *os << "</div>";
    };

    for (const auto& item: f.items()) {
        switch (item.type_case()) {
            case FormItem::kInput:
                if (item.input().type() == InputType::Hidden) {
                    // do not output label for hidden input
                    RenderInput(item.input(), os);
                } else {
                    withLabel(item, [&item, os] {
                        RenderInput(item.input(), os);
                    });
                }
                break;
            case FormItem::kTextarea:
                withLabel(item, [&item, os] {
                    RenderTextarea(item.textarea(), os);
                });
                break;
            case FormItem::kSelect:
                withLabel(item, [&item, os] {
                    RenderSelect(item.select(), os);
                });
                break;
            case FormItem::kCheck:
                *os << "<div class='mb-3 form-check'>";
                RenderCheck(item, os);
                if (item.help()) {
                    *os << "<div class='form-text'>" << item.help() << "</div>";
                }
                *os << "</div>";
                break;
            case FormItem::TYPE_NOT_SET:
                break;
        }
    }

    for (const auto& s: f.submit()) {
        RenderSubmit(s, os);
    }
}

void RenderHorizontalForm(const Form& f, IOutputStream* os) {
    auto withLabel = [os](const FormItem& item, auto&& fn) {
        *os << "<div class='col-auto'>";
        if (const auto& l = item.label()) {
            *os << "<label for='" << GetName(item) << "'>" << l << "</label>";
        }
        fn();
        *os << "</div>";
    };

    for (const auto& item: f.items()) {
        switch (item.type_case()) {
            case FormItem::kInput:
                if (item.input().type() == InputType::Hidden) {
                    // do not output label for hidden input
                    RenderInput(item.input(), os);
                } else {
                    withLabel(item, [&item, os] {
                        RenderInput(item.input(), os);
                    });
                }
                break;
            case FormItem::kTextarea:
                withLabel(item, [&item, os] {
                    RenderTextarea(item.textarea(), os);
                });
                break;
            case FormItem::kSelect:
                withLabel(item, [&item, os] {
                    RenderSelect(item.select(), os);
                });
                break;
            case FormItem::kCheck:
                *os << "<div class='col-auto form-check'>";
                RenderCheck(item, os);
                *os << "</div>";
                break;
            case FormItem::TYPE_NOT_SET:
                break;
        }
    }

    for (const auto& s: f.submit()) {
        *os << "<div class='col-auto'>";
        RenderSubmit(s, os);
        *os << "</div>";
    }
}

} // namespace

template <>
void Render<Form>(const TRenderContext&, const Form& f, IOutputStream* os) {
    *os << "<form";
    if (const auto& a = f.action()) {
        *os << " action='" << BasePrefix << a << '\'';
    }
    *os << " method='" << FormMethod_Name(f.method()) << '\'';
    if (f.layout() == FormLayout::Horizontal) {
        *os << " class='row'";
    }
    *os << '>';

    if (f.layout() == FormLayout::Vertical) {
        RenderVerticalForm(f, os);
    } else if (f.layout() == FormLayout::Horizontal) {
        RenderHorizontalForm(f, os);
    }
    *os << "</form>";
}

} // namespace NSolomon::NSelfMon
