#include "render.h"

#include <solomon/libs/cpp/selfmon/model/component.pb.h>

#include <util/datetime/base.h>
#include <util/generic/yexception.h>

#define GET_SIZE(typeCase, field) \
    case Table_Column::typeCase: \
        return col.field().values_size()

#define RENDER_CELL(typeCase, field) \
    case Table_Column::typeCase: { \
            const auto& val = col.field().values(row); \
            Render<std::decay_t<decltype(val)>>(ctx, val, os); \
        } \
        break

#define SORT_BY(typeCase, field) \
    case Table_Column::typeCase: \
        std::sort(permutations.begin(), permutations.end(), TLessByColumn{col.field()}); \
        break

using namespace yandex::monitoring::selfmon;

namespace NSolomon::NSelfMon {
namespace {

constexpr auto SortByParamKey = "sortBy";
constexpr auto OffsetParamKey = "offset";
constexpr int LIMIT = 100;
constexpr int MAX_PAGES = 7;

int GetColumnSize(const Table_Column& col) {
    switch (col.type_case()) {
        GET_SIZE(kBoolean, boolean);
        GET_SIZE(kInt32, int32);
        GET_SIZE(kUint32, uint32);
        GET_SIZE(kInt64, int64);
        GET_SIZE(kUint64, uint64);
        GET_SIZE(kFloat32, float32);
        GET_SIZE(kFloat64, float64);
        GET_SIZE(kString, string);
        GET_SIZE(kBytes, bytes);
        GET_SIZE(kReference, reference);
        GET_SIZE(kTime, time);
        GET_SIZE(kDuration, duration);
        GET_SIZE(kDataSize, data_size);
        GET_SIZE(kComponents, components);
        GET_SIZE(kFractions, fractions);

        case Table_Column::TYPE_NOT_SET:
            return -1;
    }
    Y_UNREACHABLE();
}

void RenderCell(const TRenderContext& ctx, const Table_Column& col, int row, IOutputStream* os) {
    switch (col.type_case()) {
        RENDER_CELL(kBoolean, boolean);
        RENDER_CELL(kInt32, int32);
        RENDER_CELL(kUint32, uint32);
        RENDER_CELL(kInt64, int64);
        RENDER_CELL(kUint64, uint64);
        RENDER_CELL(kFloat32, float32);
        RENDER_CELL(kFloat64, float64);
        RENDER_CELL(kString, string);
        RENDER_CELL(kBytes, bytes);
        RENDER_CELL(kReference, reference);
        RENDER_CELL(kComponents, components);
        RENDER_CELL(kFractions, fractions);

        case Table_Column::kTime:
            Render<TInstant>(ctx, TInstant::FromValue(col.time().values(row)), os);
            break;

        case Table_Column::kDuration:
            Render<TDuration>(ctx, TDuration::FromValue(col.duration().values(row)), os);
            break;

        case Table_Column::kDataSize:
            Render<TDataSize>(ctx, TDataSize{col.data_size().values(row)}, os);
            break;

        case Table_Column::TYPE_NOT_SET:
            *os << "cell value is not set";
            break;
    }
}

enum class ESortDirection {
    None,
    Asc,
    Desc,
};

ESortDirection OutputQueryWithSortParams(TStringBuf query, int col, IOutputStream* os) {
    int sortBy = col;
    auto sortDir = ESortDirection::None;

    auto outputParam = [idx{0}, os](auto key, auto val) mutable {
        if (idx++ > 0) {
            *os << '&';
        }
        *os << key << '=' << val;
    };

    while (!query.empty()) {
        auto val = query.NextTok('&');
        if (val.empty()) {
            continue; // && case
        }

        auto key = val.NextTok('=');
        if (key == SortByParamKey) {
            int prevSortBy = FromString<int>(val);
            if (std::abs(prevSortBy) == col) {
                // reverse the order
                sortBy = -prevSortBy;
                sortDir = (sortBy < 0) ? ESortDirection::Desc : ESortDirection::Asc;
            }
        } else {
            outputParam(key, val);
        }
    }

    outputParam(SortByParamKey, sortBy);
    return sortDir;
}

template <typename T>
struct TLessByColumn {
    const T& Column;

    explicit TLessByColumn(const T& column) noexcept
        : Column{column}
    {
    }

    bool operator()(int a, int b) const noexcept {
        return Column.values(a) < Column.values(b);
    }
};

std::vector<int> SortByColumn(const Table_Column& col) {
    std::vector<int> permutations(GetColumnSize(col));
    std::iota(permutations.begin(), permutations.end(), 0);

    switch (col.type_case()) {
        SORT_BY(kBoolean, boolean);
        SORT_BY(kInt32, int32);
        SORT_BY(kUint32, uint32);
        SORT_BY(kInt64, int64);
        SORT_BY(kUint64, uint64);
        SORT_BY(kFloat32, float32);
        SORT_BY(kFloat64, float64);
        SORT_BY(kString, string);
        SORT_BY(kBytes, bytes);
        SORT_BY(kTime, time);
        SORT_BY(kDuration, duration);
        SORT_BY(kDataSize, data_size);

        case Table_Column::kFractions:
            std::sort(permutations.begin(), permutations.end(), [&col](int a, int b) {
                auto& valueA = col.fractions().values(a);
                auto& valueB = col.fractions().values(b);
                return valueA.num() * valueB.denom() < valueB.num() * valueA.denom();
            });
            break;

        case Table_Column::kReference:
            // sort by displayable title or by url
            std::sort(permutations.begin(), permutations.end(), [&col](int a, int b) {
                auto& valueA = col.reference().values(a);
                auto& valueB = col.reference().values(b);
                if (!valueA.title().empty() && valueB.title().empty()) {
                    return valueA.title() < valueB.title();
                }
                TString urlA = valueA.page() + valueA.args();
                TString urlB = valueB.page() + valueB.args();
                return urlA < urlB;
            });
            break;

        case Table_Column::kComponents:
            // do not sort
            break;

        case Table_Column::TYPE_NOT_SET:
            ythrow yexception() << "column " << col.title() << " has no values";
    }

    return permutations;
}

void WriteQueryWithOverridedOffset(TStringBuf query, size_t offset, IOutputStream* os) {
    auto outputParam = [idx{0}, os](auto key, auto val) mutable {
        if (idx++ > 0) {
            *os << '&';
        }
        *os << key << '=' << val;
    };

    while (!query.empty()) {
        auto val = query.NextTok('&');
        if (val.empty()) {
            continue; // && case
        }

        auto key = val.NextTok('=');
        if (key != OffsetParamKey) {
            outputParam(key, val);
        }
    }

    outputParam(OffsetParamKey, offset);
}

class TTableRenderer {
public:
    TTableRenderer(const TRenderContext& ctx, const Table& table, IOutputStream* os) noexcept
        : Ctx_{ctx}
        , Table_{table}
        , Os_{os}
    {
    }

    void RenderHeader() {
        *Os_ << "<thead><tr>";
        if (Table_.numbered()) {
            *Os_ << "<th scope='col'>#</th>";
        }

        for (int idx = 0; idx < Table_.columns_size(); ++idx) {
            const auto& col = Table_.columns(idx);

            *Os_ << "<th scope='col'>";
            if (!Table_.non_sortable()) {
                *Os_ << "<a href='" << Ctx_.Url << '?';
                auto colSortDir = OutputQueryWithSortParams(Ctx_.Query, idx+1, Os_);
                *Os_ << "'>";
                if (colSortDir == ESortDirection::Asc) {
                    *Os_ << "<i class='bi bi-sort-down-alt'></i> ";
                    SortBy_ = idx;
                    SortDir_ = colSortDir;
                } else if (colSortDir == ESortDirection::Desc) {
                    *Os_ << "<i class='bi bi-sort-up'></i> ";
                    SortBy_ = idx;
                    SortDir_ = colSortDir;
                }
                *Os_ << col.title() << "</a>";
            } else {
                *Os_ << col.title();
            }
            *Os_ << "</th>";

            int size = GetColumnSize(col);
            Y_ENSURE(size != -1, "column " << col.title() << " has no values");

            if (Rows_ != -1) {
                Y_ENSURE(Rows_ == size, "column " << col.title()
                        << " has different size (" << size << " != " << Rows_ << ')');
            } else {
                Rows_ = size;
            }
        }
        *Os_ << "</tr></thead>";
    }

    void RenderBody() {
        *Os_ << "<tbody>";

        auto renderRow = [rowNum{0}, this](int rowIdx, int offset) mutable {
            *Os_ << "<tr>";
            if (Table_.numbered()) {
                *Os_ << "<th scope='row'>" << ++rowNum + offset << "</th>";
            }
            for (const auto& col: Table_.columns()) {
                *Os_ << "<td>";
                RenderCell(Ctx_, col, rowIdx, Os_);
                *Os_ << "</td>";
            }
            *Os_ << "</tr>";
        };

        std::vector<int> rowsOrder;
        if (SortDir_ != ESortDirection::None) {
            rowsOrder = SortByColumn(Table_.columns(SortBy_));
        }

        int offset = 0;
        TryFromString(Ctx_.Param(OffsetParamKey), offset);
        if (SortDir_ == ESortDirection::Asc) {
            for (size_t i = offset; i < Min<size_t>(offset + LIMIT, rowsOrder.size()); ++i) {
                renderRow(rowsOrder[i], offset);
            }
        } else if (SortDir_ == ESortDirection::Desc) {
            int start = static_cast<int>(rowsOrder.size()) - offset - 1;
            int end = Max(0, static_cast<int>(rowsOrder.size()) - offset - LIMIT);
            for (int i = start; i >= end; --i) {
                renderRow(rowsOrder[i], offset);
            }
        } else {
            for (int row = offset; row < Min(Rows_, offset + LIMIT); ++row) {
                renderRow(row, offset);
            }
        }

        *Os_ << "</tbody>";
    }

    void RenderPagination() {
        Y_ENSURE(Rows_ >= 0);
        if (Rows_ <= LIMIT) {
            return;
        }

        int offset = 0;
        TryFromString(Ctx_.Param(OffsetParamKey), offset);

        int pagesCount = (Rows_ + LIMIT - 1) / LIMIT;
        int activePage = offset / LIMIT;

        int leftBound = 0;
        int rightBound = pagesCount;
        if (pagesCount > MAX_PAGES) {
            leftBound = Max(leftBound, activePage - MAX_PAGES / 2);
            rightBound = Min(rightBound, leftBound + MAX_PAGES);
        }

        *Os_ << "<nav id='pagination'>";
        *Os_ << "<ul class='pagination'>";

        *Os_ << "<li class='page-item'><a class='page-link' href='" << Ctx_.Url << "?";
        WriteQueryWithOverridedOffset(Ctx_.Query, Max(0, activePage - 1) * LIMIT, Os_);
        *Os_ << "#pagination'>Previous</a></li>";

        if (leftBound > 0) {
            *Os_ << "<li class='page-item'><a class='page-link' href='" << Ctx_.Url << "?";
            WriteQueryWithOverridedOffset(Ctx_.Query, 0, Os_);
            *Os_ << "'#pagination>1...</a></li>";
        }

        for (int page = leftBound; page < rightBound; ++page) {
            *Os_ << "<li class='page-item";
            if (page == activePage) {
                *Os_ << " active";
            }
            *Os_ << "'><a class='page-link' href='" << Ctx_.Url << "?";
            WriteQueryWithOverridedOffset(Ctx_.Query, page * LIMIT, Os_);
            *Os_ << "#pagination'>" << page + 1 << "</a></li>";
        }

        if (rightBound < pagesCount) {
            *Os_ << "<li class='page-item'><a class='page-link' href='" << Ctx_.Url << "?";
            WriteQueryWithOverridedOffset(Ctx_.Query, (pagesCount - 1) * LIMIT, Os_);
            *Os_ << "#pagination'>..." << pagesCount << "</a></li>";
        }

        *Os_ << "<li class='page-item'><a class='page-link' href='" << Ctx_.Url << "?";
        WriteQueryWithOverridedOffset(Ctx_.Query, Min(pagesCount - 1, activePage + 1) * LIMIT, Os_);
        *Os_ << "#pagination'>Next</a></li>";

        *Os_ << "</ul></nav>";
    }

private:
    const TRenderContext& Ctx_;
    const Table& Table_;
    IOutputStream* Os_;
    int Rows_{-1};
    int SortBy_{0};
    ESortDirection SortDir_{ESortDirection::None};
};

} // namespace

template <>
void Render<Table>(const TRenderContext& ctx, const Table& table, IOutputStream* os) {
    Y_ENSURE(table.columns_size() != 0, "table has no columns");

    if (!table.description().empty()) {
        *os << "<div>" << table.description() << "</div>";
    }

    TTableRenderer r{ctx, table, os};

    *os << "<table class='table'>";
    r.RenderHeader();
    r.RenderBody();
    *os << "</table>";
    r.RenderPagination();
}

} // namespace NSolomon::NSelfMon
