#include "render.h"

#include <solomon/libs/cpp/selfmon/model/value.pb.h>
#include <solomon/libs/cpp/selfmon/view/prefix.h>

#include <util/datetime/base.h>

using namespace yandex::monitoring::selfmon;

namespace NSolomon::NSelfMon {

#define RENDER_TYPE(typeCase, typeName) \
    case Value::typeCase: \
        *os << "(" typeName ")"; \
        break

#define RENDER_VALUE(typeCase, field) \
    case Value::typeCase: { \
            const auto& val = value.field(); \
            Render<std::decay_t<decltype(val)>>(ctx, val, os); \
        } \
        break

#define RENDER_ARRAY_VALUE(typeCase, field) \
    case ArrayValue::typeCase: \
        RenderArray(value.field().values(), os, [&ctx, os](auto& value) { \
            Render<std::decay_t<decltype(value)>>(ctx, value, os); \
        }); \
        break

template <>
void Render<Reference>(const TRenderContext&, const Reference& ref, IOutputStream* os) {
    *os << "<a href='";
    if (ref.page().StartsWith("http://") || ref.page().StartsWith("https://")) {
        *os << ref.page();
    } else {
        *os << BasePrefix;
        if (!ref.page().StartsWith('/')) {
            *os << '/';
        }
        *os << ref.page();
    }
    if (!ref.args().empty()) {
        *os << '?' << ref.args();
    }
    *os << "'>";

    if (!ref.title().empty()) {
        *os << ref.title();
    } else {
        *os << ref.page();
    }
    *os << "</a>";
}

template <>
void Render<Fraction>(const TRenderContext&, const Fraction& element, IOutputStream* os) {
    switch (element.view()) {
        case Fraction::AsIs:
            *os << Sprintf("%.2f", element.num() / element.denom());
            break;

        case Fraction::AsPercent:
            *os << Sprintf("%.2f%%", element.num() / element.denom() * 100);
            break;

        case Fraction::AsProgress:
        {
            TString percent = Sprintf("%.2f", element.num() / element.denom() * 100);
            *os << "<div class='progress' style='position: relative;'>";
                *os << "<div class='progress-bar bg-info' role='progressbar' aria-valuenow='" << percent
                    << "' aria-valuemin='0' aria-valuemax='100' style='width: " << percent << "%;'></div>";
                *os << "<div class='progress-bar-title'>" << percent << "%</div>";
            *os << "</div>";
            break;
        }
        case Fraction_View_Fraction_View_INT_MIN_SENTINEL_DO_NOT_USE_:
        case Fraction_View_Fraction_View_INT_MAX_SENTINEL_DO_NOT_USE_:
            Y_FAIL();
    }
}

template <>
void Render<Value::TypeCase>(const TRenderContext&, Value::TypeCase type, IOutputStream* os) {
    switch (type) {
        RENDER_TYPE(kBoolean, "bool");
        RENDER_TYPE(kInt32, "i32");
        RENDER_TYPE(kUint32, "ui32");
        RENDER_TYPE(kInt64, "i64");
        RENDER_TYPE(kUint64, "ui64");
        RENDER_TYPE(kFloat32, "float");
        RENDER_TYPE(kFloat64, "double");
        RENDER_TYPE(kString, "string");
        RENDER_TYPE(kBytes, "bytes");
        RENDER_TYPE(kReference, "ref");
        RENDER_TYPE(kTime, "time");
        RENDER_TYPE(kDuration, "duration");
        RENDER_TYPE(kDataSize, "data_size");
        RENDER_TYPE(kFraction, "fraction");

        case Value::TYPE_NOT_SET:
            *os << "value is not set";
            break;
    }
}

template <>
void Render<Value>(const TRenderContext& ctx, const Value& value, IOutputStream* os) {
    switch (value.type_case()) {
        RENDER_VALUE(kBoolean, boolean);
        RENDER_VALUE(kInt32, int32);
        RENDER_VALUE(kUint32, uint32);
        RENDER_VALUE(kInt64, int64);
        RENDER_VALUE(kUint64, uint64);
        RENDER_VALUE(kFloat32, float32);
        RENDER_VALUE(kFloat64, float64);
        RENDER_VALUE(kString, string);
        RENDER_VALUE(kBytes, bytes);
        RENDER_VALUE(kReference, reference);
        RENDER_VALUE(kFraction, fraction);

        case Value::kTime:
            Render<TInstant>(ctx, TInstant::FromValue(value.time()), os);
            break;

        case Value::kDuration:
            Render<TDuration>(ctx, TDuration::FromValue(value.duration()), os);
            break;

        case Value::kDataSize:
            Render<TDataSize>(ctx, TDataSize{value.data_size()}, os);
            break;

        case Value::TYPE_NOT_SET:
            *os << "value is not set";
            break;
    }
}

template <typename TRepeatedField, typename TRenderer>
void RenderArray(const TRepeatedField& field, IOutputStream* os, TRenderer&& renderer) {
    constexpr int K = 5;

    *os << '[';
    if (field.size() > K * 2) {
        // first K elements
        for (int i = 0; i < K; i++) {
            renderer(field[i]);
            *os << ", ";
        }

        *os << "...";

        // last K elements
        for (int i = field.size() - K - 1; i < field.size(); i++) {
            *os << ", ";
            renderer(field[i]);
        }
    } else {
        for (int i = 0; i < field.size(); i++) {
            if (i > 0) {
                *os << ", ";
            }
            renderer(field[i]);
        }
    }
    *os << ']';
}

template <>
void Render<ArrayValue>(const TRenderContext& ctx, const ArrayValue& value, IOutputStream* os) {
    switch (value.type_case()) {
        RENDER_ARRAY_VALUE(kBoolean, boolean);
        RENDER_ARRAY_VALUE(kInt32, int32);
        RENDER_ARRAY_VALUE(kUint32, uint32);
        RENDER_ARRAY_VALUE(kInt64, int64);
        RENDER_ARRAY_VALUE(kUint64, uint64);
        RENDER_ARRAY_VALUE(kFloat32, float32);
        RENDER_ARRAY_VALUE(kFloat64, float64);
        RENDER_ARRAY_VALUE(kString, string);
        RENDER_ARRAY_VALUE(kBytes, bytes);
        RENDER_ARRAY_VALUE(kReference, reference);
        RENDER_ARRAY_VALUE(kFraction, fraction);

        case ArrayValue::kTime:
            RenderArray(value.time().values(), os, [&ctx, os](ui64 value) {
                Render<TInstant>(ctx, TInstant::FromValue(value), os);
            });
            break;

        case ArrayValue::kDuration:
            RenderArray(value.duration().values(), os, [&ctx, os](ui64 value) {
                Render<TDuration>(ctx, TDuration::FromValue(value), os);
            });
            break;

        case ArrayValue::kDataSize:
            RenderArray(value.data_size().values(), os, [&ctx, os](double value) {
                Render<TDataSize>(ctx, TDataSize{value}, os);
            });
            break;

        case ArrayValue::TYPE_NOT_SET:
            *os << "value is not set";
            break;
    }
}

} // namespace NSolomon::NSelfMon
