#include "immutable_registry.h"

#include <library/cpp/containers/stack_vector/stack_vec.h>
#include <library/cpp/monlib/encode/text/text.h>
#include <library/cpp/monlib/metrics/histogram_snapshot.h>

#include <util/stream/output.h>
#include <util/generic/array_ref.h>

#include <array>

using namespace NMonitoring;

namespace NSolomon {
namespace {
using THistBounds = std::array<double, HISTOGRAM_MAX_BUCKETS_COUNT>;

class THistogramCollector: public IHistogramCollector {
public:
    THistogramCollector(TArrayRef<double> bounds)
        : Values_{}
        , Size_{bounds.size()}
    {
        CopyN(bounds.begin(), Size_, Bounds_.data());
        Bounds_[bounds.size()] = Max<TBucketBound>();
    }

    void Collect(double value, ui64 count) override {
        auto it = LowerBound(begin(Bounds_), end(Bounds_), value);
        auto index = std::distance(begin(Bounds_), it);
        Y_VERIFY_DEBUG((size_t)index < Size_);
        Values_[index].fetch_add(count, std::memory_order_relaxed);
    }

    IHistogramSnapshotPtr Snapshot() const override {
        TStackVec<ui64, HISTOGRAM_MAX_BUCKETS_COUNT> values;
        values.resize(Size_);
        for (auto i = 0u; i < Size_; ++i) {
            values[i] = Values_[i].load(std::memory_order_relaxed);
        }

        return ExplicitHistogramSnapshot(TArrayRef{Bounds_.begin(), Size_}, values);
    }

    void Reset() override {
        for (auto i = 0u; i < Values_.size(); ++i) {
            Values_[i].fetch_and(0);
        }
    }

private:
    std::array<std::atomic_uint64_t, HISTOGRAM_MAX_BUCKETS_COUNT> Values_;
    size_t Size_;
    THistBounds Bounds_;
};

IHistogramCollectorPtr Clone(const IHistogramSnapshotPtr& snap, TMemoryPool& arena) {
    THistBounds bounds;
    for (auto i = 0u; i < snap->Count(); ++i) {
        bounds[i] = snap->UpperBound(i);
    }

    return THolder(arena.New<THistogramCollector>(MakeArrayRef(bounds.begin(), snap->Count())));
}

IMetric* CloneMetric(const IMetric& m, TMemoryPool& arena) {
    const auto type = m.Type();
    switch (type) {
        case EMetricType::GAUGE:
            return arena.New<TGauge>();
        case EMetricType::IGAUGE:
            return arena.New<TIntGauge>();
        case EMetricType::RATE:
            return arena.New<TRate>();
        case EMetricType::COUNTER:
            return arena.New<TCounter>();
        case EMetricType::HIST_RATE:
        case EMetricType::HIST: {
            auto c = static_cast<const THistogram&>(m).TakeSnapshot();
            return arena.New<THistogram>(Clone(c, arena), m.Type() == EMetricType::HIST_RATE ? true : false);
        }

        default:
            Y_VERIFY_DEBUG(false, "Unsupported metric type: %s", TString{MetricTypeToStr(type)}.c_str());
    };

    return nullptr;
}

template <typename T>
void ResetValue(IMetric& m) {
    static_cast<T&>(m).Reset();
}

template <typename T>
void AddValue(IMetric& to, const IMetric& from) {
    static_cast<T&>(to).Add(static_cast<const T&>(from).Get());
}

template <>
void AddValue<IHistogram>(IMetric& to, const IMetric& from) {
    auto& lhs = static_cast<IHistogram&>(to);
    auto& rhs = static_cast<const IHistogram&>(from);

    auto snapshot = rhs.TakeSnapshot();
    for (auto i = 0u; i < snapshot->Count() - 1; ++i) {
        lhs.Record(snapshot->UpperBound(i), snapshot->Value(i));
    }

    lhs.Record(Max<i64>(), snapshot->Value(snapshot->Count() - 1));
}

void Combine(IMetric& to, const IMetric& from) {
    Y_VERIFY_DEBUG(to.Type() == from.Type());

    auto type = to.Type();
    switch (type) {
        case EMetricType::GAUGE:
            AddValue<IGauge>(to, from);
            break;
        case EMetricType::IGAUGE:
            AddValue<IIntGauge>(to, from);
            break;
        case EMetricType::RATE:
            AddValue<IRate>(to, from);
            break;
        case EMetricType::COUNTER:
            AddValue<ICounter>(to, from);
            break;
        case EMetricType::HIST_RATE:
        case EMetricType::HIST:
            AddValue<IHistogram>(to, from);
            break;

        default:
            Y_VERIFY_DEBUG(false, "Unsupported metric type: %s", TString{MetricTypeToStr(type)}.c_str());
    };
}
} // namespace

TImmutableMetricRegistry::TImmutableMetricRegistry(TMetricMap&& metrics, THolder<TMemoryPool>&& pool, TCollectorHolder&& collectors)
    : Metrics_{std::move(metrics)}
    , MetricArena_{std::move(pool)}
    , Collectors_{std::move(collectors)}
{
}

TImmutableMetricRegistry::~TImmutableMetricRegistry() {
}

void TImmutableMetricRegistry::Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const {
    consumer->OnStreamBegin();
    Append(time, consumer);
    consumer->OnStreamEnd();
}

void TImmutableMetricRegistry::Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const {
    for (const auto& it: Metrics_) {
        IMetric* metric = it.second;
        consumer->OnMetricBegin(metric->Type());
        WriteLabels(consumer, *it.first);
        metric->Accept(time, consumer);
        consumer->OnMetricEnd();
    }
}

void TImmutableMetricRegistry::Swap(TImmutableMetricRegistry&& other) {
    std::swap(Metrics_, other.Metrics_);
    std::swap(Collectors_, other.Collectors_);
    std::swap(MetricArena_, other.MetricArena_);
}

TImmutableMetricRegistry TImmutableMetricRegistry::Clone() const {
    TMetricMap clone;
    auto arena = MakeHolder<TMemoryPool>(MetricArena_->MemoryAllocated(), TMemoryPool::TLinearGrow::Instance());
    clone.reserve(Metrics_.size());

    for (auto&& m: Metrics_) {
        clone.emplace_back(m.first, CloneMetric(*m.second, *arena));
    }

    return {std::move(clone), std::move(arena)};
}

// XXX pass explicit label by which we are aggregating values?
void TImmutableMetricRegistry::CombineValues(const TImmutableMetricRegistry& other) {
    for (auto i = 0u; i < Metrics_.size(); ++i) {
        Combine(*Metrics_[i].second, *other.Metrics_[i].second);
    }
}

void TImmutableMetricRegistry::ResetValues() {
    for (auto& [labels, metric]: Metrics_) {
        auto type = metric->Type();
        switch (type) {
            case EMetricType::GAUGE:
                ResetValue<IGauge>(*metric);
                break;
            case EMetricType::IGAUGE:
                ResetValue<IIntGauge>(*metric);
                break;
            case EMetricType::RATE:
                ResetValue<IRate>(*metric);
                break;
            case EMetricType::COUNTER:
                ResetValue<ICounter>(*metric);
                break;
            case EMetricType::HIST_RATE:
            case EMetricType::HIST:
                ResetValue<IHistogram>(*metric);
                break;

            default:
                Y_VERIFY_DEBUG(false, "Unsupported metric type: %s", TString{MetricTypeToStr(type)}.c_str());
        }
    }
}

TImmutableMetricRegistry TImmutableMetricRegistryBuilder::Build() {
    TVector<std::pair<ISharedLabelsPtr, IMetric*>> items;
    items.reserve(Items_.size());

    for (auto it = Items_.begin(); it != Items_.end();) {
        auto prev = it;
        ++it;
        auto node = Items_.extract(prev);
        items.emplace_back(std::move(node.key()), std::move(node.mapped()));
    }

    return TImmutableMetricRegistry{std::move(items), std::move(MetricArena_), std::move(Collectors_)};
}

template <typename T, typename TRet, typename... TArgs>
TRet* TImmutableMetricRegistryBuilder::Metric(const ILabelsPtr& labels, TArgs&&... args) {
    IMetric* metric = MetricArena_->New<T>(std::forward<TArgs>(args)...);
    auto [it, uniq] = Items_.emplace(labels.Release(), std::move(metric));
    Y_ENSURE(uniq, "Attempting to insert a duplicate metric with labels " << *it->first);
    return static_cast<TRet*>(it->second);
}

IGauge* TImmutableMetricRegistryBuilder::Gauge(ILabelsPtr labels) {
    return Metric<TGauge, IGauge>(std::move(labels));
}

ILazyGauge* TImmutableMetricRegistryBuilder::LazyGauge(ILabelsPtr labels, std::function<double()> supplier) {
    return Metric<TLazyGauge, ILazyGauge>(std::move(labels), std::move(supplier));
}

IIntGauge* TImmutableMetricRegistryBuilder::IntGauge(ILabelsPtr labels) {
    return Metric<TIntGauge, IIntGauge>(std::move(labels));
}

ILazyIntGauge* TImmutableMetricRegistryBuilder::LazyIntGauge(ILabelsPtr labels, std::function<i64()> supplier) {
    return Metric<TLazyIntGauge, ILazyIntGauge>(std::move(labels), std::move(supplier));
}

ICounter* TImmutableMetricRegistryBuilder::Counter(ILabelsPtr labels) {
    return Metric<TCounter, ICounter>(std::move(labels));
}

ILazyCounter* TImmutableMetricRegistryBuilder::LazyCounter(ILabelsPtr labels, std::function<ui64()> supplier) {
    return Metric<TLazyCounter, ILazyCounter>(std::move(labels), std::move(supplier));
}

IRate* TImmutableMetricRegistryBuilder::Rate(ILabelsPtr labels) {
    return Metric<TRate, IRate>(std::move(labels));
}

ILazyRate* TImmutableMetricRegistryBuilder::LazyRate(ILabelsPtr labels, std::function<ui64()> supplier) {
    return Metric<TLazyRate, ILazyRate>(std::move(labels), std::move(supplier));
}

IHistogram* TImmutableMetricRegistryBuilder::HistogramCounter(ILabelsPtr labels, IHistogramCollectorPtr collector) {
    Collectors_.Collectors.push_back(collector.Get());
    return Metric<THistogram, IHistogram>(std::move(labels), std::move(collector), false);
}

IHistogram* TImmutableMetricRegistryBuilder::HistogramRate(ILabelsPtr labels, IHistogramCollectorPtr collector) {
    Collectors_.Collectors.push_back(collector.Get());
    return Metric<THistogram, IHistogram>(std::move(labels), std::move(collector), true);
}

IHistogram* TImmutableMetricRegistryBuilder::HistogramCounter(ILabelsPtr labels, std::function<IHistogramCollectorPtr()> collectorSupplier) {
    return Metric<THistogram, IHistogram>(std::move(labels), [this, collectorSupplier{std::move(collectorSupplier)}]() {
        auto collector = collectorSupplier();
        Collectors_.Collectors.push_back(collector.Get());
        return std::move(collector);
    }, false);
}

IHistogram* TImmutableMetricRegistryBuilder::HistogramRate(ILabelsPtr labels, std::function<IHistogramCollectorPtr()> collectorSupplier) {
    return Metric<THistogram, IHistogram>(std::move(labels), [this, collectorSupplier{std::move(collectorSupplier)}]() {
        auto collector = collectorSupplier();
        Collectors_.Collectors.push_back(collector.Get());
        return std::move(collector);
    }, true);
}

bool TImmutableMetricRegistry::IsEmpty() const {
    return MetricArena_ == nullptr;
}

TImmutableMetricRegistryBuilder::~TImmutableMetricRegistryBuilder() {
}

TCollectorHolder::~TCollectorHolder() {
    for (auto* c: Collectors) {
        delete c;
    }
}
} // namespace NSolomon

template <>
void Out<NSolomon::TImmutableMetricRegistry>(IOutputStream& os, const NSolomon::TImmutableMetricRegistry& r) {
    auto e = EncoderText(&os);
    r.Accept(TInstant::Zero(), e.Get());
}
