#pragma once

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/map.h>
#include <util/memory/pool.h>

namespace NSolomon {

using ISharedLabelsPtr = std::shared_ptr<NMonitoring::ILabels>;

struct TCollectorHolder: TMoveOnly {
    TCollectorHolder() = default;
    ~TCollectorHolder();
    TCollectorHolder(TCollectorHolder&& other) = default;
    TCollectorHolder& operator=(TCollectorHolder&& other) = default;

    TVector<NMonitoring::IHistogramCollector*> Collectors;
};

class TImmutableMetricRegistry: public NMonitoring::IMetricSupplier {
    friend class TImmutableMetricRegistryBuilder;

    using TMetricMap = TVector<std::pair<ISharedLabelsPtr, NMonitoring::IMetric*>>;
    TImmutableMetricRegistry(TMetricMap&& metrics, THolder<TMemoryPool>&& pool, TCollectorHolder&& collectors = {});

public:
    TImmutableMetricRegistry() = default;
    ~TImmutableMetricRegistry();

    TImmutableMetricRegistry(TImmutableMetricRegistry&& other) = default;
    TImmutableMetricRegistry& operator=(TImmutableMetricRegistry&& other) = default;

    bool IsEmpty() const;

    void Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const override;
    void Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const override;

    void Swap(TImmutableMetricRegistry&& other);

    void CombineValues(const TImmutableMetricRegistry& other);
    void ResetValues();
    TImmutableMetricRegistry Clone() const;

private:
    TMetricMap Metrics_;
    THolder<TMemoryPool> MetricArena_;
    TCollectorHolder Collectors_;
};

class TImmutableMetricRegistryBuilder: public NMonitoring::IMetricFactory {
public:
    ~TImmutableMetricRegistryBuilder();
    NMonitoring::IGauge* Gauge(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::ILazyGauge* LazyGauge(NMonitoring::ILabelsPtr labels, std::function<double()> supplier) override;
    NMonitoring::IIntGauge* IntGauge(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::ILazyIntGauge* LazyIntGauge(NMonitoring::ILabelsPtr labels, std::function<i64()> supplier) override;
    NMonitoring::ICounter* Counter(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::ILazyCounter* LazyCounter(NMonitoring::ILabelsPtr labels, std::function<ui64()> supplier) override;

    NMonitoring::IRate* Rate(NMonitoring::ILabelsPtr labels) override;
    NMonitoring::ILazyRate* LazyRate(NMonitoring::ILabelsPtr labels, std::function<ui64()> supplier) override;


    NMonitoring::IHistogram* HistogramCounter(
            NMonitoring::ILabelsPtr labels,
            NMonitoring::IHistogramCollectorPtr collector) override;

    NMonitoring::IHistogram* HistogramRate(
            NMonitoring::ILabelsPtr labels,
            NMonitoring::IHistogramCollectorPtr collector) override;

    NMonitoring::IHistogram* HistogramCounter(
            NMonitoring::ILabelsPtr labels,
            std::function<NMonitoring::IHistogramCollectorPtr()> collector) override;

    NMonitoring::IHistogram* HistogramRate(
            NMonitoring::ILabelsPtr labels,
            std::function<NMonitoring::IHistogramCollectorPtr()> collector) override;

    TImmutableMetricRegistry Build();

private:
    struct TPointeeLess {
        bool operator()(const ISharedLabelsPtr& lhs, const ISharedLabelsPtr& rhs) const {
            if (lhs->Size() != rhs->Size()) {
                return lhs->Size() < rhs->Size();
            }

            for (auto i = 0u; i < lhs->Size(); ++i) {
                auto lhsName = lhs->Get(i)->Name();
                auto rhsName = rhs->Get(i)->Name();
                auto lhsValue = lhs->Get(i)->Value();
                auto rhsValue = rhs->Get(i)->Value();

                if (lhsName < rhsName || lhsValue < rhsValue) {
                    return true;
                }
            }

            return false;
        }
    };

    TMap<ISharedLabelsPtr, NMonitoring::IMetric*, TPointeeLess> Items_;
    THolder<TMemoryPool> MetricArena_{MakeHolder<TMemoryPool>(512, TMemoryPool::TLinearGrow::Instance())};
    // we need to keep track of collectors separatly since they are allocated not on our arena
    TCollectorHolder Collectors_;

private:
    template <typename T, typename TRet, typename... TArgs>
    TRet* Metric(const NMonitoring::ILabelsPtr& labels, TArgs&&... args);
};

} // namespace NSolomon
