#include "repository.h"

#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/libs/cpp/sync/rw_lock.h>

#include <util/random/random.h>
#include <util/generic/scope.h>

#include <memory>

using namespace NMonitoring;

namespace NSolomon {

class TRepositoryImpl;

namespace {
// this will allow us to use factory in try_emplace and thus avoid constructing objects
// in case they are already present in the map
template <class T>
class TFactoryWrapper {
public:
    TFactoryWrapper(T&& f) : F_{std::move(f)} {}
    operator std::invoke_result_t<T>() {
        return F_();
    }

private:
    T F_;
};

struct TShardMap {
    using TLockedResorcesMap = NSync::TLightRwLock<TStringMap<std::weak_ptr<TResourceUsageContextHolder>>>;

    template <typename... TArgs>
    TResourceUsageContextPtr GetOrCreate(TStringBuf key, TArgs&&... args) {
        {
            auto guard = Resources_.Read();
            if (auto it = guard->find(key); it != guard->end()) {
                if (auto resource = it->second.lock()) {
                    return resource;
                }
            }
        }

        auto guard = Resources_.Write();
        if (auto it = guard->find(key); it != guard->end()) {
            if (auto resource = it->second.lock()) {
                return resource;
            }

            auto resource = std::make_shared<TResourceUsageContextHolder>(std::forward<TArgs>(args)...);
            it->second = resource;
            return resource;
        }

        auto resource = std::make_shared<TResourceUsageContextHolder>(std::forward<TArgs>(args)...);
        guard->emplace(key, resource);
        return resource;
    }

    TLockedResorcesMap::TRead Read() const {
        return Resources_.Read();
    }

    TLockedResorcesMap::TWrite Write() {
        return Resources_.Write();
    }

private:
    TLockedResorcesMap Resources_;
};

struct TProjectMap {
    struct TProjectContext {
        explicit TProjectContext(TImmutableMetricRegistry&& total) noexcept
            : Total{std::move(total)}
        {
        }

        bool Cleanup() {
            auto shardMap = ShardMap.Write();
            absl::erase_if(*shardMap, [](const auto& it) {
                return it.second.expired();
            });
            return shardMap->empty();
        }

        TImmutableMetricRegistry Total;
        TShardMap ShardMap;
    };

    using TLockedProjectMap = NSync::TLightRwLock<TStringMap<std::shared_ptr<TProjectContext>>>;

    template <typename... TArgs>
    std::shared_ptr<TProjectContext> GetOrCreate(TStringBuf key, TArgs&&... args) {
        {
            auto guard = Projects_.Read();
            if (auto it = guard->find(key); it != guard->end()) {
                return it->second;
            }
        }

        auto guard = Projects_.Write();
        if (auto it = guard->find(key); it != guard->end()) {
            return it->second;
        }

        auto project = std::make_shared<TProjectContext>(std::forward<TArgs>(args)...);
        guard->emplace(key, project);
        return project;
    }

    TLockedProjectMap::TRead Read() const {
        return Projects_.Read();
    }

    TLockedProjectMap::TWrite Write() {
        return Projects_.Write();
    }

private:
    TLockedProjectMap Projects_;
};

class TRepositoryConsumer: public IRepositoryVisitor {
public:
    TRepositoryConsumer(TInstant time, IMetricConsumer* consumer)
        : Time_{time}
        , Consumer_{consumer}
    {
    }

private:
    void OnShard(TResourceUsageContextHolder& shardCtx) override {
        shardCtx.Append(Time_, Consumer_);
    }

    void OnProject(TImmutableMetricRegistry& projectTotals) override {
        projectTotals.Append(Time_, Consumer_);
    }

    void OnTotal(TImmutableMetricRegistry& totals) override {
        totals.Append(Time_, Consumer_);
    }

private:
    TInstant Time_;
    IMetricConsumer* Consumer_;
};

} // namespace

class TRepositoryImpl: public IRepositoryImpl {
public:
    explicit TRepositoryImpl(TFactoryFunc&& f)
        : Factory_{std::move(f)}
        , GrandTotal_{MakeRegistry("total", "total")}
    {
    }

    IResourceUsageContextPtr Emplace(TString project, TString shard) override;

    void Visit(IRepositoryVisitor& visitor) const override;
    void Accept(TInstant time, IMetricConsumer* consumer) const override;
    void Append(TInstant time, IMetricConsumer* consumer) const override;

    // this one is currently called from Append, so have to make it const although it is actually not
    void CollectGarbage() const;

private:
    TImmutableMetricRegistry MakeRegistry(TString project, TString shard) const {
        auto impl = Factory_(std::move(project), std::move(shard));
        auto&& r = const_cast<TImmutableMetricRegistry&>(impl->Registry());

        return std::move(r);
    }

private:
    mutable TProjectMap ContextTable_;
    TFactoryFunc Factory_;
    TImmutableMetricRegistry GrandTotal_;
    mutable std::atomic_flag IsGcRunning_{false};
};

IResourceUsageContextPtr TRepositoryImpl::Emplace(TString project, TString shard) {
    auto projectCtx = ContextTable_.GetOrCreate(
            project,
            TFactoryWrapper([&] { return MakeRegistry(project, "total"); }));

    auto shardCtx = projectCtx->ShardMap.GetOrCreate(
            shard,
            TFactoryWrapper([&] { return Factory_(project, shard); }),
            project,
            shard);

    return {shardCtx, shardCtx->Impl()};
}

void TRepositoryImpl::Visit(IRepositoryVisitor& visitor) const {
    auto grandTotal = GrandTotal_.Clone();

    if (auto projectMap = ContextTable_.Read()) {
        for (const auto& [projectId, projectCtx]: *projectMap) {
            auto projectTotal = projectCtx->Total.Clone();

            if (auto shardMap = projectCtx->ShardMap.Read()) {
                for (const auto& [shardId, shardCtx]: *shardMap) {
                    if (auto ctx = shardCtx.lock()) {
                        visitor.OnShard(*ctx);
                        projectTotal.CombineValues(ctx->Registry());
                    }
                }
            } // release shardMap lock ASAP

            grandTotal.CombineValues(projectTotal);
            visitor.OnProject(projectTotal);
        }
    }  // release projectMap lock ASAP

    visitor.OnTotal(grandTotal);

    // XXX allow passing an interface for async collection?
    if (RandomNumber<double>() < 0.05) {
        CollectGarbage();
    }
}

void TRepositoryImpl::Accept(TInstant time, IMetricConsumer* consumer) const {
    consumer->OnStreamBegin();
    Append(time, consumer);
    consumer->OnStreamEnd();
}

void TRepositoryImpl::Append(TInstant time, IMetricConsumer* consumer) const {
    TRepositoryConsumer repoConsumer{time, consumer};
    Visit(repoConsumer);
}

void TRepositoryImpl::CollectGarbage() const {
    if (IsGcRunning_.test_and_set(std::memory_order_relaxed)) {
        return;
    }

    Y_DEFER {
        IsGcRunning_.clear(std::memory_order_relaxed);
    };

    auto projectMap = ContextTable_.Write();
    absl::erase_if(*projectMap, [](auto& it) {
        return it.second->Cleanup();
    });
}

IRepositoryImplPtr DefaultRepositoryImpl(IRepositoryImpl::TFactoryFunc&& factory) {
    return MakeHolder<TRepositoryImpl>(std::move(factory));
}

} // namespace NSolomon
