#pragma once

#include "proxy.h"
#include "resource_context.h"

#include <util/generic/cast.h>

#include <memory>

namespace NSolomon {

class IRepository: public NMonitoring::IMetricSupplier {
public:
    virtual IResourceUsageContextPtr GetContext(TString project, TString shard) = 0;
};

class TResourceUsageContextHolder: public TResourceUsageContextProxy {
public:
    TResourceUsageContextHolder(IResourceUsageContextPtr impl, TStringBuf project, TStringBuf shard)
        : TResourceUsageContextProxy{std::move(impl)}
        , ProjectId{project}
        , ShardId{shard}
    {
    }

    IResourceUsageContext* Impl() {
        return Impl_.get();
    }

    const IResourceUsageContext* Impl() const {
        return Impl_.get();
    }

public:
// TODO: make a getter and a setter
    TStringBuf ProjectId;
    TStringBuf ShardId;
};

using TResourceUsageContextPtr = std::shared_ptr<TResourceUsageContextHolder>;

struct IRepositoryVisitor {
    virtual ~IRepositoryVisitor() = default;

    virtual void OnShard(TResourceUsageContextHolder& shardCtx) = 0;
    virtual void OnProject(TImmutableMetricRegistry& projectTotals) = 0;
    virtual void OnTotal(TImmutableMetricRegistry& totals) = 0;
};

class IRepositoryImpl: public NMonitoring::IMetricSupplier {
public:
    using TFactoryFunc = std::function<IResourceUsageContextPtr(TString, TString)>;
    virtual IResourceUsageContextPtr Emplace(TString project, TString shard) = 0;

    virtual void Visit(IRepositoryVisitor& visitor) const = 0;
};

using IRepositoryImplPtr = THolder<IRepositoryImpl>;
IRepositoryImplPtr DefaultRepositoryImpl(IRepositoryImpl::TFactoryFunc&& factory);


template <typename TContext>
class TRepository: public NMonitoring::IMetricSupplier {
    static_assert(std::is_base_of_v<IResourceUsageContext, TContext>);
    static_assert(std::is_constructible_v<TContext, TString, TString>);
public:
    using TCtx = TContext;
    using TCtxPtr = std::shared_ptr<TContext>;

private:
    auto MakeFactory() {
        return [this] (auto&& project, auto&& shard) {
            return DoCreate(std::move(project), std::move(shard));
        };
    }

public:
    TRepository()
        : Impl_{DefaultRepositoryImpl(MakeFactory())}
    {
    }

    TRepository(IRepositoryImplPtr impl)
        : Impl_{std::move(impl)}
    {
    }

    ~TRepository() = default;

    /**
     *
     * @param project label that will be propagated as projectId=project
     * @param shard label that will be propagated as shardId=shard
     * @param numId id of a shard, is not present in result data. Is available by directly accessing TResourceUsageContextHolder
     * @return
     */
    TCtxPtr GetContext(TString project, TString shard) const {
         auto p = Impl_->Emplace(std::move(project), std::move(shard));
         return std::static_pointer_cast<TContext>(p);
    }

    void Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const override {
        Impl_->Accept(time, consumer);
    }

    void Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const override {
        Impl_->Append(time, consumer);
    }

    void Visit(IRepositoryVisitor& visitor) const {
        Impl_->Visit(visitor);
    }

private:
    TCtxPtr DoCreate(TString project, TString shard) {
        auto p = std::make_shared<TContext>(std::move(project), std::move(shard));
        static_cast<IResourceUsageContext&>(*p).Init();
        return p;
    }

private:
    IRepositoryImplPtr Impl_;
};

struct TDefaultRepository: public TRepository<TResourceUsageContextBase> {
};

} // namespace NSolomon
