#include "resource_context.h"
#include "immutable_registry.h"
#include "proxy.h"
#include "repository.h"

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/datetime/cputimer.h>
#include <memory>

using namespace NMonitoring;

namespace NSolomon {
namespace {
    ILabelsPtr CreateLabels(TStringBuf projectId, TStringBuf shardId, TStringBuf name) {
        return MakeLabels({
            {"sensor", name},
            {"projectId", projectId},
            {"shardId", shardId},
        });
    }

    void AddLabels(ILabels& labels, TStringBuf projectId, TStringBuf shardId) {
        labels.Add("projectId", projectId);
        labels.Add("shardId", shardId);
    }

    struct TAddLabelsProxy: public IMetricFactory {
    public:
        TAddLabelsProxy(IMetricFactory& impl, TStringBuf projectId, TStringBuf shardId)
            : Impl_{impl}
            , ProjectId_{projectId}
            , ShardId_{shardId}
        {
        }

        IGauge* Gauge(ILabelsPtr labels) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.Gauge(std::move(labels));
        }

        ILazyGauge* LazyGauge(ILabelsPtr labels, std::function<double()> supplier) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.LazyGauge(std::move(labels), std::move(supplier));
        }

        IIntGauge* IntGauge(ILabelsPtr labels) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.IntGauge(std::move(labels));
        }

        ILazyIntGauge* LazyIntGauge(ILabelsPtr labels, std::function<i64()> supplier) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.LazyIntGauge(std::move(labels), std::move(supplier));
        }

        ICounter* Counter(ILabelsPtr labels) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.Counter(std::move(labels));
        }

        ILazyCounter* LazyCounter(ILabelsPtr labels, std::function<ui64()> supplier) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.LazyCounter(std::move(labels), std::move(supplier));
        }

        IRate* Rate(ILabelsPtr labels) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.Rate(std::move(labels));
        }

        ILazyRate* LazyRate(ILabelsPtr labels, std::function<ui64()> supplier) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.LazyRate(std::move(labels), std::move(supplier));
        }

        IHistogram* HistogramCounter(ILabelsPtr labels, IHistogramCollectorPtr collector) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.HistogramCounter(std::move(labels), std::move(collector));
        }

        IHistogram* HistogramRate(ILabelsPtr labels, IHistogramCollectorPtr collector) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.HistogramRate(std::move(labels), std::move(collector));
        }

        IHistogram* HistogramCounter(ILabelsPtr labels, std::function<IHistogramCollectorPtr()> collector) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.HistogramCounter(std::move(labels), std::move(collector));
        }

        IHistogram* HistogramRate(ILabelsPtr labels, std::function<IHistogramCollectorPtr()> collector) override {
            AddLabels(*labels, ProjectId_, ShardId_);
            return Impl_.HistogramRate(std::move(labels), std::move(collector));
        }

    private:
        IMetricFactory& Impl_;
        TStringBuf ProjectId_;
        TStringBuf ShardId_;
    };
} // namespace

ILabelsPtr TResourceUsageContextBlank::Labels(TStringBuf name) const {
    return CreateLabels(ProjectId_, ShardId_, name);
}

void TResourceUsageContextBlank::Accept(TInstant time, IMetricConsumer* consumer) const {
    Registry_.Accept(time, consumer);
}

void TResourceUsageContextBlank::Append(TInstant time, IMetricConsumer* consumer) const {
    Registry_.Append(time, consumer);
}

void TResourceUsageContextBase::SetMemoryBytes(ui64 val) {
    MemBytes_->Set(val);
}

void TResourceUsageContextBase::AddNetworkRxBytes(ui64 val) {
    NetworkRxBytes_->Add(val);
}

void TResourceUsageContextBase::AddNetworkTxBytes(ui64 val) {
    NetworkTxBytes_->Add(val);
}

void TResourceUsageContextBase::AddCpuTime(TDuration cpuTime) {
    CpuUTimeMillis_->Add(cpuTime.MilliSeconds());
}

void TResourceUsageContextBlank::Init() {
    TImmutableMetricRegistryBuilder b;
    TAddLabelsProxy proxy{b, ProjectId_, ShardId_};

    Init(proxy);

    Registry_.Swap(b.Build());
}

void TResourceUsageContextBase::Init() {
    TImmutableMetricRegistryBuilder b;
    MemBytes_ = b.IntGauge(Labels("memBytes"));
    NetworkTxBytes_ = b.Rate(Labels("network.rxBytes"));
    NetworkRxBytes_ = b.Rate(Labels("network.txBytes"));
    CpuUTimeMillis_ = b.Rate(Labels("cpu.userTimeMillis"));

    TAddLabelsProxy proxy{b, ProjectId_, ShardId_};
    Init(proxy);
    Registry_.Swap(b.Build());
}

IResourceUsageContext::TCpuUsageContext IResourceUsageContext::GetScopeTimer() {
    return shared_from_this();
}
} // namespace NSolomon
