#pragma once

#include "immutable_registry.h"

#include <util/generic/ptr.h>
#include <util/datetime/cputimer.h>

#include <utility>

namespace NSolomon {

class IResourceUsageContext:
    public NMonitoring::IMetricSupplier,
    public std::enable_shared_from_this<IResourceUsageContext>
{
public:
    class TCpuUsageContext;

    virtual ~IResourceUsageContext() = default;
    virtual void Init() { }
    virtual void SetMemoryBytes(ui64 val) = 0;

    virtual void AddNetworkRxBytes(ui64 val) = 0;
    virtual void AddNetworkTxBytes(ui64 val) = 0;

    virtual void AddCpuTime(TDuration cpuTime) = 0;

    TCpuUsageContext GetScopeTimer();

    // XXX
    virtual const TImmutableMetricRegistry& Registry() const = 0;
};

using IResourceUsageContextPtr = std::shared_ptr<IResourceUsageContext>;

class IResourceUsageContext::TCpuUsageContext: TMoveOnly {
public:
    TCpuUsageContext(IResourceUsageContextPtr parent)
        : Parent_{std::move(parent)}
    {
    }

    ~TCpuUsageContext() {
        Parent_->AddCpuTime(Impl_.Get());
    }

private:
    TProfileTimer Impl_;
    IResourceUsageContextPtr Parent_;
};


/**
 * A "blank" resource context -- with no predefined metrics
 */
class TResourceUsageContextBlank: public IResourceUsageContext {
public:
    TResourceUsageContextBlank(TString project, TString shard) noexcept
        : ProjectId_{std::move(project)}
        , ShardId_{std::move(shard)}
    {
    }

    void Init() override;

    void Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const override final;
    void Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const override;

    const TImmutableMetricRegistry& Registry() const override final {
        return Registry_;
    }

protected:
    virtual void Init(NMonitoring::IMetricFactory&) = 0;

    TStringBuf ShardId() const {
        return ShardId_;
    }

    TStringBuf ProjectId() const {
        return ProjectId_;
    }

    NMonitoring::ILabelsPtr Labels(TStringBuf name) const;

// for testing purposes
protected:
    TImmutableMetricRegistry Registry_;

    TString ProjectId_;
    TString ShardId_;
};

/**
 * A resource context with predefined metrics for cpu, mem, net
 * and methods setting values for those
 */
class TResourceUsageContextBase: public TResourceUsageContextBlank {
public:
    using TBase = TResourceUsageContextBlank;
    using TBase::TBase;

    void Init() override final;
    void SetMemoryBytes(ui64 val) override final;

    void AddNetworkRxBytes(ui64 val) override final;
    void AddNetworkTxBytes(ui64 val) override final;

    void AddCpuTime(TDuration cpuTime) override final;

private:
    void Init(NMonitoring::IMetricFactory&) override {
    }

// for testing purposes
protected:
    NMonitoring::IIntGauge* MemBytes_{nullptr};
    NMonitoring::IRate* CpuUTimeMillis_{nullptr};
    NMonitoring::IRate* NetworkRxBytes_{nullptr};
    NMonitoring::IRate* NetworkTxBytes_{nullptr};
};
} // namespace NSolomon
