#include "helpers.h"

#include <solomon/libs/cpp/shard_metrics/immutable_registry.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NMonitoring;
using namespace NSolomon;
using namespace testing;

namespace {

TLabels FIRST_LABELS{{{"foo", "bar"}}};
TLabels SECOND_LABELS{{{"foo", "baz"}}};

ILabelsPtr FirstLabels() {
    return MakeIntrusive<TLabels>(FIRST_LABELS);
}

ILabelsPtr SecondLabels() {
    return MakeIntrusive<TLabels>(SECOND_LABELS);
}

const TVector<TMetricData> EXPECTED = [] {
    TVector<TMetricData> expected;
    expected.emplace_back(FIRST_LABELS, EMetricType::RATE, SinglePointSeries<ui64>(3));
    expected.emplace_back(SECOND_LABELS, EMetricType::GAUGE, SinglePointSeries<double>(12));
    return expected;
}();

} // namespace

class TImmutableMetricRegistryTest: public testing::Test {
protected:
    void SetUp() override {
        TVector<std::pair<ISharedLabelsPtr, IMetricPtr>> v;
        TImmutableMetricRegistryBuilder b;
        b.Rate(FirstLabels())->Add(3);
        b.Gauge(SecondLabels())->Set(12);

        Registry_ = MakeHolder<TImmutableMetricRegistry>(b.Build());
        Consumer_ = {};
    }

    void TearDown() override {
        Registry_.Reset();
        Consumer_ = {};
    }

    TCollectingConsumer Consumer_;
    THolder<TImmutableMetricRegistry> Registry_;
};

TEST_F(TImmutableMetricRegistryTest, EmptyRegistry) {
    TImmutableMetricRegistry r;
    r.Accept(TInstant::Zero(), &Consumer_);
    ASSERT_THAT(Consumer_.Metrics, IsEmpty());

    r.Append(TInstant::Zero(), &Consumer_);
    ASSERT_THAT(Consumer_.Metrics, IsEmpty());
}

TEST_F(TImmutableMetricRegistryTest, AcceptReturnsAllMetrics) {
    Registry_->Accept(TInstant::Zero(), &Consumer_);
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(EXPECTED));
}

TEST_F(TImmutableMetricRegistryTest, AppendReturnsAllMetrics) {
    auto ts = TInstant::Minutes(1);
    Consumer_.OnStreamBegin();
    Registry_->Append(ts, &Consumer_);
    Consumer_.OnStreamEnd();

    TVector<TMetricData> expected;
    expected.emplace_back(FIRST_LABELS, EMetricType::RATE, SinglePointSeries<ui64>(3, ts));
    expected.emplace_back(SECOND_LABELS, EMetricType::GAUGE, SinglePointSeries<double>(12, ts));
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));
}

TEST_F(TImmutableMetricRegistryTest, PointersAreUsable) {
    TImmutableMetricRegistryBuilder b;
    auto* rate = b.Rate(FirstLabels());
    auto* gauge = b.Gauge(SecondLabels());
    rate->Add(3);
    gauge->Set(12);

    Registry_ = MakeHolder<TImmutableMetricRegistry>(b.Build());

    rate->Add(3);
    gauge->Add(30);

    Registry_->Accept(TInstant::Zero(), &Consumer_);

    TVector<TMetricData> expected;
    expected.emplace_back(FIRST_LABELS, EMetricType::RATE, SinglePointSeries<ui64>(6));
    expected.emplace_back(SECOND_LABELS, EMetricType::GAUGE, SinglePointSeries<double>(42));
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));
}

TEST_F(TImmutableMetricRegistryTest, CombineWorksForPlainTypes) {
    TLabels rateLabels{{"metric", "rate"}};
    TLabels gaugeLabels{{"metric", "gauge"}};
    TLabels intGaugeLabels{{"metric", "intGauge"}};
    TLabels counterLabels{{"metric", "counter"}};

    TImmutableMetricRegistryBuilder b;
    b.Rate(Clone(rateLabels))->Add(10);
    b.Counter(Clone(counterLabels))->Add(20);
    b.Gauge(Clone(gaugeLabels))->Add(30);
    b.IntGauge(Clone(intGaugeLabels))->Add(40);
    auto to = b.Build();

    TImmutableMetricRegistryBuilder b2;
    b2.Rate(Clone(rateLabels))->Add(10);
    b2.Counter(Clone(counterLabels))->Add(20);
    b2.Gauge(Clone(gaugeLabels))->Add(30);
    b2.IntGauge(Clone(intGaugeLabels))->Add(40);
    auto from = b2.Build();

    to.CombineValues(from);
    to.Accept(TInstant::Zero(), &Consumer_);

    TVector<TMetricData> expected;
    expected.emplace_back(rateLabels, EMetricType::RATE, SinglePointSeries<ui64>(20));
    expected.emplace_back(counterLabels, EMetricType::COUNTER, SinglePointSeries<ui64>(40));
    expected.emplace_back(gaugeLabels, EMetricType::GAUGE, SinglePointSeries<double>(60));
    expected.emplace_back(intGaugeLabels, EMetricType::IGAUGE, SinglePointSeries<i64>(80));

    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));
}

TEST_F(TImmutableMetricRegistryTest, CombineWorksForHistogram) {
    TLabels rateLabels{{"metric", "rate"}};
    TLabels counterLabels{{"metric", "counter"}};

    TImmutableMetricRegistryBuilder b;
    auto* rateHist = b.HistogramRate(Clone(rateLabels), ExplicitHistogram({1, 10, 100}));
    auto* counterHist = b.HistogramCounter(Clone(counterLabels), ExplicitHistogram({1, 10, 100}));
    auto to = b.Build();

    rateHist->Record(1, 5);
    rateHist->Record(0, 3);
    rateHist->Record(101, 4);
    counterHist->Record(1, 5);
    counterHist->Record(0, 3);
    counterHist->Record(101, 4);

    TImmutableMetricRegistryBuilder b2;
    rateHist = b2.HistogramRate(Clone(rateLabels), ExplicitHistogram({1, 10, 100}));
    counterHist = b2.HistogramCounter(Clone(counterLabels), ExplicitHistogram({1, 10, 100}));
    auto from = b2.Build();

    rateHist->Record(1);
    rateHist->Record(0);
    rateHist->Record(101);
    counterHist->Record(1);
    counterHist->Record(0);
    counterHist->Record(101);

    to.CombineValues(from);
    to.Accept(TInstant::Zero(), &Consumer_);

    TVector<TMetricData> expected;

    auto expectedSnapshot = MakeSnapshot({{1, 10}, {10, 0}, {100, 0}, {HISTOGRAM_INF_BOUND, 5}});
    auto s1 = SinglePointSeries(expectedSnapshot.Get());
    auto s2 = SinglePointSeries(expectedSnapshot.Get());

    expected.emplace_back(rateLabels, EMetricType::HIST_RATE, std::move(s1));
    expected.emplace_back(counterLabels, EMetricType::HIST, std::move(s2));
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));
}

TEST_F(TImmutableMetricRegistryTest, ValuesAreReset) {
    Registry_->ResetValues();
    Registry_->Accept(TInstant::Zero(), &Consumer_);

    ASSERT_THAT(Consumer_.Metrics, SizeIs(2));

    TVector<TMetricData> expected;
    expected.emplace_back(FIRST_LABELS, EMetricType::RATE, SinglePointSeries<ui64>(0));
    expected.emplace_back(SECOND_LABELS, EMetricType::GAUGE, SinglePointSeries<double>(0.));
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));
}

TEST_F(TImmutableMetricRegistryTest, BuilderThrowsOnDuplicateMetric) {
    TLabels rateLabels{{"metric", "rate"}};
    TLabels counterLabels{{"metric", "counter"}};

    TImmutableMetricRegistryBuilder b;
    b.HistogramRate(Clone(rateLabels), ExplicitHistogram({1, 10, 100}));
    b.HistogramCounter(Clone(counterLabels), ExplicitHistogram({1, 10, 100}));
    ASSERT_THROW(b.HistogramRate(Clone(rateLabels), ExplicitHistogram({1, 10, 100})), yexception);
}

TEST_F(TImmutableMetricRegistryTest, CloneWorks) {
    TLabels histRate{{"metric", "histRate"}};
    TImmutableMetricRegistryBuilder b;
    b.Rate(FirstLabels())->Add(3);
    b.Gauge(SecondLabels())->Set(12);
    auto* hist = b.HistogramRate(Clone(histRate), ExplicitHistogram({1, 10, 100}));
    hist->Record(101, 4);
    hist->Record(0, 5);

    Registry_ = MakeHolder<TImmutableMetricRegistry>(b.Build());
    auto expectedSnapshot = MakeSnapshot({{1, 5}, {10, 0}, {100, 0}, {HISTOGRAM_INF_BOUND, 4}});

    TVector<TMetricData> expected;
    expected.emplace_back(FIRST_LABELS, EMetricType::RATE, SinglePointSeries<ui64>(3));
    expected.emplace_back(SECOND_LABELS, EMetricType::GAUGE, SinglePointSeries<double>(12));

    {
        auto s = SinglePointSeries(expectedSnapshot.Get());
        expected.emplace_back(histRate, EMetricType::HIST_RATE, std::move(s));
    }

    auto newRegistry = Registry_->Clone();

    // old one is still holds original values
    Registry_->Accept(TInstant::Zero(), &Consumer_);
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));

    Consumer_ = {};

    expectedSnapshot = MakeSnapshot({{1, 0}, {10, 0}, {100, 0}, {HISTOGRAM_INF_BOUND, 0}});
    // new one has metrics with same labels, all reset to 0
    expected.clear();
    expected.emplace_back(FIRST_LABELS, EMetricType::RATE, SinglePointSeries<ui64>(0));
    expected.emplace_back(SECOND_LABELS, EMetricType::GAUGE, SinglePointSeries<double>(0));

    {
        auto s = SinglePointSeries(expectedSnapshot.Get());
        expected.emplace_back(histRate, EMetricType::HIST_RATE, std::move(s));
    }

    newRegistry.Accept(TInstant::Zero(), &Consumer_);
    ASSERT_THAT(Consumer_.Metrics, MetricsEq(expected));
}
