#include "helpers.h"

#include <solomon/libs/cpp/shard_metrics/resource_context.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/openssl/crypto/sha.h>
#include <library/cpp/testing/benchmark/bench.h>

#include <util/random/random.h>
#include <util/generic/size_literals.h>

#include <memory>

using namespace NMonitoring;
using namespace NSolomon;

class TTestUsageContext: public TResourceUsageContextBase {
public:
    using TResourceUsageContextBase::TResourceUsageContextBase;

    ui64 MemBytes() const {
        return MemBytes_->Get();
    }

    ui64 NetworkTxBytes() const {
        return NetworkTxBytes_->Get();
    }

    ui64 NetworkRxBytes() const {
        return NetworkRxBytes_->Get();
    }

    ui64 CpuTimeMillis() const {
        return CpuUTimeMillis_->Get();
    }
};

class TResourceUsageContextTest: public ::testing::Test {
public:
    void SetUp() override {
        Ctx_ = std::make_shared<TTestUsageContext>("foo", "bar");
        Ctx_->Init();
        Consumer_ = {};
    }

protected:
    std::shared_ptr<TTestUsageContext> Ctx_;
    TCollectingConsumer Consumer_;
};

TEST_F(TResourceUsageContextTest, CountersAreUsable) {
    Ctx_->SetMemoryBytes(10);
    Ctx_->AddNetworkRxBytes(11);
    Ctx_->AddNetworkTxBytes(12);
    Ctx_->AddCpuTime(TDuration::MilliSeconds(14));
    ASSERT_GE(Ctx_->MemBytes(), 0u);
    ASSERT_EQ(Ctx_->NetworkRxBytes(), 11u);
    ASSERT_EQ(Ctx_->NetworkTxBytes(), 12u);
    ASSERT_EQ(Ctx_->CpuTimeMillis() ,14u);
}

TEST_F(TResourceUsageContextTest, CpuTimerWorks) {
    {
        auto t = Ctx_->GetScopeTimer();

        NOpenSsl::NSha256::TDigest dummy;

        ui8 (*r)() = RandomNumber<ui8>;

        TVector<ui8> data;
        data.resize(10_MB);
        GenerateN(data.begin(), data.size(), r);

        for (auto i = 0; i < 10; ++i) {
            dummy = NOpenSsl::NSha256::Calc(
                    data.data(),
                    data.size() * sizeof(decltype(data)::value_type)
            );
        }

        Y_DO_NOT_OPTIMIZE_AWAY(dummy);
    }


    ASSERT_GT(Ctx_->CpuTimeMillis(), 0u);
}

TEST_F(TResourceUsageContextTest, AcceptWritesToConsumer) {
    Ctx_->Accept(TInstant::Minutes(1), &Consumer_);
    ASSERT_EQ(Consumer_.Metrics.size(), 4u);
}

TEST_F(TResourceUsageContextTest, AppendWritesToConsumer) {
    Consumer_.OnStreamBegin();
    Ctx_->Append(TInstant::Minutes(1), &Consumer_);
    Consumer_.OnStreamEnd();
    ASSERT_EQ(Consumer_.Metrics.size(), 4u);
}
