#include "signals.h"

#include <util/stream/output.h>
#include <util/system/getpid.h>
#include <util/system/backtrace.h>
#include <util/generic/yexception.h>

#ifdef _win_
#include <util/string/cast.h>
#include <util/system/winint.h>
#include <io.h>
#endif

namespace NSolomon {

volatile sig_atomic_t NeedTerminate = 0;
volatile sig_atomic_t NeedReopenLog = 0;

namespace {

#ifdef _unix_
#define ERR_FILENO STDERR_FILENO
#else
#define ERR_FILENO _fileno( stderr )
#endif

void SignalHandler(int signo) {
    switch (signo) {
    case SIGTERM:
        NeedTerminate = 1;
        break;
#ifdef _unix_
    case SIGUSR1:
    case SIGUSR2:
        NeedReopenLog = 1;
        break;
#endif
    default:
        break;
    }
}

class TEmergencyOutput: public IOutputStream {
public:
    TEmergencyOutput()
        : Current_(Buf_)
        , End_(Y_ARRAY_END(Buf_))
    {
    }

private:
    inline size_t Avail() const noexcept {
        return End_ - Current_;
    }

    void DoFlush() override {
        if (Current_ != Buf_) {
            write(ERR_FILENO, Buf_, Current_ - Buf_);
            Current_ = Buf_;
        }
    }

    void DoWrite(const void* buf, size_t len) override {
        size_t writeLen = Min(len, Avail());
        if (writeLen > 0) {
            char* end = Current_ + writeLen;
            memcpy(Current_, buf, writeLen);
            Current_ = end;
        }
    }

private:
    char Buf_[1 << 20];
    char* Current_;
    char* const End_;

};

TEmergencyOutput EMERGENCY_OUTPUT;

void LogBacktraceOnSigsegv(int signo) {
    // default handler must be restored first
    signal(signo, SIG_DFL);

    EMERGENCY_OUTPUT << TStringBuf("SIGSEGV (pid=") << GetPID() << TStringBuf("): ");
    FormatBackTrace(&EMERGENCY_OUTPUT);
    EMERGENCY_OUTPUT.Flush();

    // Now reraise the signal. We reactivate the signas default handling,
    // which is to terminate the process. We could just call exit or abort,
    // but reraising the signal sets the return status from the process
    // correctly.
    raise(signo);
}

void GracefulExitOnSigint(int signo) {
    // default handler must be restored first
    signal(signo, SIG_DFL);

    EMERGENCY_OUTPUT << TStringBuf("SIGINT (pid=") << GetPID() << TStringBuf(")\n");
    EMERGENCY_OUTPUT << TStringBuf("Performing graceful shutdown; press ^C again to terminate immediately.\n");
    EMERGENCY_OUTPUT.Flush();

    NeedTerminate = 1;
}

int SetSignalHandler(int signo, void (*handler)(int)) {
#ifdef _unix_
    struct sigaction sa;
    sa.sa_flags = SA_RESTART;
    sa.sa_handler = handler;
    sigemptyset(&sa.sa_mask);
    return sigaction(signo, &sa, nullptr);
#else
    return reinterpret_cast<int>(signal(signo, handler));
#endif
}

TString SignalDescription(int signo) {
#ifdef _unix_
    return TString(strsignal(signo));
#else
    return ToString(signo);
#endif
}

} // namespace

void InitSignals() {
    struct {
        int Signo;
        void (*Handler)(int);
        bool Block;
    } handlers[] = {
        { SIGTERM, SignalHandler, true },
#ifdef _unix_
        { SIGUSR1, SignalHandler, true },
        { SIGUSR2, SignalHandler, true },
#endif
        { SIGINT,  GracefulExitOnSigint, true },
        { SIGSEGV, LogBacktraceOnSigsegv, false },
        { 0, nullptr, false }
    };

    sigset_t handlingSignals;
    SigEmptySet(&handlingSignals);

    for (int i = 0; handlers[i].Handler; i++) {
        if (handlers[i].Block) {
            SigAddSet(&handlingSignals, handlers[i].Signo);
        }
        if (SetSignalHandler(handlers[i].Signo, handlers[i].Handler) == -1) {
            ythrow TSystemError() << "Cannot set handler for signal "
                                  << SignalDescription(handlers[i].Signo);
        }
    }

    // block all handling signals delivery before they are implicitly
    // become allowed by SigSuspend()
    if (SigProcMask(SIG_BLOCK, &handlingSignals, NULL) == -1) {
        ythrow TSystemError() << "Cannot set sigprocmask";
    }
}

void SigSuspend(const sigset_t* mask) {
#ifdef _unix_
    sigsuspend(mask);
#else
    Sleep(5000);
#endif
}

} // namespace NSolomon
