#include "operations.h"

#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>

using namespace NSolomon::NSlicer::NApi;

namespace NSolomon {

// TODO(ivanzhukov): move all operations inside a new class TSlices and use TVector or smth else as a backend instead of std::set

bool SlicesContain(const TSlices& slices, TNumId numId) {
    // find a slice whose End is _greater than or equal to_ numId
    auto it = slices.upper_bound(numId);

    if (it == slices.end() || numId < it->Start || numId > it->End) {
        return false;
    }

    return true;
}

TMatchedNumIds MatchSlicesWithNumIds(
        const TSlices& slices,
        const TVector<TNumId>& numIds) // has to be sorted
{
    TMatchedNumIds result;
    if (slices.empty()) {
        return result;
    }

    auto sIt = slices.begin();
    auto nIt = numIds.begin();

    while (sIt != slices.end() && nIt != numIds.end()) {
        if (*nIt < sIt->Start) {
            result.NotMatched.emplace_back(*nIt);
            ++nIt;
            continue;
        }

        if (*nIt > sIt->End) {
            ++sIt;
            continue;
        }

        result.Matched.emplace_back(*nIt);
        ++nIt;
    }

    while (nIt != numIds.end()) {
        result.NotMatched.emplace_back(*nIt);
        ++nIt;
    }

    return result;
}

ui32 SlicesSize(const TSlices& slices) {
    return Accumulate(slices, 0, [](auto acc, auto slice) { return acc + slice.Size(); });
}

TSlices::iterator ExtractSlicesOfTotalSize(
    TSlices& slices,
    TSlices::iterator it,
    ui32 numOfNumIdNeeded,
    TSlices& extractTo)
{
    while (numOfNumIdNeeded > 0) {
        if (numOfNumIdNeeded < it->Size()) {
            auto oldSliceStart = it->Start;
            auto newSliceStart = it->Start + numOfNumIdNeeded;
            auto newSliceEnd = it->End;
            // TODO(ivanzhukov): if we used TVector as a backend, we could change the element in place:
            slices.erase(it);
            it = slices.emplace(TSlice{newSliceStart, newSliceEnd}).first;

            extractTo.emplace(TSlice{oldSliceStart, newSliceStart - 1});
            break;
        }

        extractTo.emplace(*it);
        numOfNumIdNeeded -= it->Size();
        it = slices.erase(it);
    }

    return it;
}

TVector<TSlices> RearrangeSlicesIntoEvenParts(TSlices& originalSlices, ui32 numOfParts) {
    Y_ENSURE(numOfParts > 0, "numOfParts cannot be zero");

    if (originalSlices.empty()) {
        return {};
    }

    if (numOfParts == 1) {
        return {originalSlices};
    }

    ui32 keysInEachSlice = 0;
    ui32 keySpaceSize = 0;
    ui32 numOfNumIdsLeft = 0;

    for (auto& slice: originalSlices) {
        keySpaceSize += slice.Size();
    }

    keysInEachSlice = keySpaceSize / numOfParts;
    numOfNumIdsLeft = keySpaceSize % numOfParts;

    TVector<TSlices> result(keySpaceSize >= numOfParts ? numOfParts : numOfNumIdsLeft);
    auto currPart = result.begin();

    auto it = originalSlices.begin();

    while (it != originalSlices.end()) {
        ui32 numOfNumIdNeeded = keysInEachSlice;
        if (numOfNumIdsLeft > 0) {
            ++numOfNumIdNeeded;
            --numOfNumIdsLeft;
        }

        it = ExtractSlicesOfTotalSize(originalSlices, it, numOfNumIdNeeded, *currPart);
        ++currPart;
    }

    return result;
}

} // namespace NSolomon

