#include <solomon/libs/cpp/slices/operations.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/builder.h>

using namespace ::testing;
using namespace NSolomon::NSlicer::NApi;

namespace NSolomon {

TString DebugPrint(const TSlices& slices) {
    TString res;
    res += "[";

    size_t i = 0;
    for (const auto& slice: slices) {
        res += TStringBuilder() << " {" << slice.Start << "; " << slice.End << "} ";

        if (i != slices.size() - 1) {
            res += ",";
        }

        ++i;
    }

    res += "]";

    return res;
}

MATCHER_P(InsideSlices, slices, "checks if a numId is present (or not) inside slices") {
    return SlicesContain(slices, arg);
}

TEST(SlicesOperations, Contains) {
    auto slices = TSlices{
            {10, 20},
            {30, 40},
            {50, 60},
            {70, 80},
    };

    for (const auto& slice: slices) {
        for (size_t numId = slice.Start; numId <= slice.End; ++numId) {
            EXPECT_THAT(numId, InsideSlices(slices)) << "slices: " << DebugPrint(slices);
        }
    }

    for (size_t numId = 0; numId != slices.begin()->Start; ++numId) {
        EXPECT_THAT(numId, Not(InsideSlices(slices))) << "slices: " << DebugPrint(slices);
    }

    auto lastEnd = slices.rbegin()->End;
    for (TNumId numId = lastEnd + 1; numId != lastEnd + 11; ++numId) {
        EXPECT_THAT(numId, Not(InsideSlices(slices))) << "slices: " << DebugPrint(slices);
    }

    TVector<TSlice> sortedSlices(::Reserve(slices.size()));
    for (auto slice: slices) {
        sortedSlices.emplace_back(slice);
    }

    for (size_t i = 0; i != sortedSlices.size() - 1; ++i) {
        for (auto numId = sortedSlices[i].End + 1; numId != sortedSlices[i + 1].Start; ++numId) {
            EXPECT_THAT(numId, Not(InsideSlices(slices))) << "sortedSlices: " << DebugPrint(slices);
        }
    }
}

TEST(RearrangeSlicesIntoEvenPartsTest, Divibisble) {
    {
        TSlices slices;
        slices.emplace(1, 7);
        slices.emplace(8, 10);
        slices.emplace(11, 15);

        auto newSlices = RearrangeSlicesIntoEvenParts(slices, 5);
        ASSERT_EQ(newSlices.size(), 5ul);

        EXPECT_EQ(newSlices[0], (TSlices{
            {1, 3},
        }));
        EXPECT_EQ(newSlices[1], (TSlices{
            {4, 6},
        }));
        EXPECT_EQ(newSlices[2], (TSlices{
            {7, 7},
            {8, 9},
        }));
        EXPECT_EQ(newSlices[3], (TSlices{
            {10, 10},
            {11, 12}
        }));
        EXPECT_EQ(newSlices[4], (TSlices{
            {13, 15},
        }));
    }

    {
        TSlices slices;
        slices.emplace(1, 1);
        slices.emplace(2, 4);
        slices.emplace(5, 6);
        slices.emplace(7, 8);
        slices.emplace(9, 12);

        auto newSlices = RearrangeSlicesIntoEvenParts(slices, 3);
        ASSERT_EQ(newSlices.size(), 3ul);

        EXPECT_EQ(newSlices[0], (TSlices{
            {1, 1},
            {2, 4},
        }));
        EXPECT_EQ(newSlices[1], (TSlices{
            {5, 6},
            {7, 8},
        }));
        EXPECT_EQ(newSlices[2], (TSlices{
            {9, 12},
        }));
    }
}

TEST(RearrangeSlicesIntoEvenPartsTest, NotDivisible) {
    {
        TSlices slices;
        slices.emplace(1, 7);
        slices.emplace(8, 10);
        slices.emplace(11, 13);

        auto newSlices = RearrangeSlicesIntoEvenParts(slices, 5);
        ASSERT_EQ(newSlices.size(), 5ul);

        EXPECT_EQ(newSlices[0], (TSlices{
            {1, 3},
        }));
        EXPECT_EQ(newSlices[1], (TSlices{
            {4, 6},
        }));
        EXPECT_EQ(newSlices[2], (TSlices{
            {7, 7},
            {8, 9},
        }));
        EXPECT_EQ(newSlices[3], (TSlices{
            {10, 10},
            {11, 11},
        }));
        EXPECT_EQ(newSlices[4], (TSlices{
            {12, 13},
        }));
    }

    {
        TSlices slices;
        slices.emplace(10, 19);
        slices.emplace(30, 39); //  -- 10 + 10 + 10 == 30 keys. 30 == 7 * 4 + 2
        slices.emplace(50, 59);

        auto newSlices = RearrangeSlicesIntoEvenParts(slices, 7);
        ASSERT_EQ(newSlices.size(), 7ul);

        EXPECT_EQ(newSlices[0], (TSlices{
            {10, 14}, // 4 + 1
        }));
        EXPECT_EQ(newSlices[1], (TSlices{
            {15, 19}, // 4 + 1
        }));
        EXPECT_EQ(newSlices[2], (TSlices{
            {30, 33},
        }));
        EXPECT_EQ(newSlices[3], (TSlices{
            {34, 37}
        }));
        EXPECT_EQ(newSlices[4], (TSlices{
            {38, 39},
            {50, 51},
        }));
        EXPECT_EQ(newSlices[5], (TSlices{
            {52, 55},
        }));
        EXPECT_EQ(newSlices[6], (TSlices{
            {56, 59}
        }));
    }
}

TEST(RearrangeSlicesIntoEvenPartsTest, TooManySlices) {
    TSlices slices;
    slices.emplace(1, 4);

    auto newSlices = RearrangeSlicesIntoEvenParts(slices, 5);
    ASSERT_EQ(newSlices.size(), 4ul);

    EXPECT_EQ(newSlices[0], (TSlices{
        {1, 1},
    }));
    EXPECT_EQ(newSlices[1], (TSlices{
        {2, 2},
    }));
    EXPECT_EQ(newSlices[2], (TSlices{
        {3, 3},
    }));
    EXPECT_EQ(newSlices[3], (TSlices{
        {4, 4},
    }));
}

TEST(RearrangeSlicesIntoEvenPartsTest, OnlyOnePart) {
    TSlices slices;
    slices.emplace(1, 3);
    slices.emplace(7, 13);
    slices.emplace(21, 47);

    auto newSlices = RearrangeSlicesIntoEvenParts(slices, 1);
    ASSERT_EQ(newSlices.size(), 1ul);

    EXPECT_EQ(newSlices[0], (TSlices{
        slices,
    }));
}

TEST(MatchSlicesAndNumIds, Test) {
    TSlices slices;
    slices.emplace(3, 10);
    slices.emplace(12, 20);
    slices.emplace(37, 45);

    TVector<TNumId> numIds{
        1, 2, // before all slices
        3, 5, 10, // inside 1st
        11, // between 1st and 2nd
        12, 13, 20, // inside 2nd
        24, 25, // between 2nd and 3rd
        37, 40, 45, // inside 3rd
        99, 123,  // after all slices
    };

    auto result = MatchSlicesWithNumIds(slices, numIds);

    ASSERT_EQ(result.Matched.size(), 9ul);
    EXPECT_THAT(result.Matched, ElementsAre(3, 5, 10, 12, 13, 20, 37, 40, 45));

    ASSERT_EQ(result.NotMatched.size(), 7ul);
    EXPECT_THAT(result.NotMatched, ElementsAre(1, 2, 11, 24, 25, 99, 123));
}

} // namespace NSolomon
