#include "proto_decode.h"

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/metrics/metric_type.h>

using NMonitoring::ECompression;
using NMonitoring::EMetricType;
using NMonitoring::ETimePrecision;
using NMonitoring::TLabels;
using yandex::monitoring::slog::TShardData;
using yandex::solomon::model::MetricType;

namespace NSolomon::NSlog {
namespace {

EMetricType FromProto(MetricType type) {
    switch (type) {
        case MetricType::DGAUGE:
            return EMetricType::GAUGE;
        case MetricType::IGAUGE:
            return EMetricType::IGAUGE;
        case MetricType::COUNTER:
            return EMetricType::COUNTER;
        case MetricType::RATE:
            return EMetricType::RATE;
        case MetricType::HIST:
            return EMetricType::HIST;
        case MetricType::HIST_RATE:
            return EMetricType::HIST_RATE;
        case MetricType::DSUMMARY:
            return EMetricType::DSUMMARY;
        default:
            ythrow yexception() << "unsupported metric type: " << MetricType_Name(type);
    }
}

} // namespace

void DecodeProto(const TShardData& data, ILogWriter* logWriter) {
    logWriter->OnStreamBegin();

    TInstant commonTime = TInstant::MilliSeconds(data.GetCommonTimeMillis());
    if (commonTime) {
        logWriter->OnCommonTime(commonTime);
    }

    if (data.CommonLabelsSize() > 0) {
        logWriter->OnLabelsBegin();

        for (auto& cl: data.GetCommonLabels()) {
            logWriter->OnLabel(cl.GetName(), cl.GetValue());
        }

        logWriter->OnLabelsEnd();
    }

    if (auto stepMillis = data.GetStepMillis()) {
        logWriter->OnStep(TDuration::MilliSeconds(stepMillis));
    }

    for (auto& metric: data.GetMetrics()) {
        EMetricType type = FromProto(metric.GetType());

        logWriter->OnMetricBegin(type);

        logWriter->OnLabelsBegin();
        for (auto& l: metric.GetLabels()) {
            logWriter->OnLabel(l.GetName(), l.GetValue());
        }
        logWriter->OnLabelsEnd();

        TAggrTimeSeries series;
        bool isMergeSet = metric.GetMerge();
        bool isCountSet = false;
        bool isDenomSet = false;

        for (auto& point: metric.GetPoints()) {
            auto ts = TInstant::MilliSeconds(point.GetTimestampsMillis());

            size_t count = point.GetCount();
            ui64 denom = 0;

            isCountSet = count > 0;

            switch (type) {
                case EMetricType::GAUGE: {
                    double value = point.GetDoubleValue();
                    series.Add(ts, value, denom, count);
                    break;
                }
                case EMetricType::IGAUGE: {
                    i64 value = point.GetLongValue();
                    series.Add(ts, value, denom, count);
                    break;
                }
                case EMetricType::COUNTER:
                case EMetricType::RATE: {
                    ui64 value = point.GetLongValue();

                    if (EMetricType::RATE == type) {
                        denom = point.GetDenom();
                        isDenomSet = denom > 0;
                    }

                    series.Add(ts, value, denom, count);
                    break;
                }
                case EMetricType::HIST:
                case EMetricType::HIST_RATE: {
                    auto& histogram = point.GetHistogram();

                    if (EMetricType::HIST_RATE == type) {
                        denom = point.GetDenom();
                        isDenomSet = denom > 0;
                    }

                    NMonitoring::TBucketBounds bounds{::Reserve(histogram.bounds_size())};
                    NMonitoring::TBucketValues values{::Reserve(histogram.buckets_size())};

                    for (auto& b: histogram.bounds()) {
                        bounds.emplace_back(b);
                    }

                    for (auto& v: histogram.buckets()) {
                        values.emplace_back(v);
                    }

                    auto snapshot = NMonitoring::ExplicitHistogramSnapshot(std::move(bounds), std::move(values));
                    series.Add(ts, snapshot.Get(), denom, count);

                    break;
                }
                default:
                    ythrow yexception() << "unsupported metric type: " << type;
            }
        }

        Y_ENSURE(!series.Empty(), "no points are specified in data");

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        if (isMergeSet) {
            flags = CombineLogFlags(flags, ELogFlags::Merge);
        }

        if (isCountSet) {
            flags = CombineLogFlags(flags, ELogFlags::Count);
        }

        if (isDenomSet) {
            flags = CombineLogFlags(flags, ELogFlags::Denom);
        }

        logWriter->OnTimeSeries(flags, std::move(series));

        logWriter->OnMetricEnd();
    }

    logWriter->OnStreamEnd();
}

} // namespace NSolomon::NSlog
