#pragma once

#if !defined(INCLUDE_CODEC_IMPL_H)
#error "you should never include codec_impl.h directly"
#endif

#include <solomon/libs/cpp/slog/codec/fwd.h>
#include <solomon/libs/cpp/ts_model/point_type.h>

#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/spack/varint.h>
#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/stream/input.h>
#include <util/stream/output.h>

namespace NSolomon::NSlog {

inline NMonitoring::EMetricType ToMonlibType(NTsModel::EPointType type) {
    switch (type) {
        case NTsModel::EPointType::Unknown:
            return NMonitoring::EMetricType::UNKNOWN;
        case NTsModel::EPointType::DGauge:
            return NMonitoring::EMetricType::GAUGE;
        case NTsModel::EPointType::IGauge:
            return NMonitoring::EMetricType::IGAUGE;
        case NTsModel::EPointType::Counter:
            return NMonitoring::EMetricType::COUNTER;
        case NTsModel::EPointType::Rate:
            return NMonitoring::EMetricType::RATE;
        case NTsModel::EPointType::Hist:
            return NMonitoring::EMetricType::HIST;
        case NTsModel::EPointType::HistRate:
            return NMonitoring::EMetricType::HIST_RATE;
        case NTsModel::EPointType::LogHist:
            return NMonitoring::EMetricType::LOGHIST;
        case NTsModel::EPointType::DSummary:
            return NMonitoring::EMetricType::DSUMMARY;
    }
}

inline std::optional<NTsModel::EPointType> FromMonlibType(NMonitoring::EMetricType type) {
    switch (type) {
        case NMonitoring::EMetricType::UNKNOWN:
            return std::nullopt;
        case NMonitoring::EMetricType::GAUGE:
            return NTsModel::EPointType::DGauge;
        case NMonitoring::EMetricType::COUNTER:
            return NTsModel::EPointType::Counter;
        case NMonitoring::EMetricType::RATE:
            return NTsModel::EPointType::Rate;
        case NMonitoring::EMetricType::IGAUGE:
            return NTsModel::EPointType::IGauge;
        case NMonitoring::EMetricType::HIST:
            return NTsModel::EPointType::Hist;
        case NMonitoring::EMetricType::HIST_RATE:
            return NTsModel::EPointType::HistRate;
        case NMonitoring::EMetricType::DSUMMARY:
            return NTsModel::EPointType::DSummary;
        case NMonitoring::EMetricType::LOGHIST:
            return NTsModel::EPointType::LogHist;
    }
}

inline ui8 PackTypes(NTsModel::EPointType type, NMonitoring::EValueType valueType) {
    return (static_cast<ui8>(ToMonlibType(type)) << 2u) | static_cast<ui8>(valueType);
}

template <typename T>
inline TBytes WriteFixed(IOutputStream* out, T value) {
    out->Write(&value, sizeof(value));
    return sizeof(value);
}

inline TBytes WriteZeros(IOutputStream* out, TBytes bytes) {
    for (TBytes index = 0; index < bytes; index++) {
        out->Write(0);
    }
    return bytes;
}

template <typename T>
inline T ReadFixed(IInputStream* in) {
    T value;
    size_t readBytes = in->Load(&value, sizeof(T));
    if (readBytes != sizeof(T)) {
        ythrow TDecodeError() << "unexpected end of file";
    }
    return value;
}

inline ui32 ReadVariadic(IInputStream* in) {
    ui32 value = 0;
    switch (NMonitoring::TryReadVarUInt32(in, &value)) {
        case NMonitoring::EReadResult::OK:
            return value;
        case NMonitoring::EReadResult::ERR_OVERFLOW:
            ythrow TDecodeError() << "varint is too large";
        case NMonitoring::EReadResult::ERR_UNEXPECTED_EOF:
            ythrow TDecodeError() << "unexpected end of file";
        default:
            ythrow TDecodeError() << "unknown error while reading varint";
    }
}

inline void Skip(IInputStream* in, size_t bytes) {
    if (Y_UNLIKELY(in->Skip(bytes) != bytes)) {
        ythrow TDecodeError{} << "unexpected end of input";
    }
}

inline NTsModel::EPointType ReadPointType(ui8 typesByte) {
    NMonitoring::EMetricType type = NMonitoring::EMetricType::UNKNOWN;
    if (!NMonitoring::TryDecodeMetricType(typesByte >> 2, &type)) {
        ythrow TDecodeError() << "unknown metric type: " << (typesByte >> 2);
    }
    if (auto modelType = FromMonlibType(type)) {
        return *modelType;
    } else {
        ythrow TDecodeError() << "metric type is set to Unknown";
    }
}

} // namespace NSolomon::NSlog
