#include "log_data.h"
#include "slog.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/spack/compression.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/spack/varint.h>

#include <util/generic/buffer.h>
#include <util/generic/maybe.h>
#include <util/stream/buffer.h>
#include <util/generic/ymath.h>

using NMonitoring::TryDecodeMetricType;
using NMonitoring::TryDecodeValueType;
using NMonitoring::ECompression;
using NMonitoring::EMetricType;
using NMonitoring::EMetricValueType;
using NMonitoring::ETimePrecision;
using NMonitoring::EValueType;
using NMonitoring::EncodeMetricType;
using NMonitoring::IHistogramSnapshotPtr;
using NMonitoring::IMetricConsumer;
using NMonitoring::IMetricEncoder;
using NMonitoring::IMetricEncoderPtr;
using NMonitoring::TBucketBound;
using NMonitoring::TBucketValue;
using NMonitoring::TExplicitHistogramSnapshot;
using NMonitoring::ISummaryDoubleSnapshotPtr;
using NMonitoring::TLogHistogramSnapshotPtr;

namespace NSolomon::NSlog {
namespace {

constexpr TDenom MAX_DENOM_VALUE = static_cast<TDenom>(TDuration::Days(7).MicroSeconds());
constexpr TCount MAX_COUNT_VALUE = 10'000'000U;
const TInstant MIN_TIME = TInstant::ParseIso8601("2000-01-01T00:00:00Z");
const TInstant MAX_TIME = TInstant::ParseIso8601("2038-01-19T03:14:07Z");

struct Y_PACKED NSlogDataHeader {
    ui16 Magic; // 2
    ui16 Version; // 4
    ui32 NumId; // 8
    ui64 CommonTsMillis; // 16
    ui32 StepMillis; // 20
    ui32 MetricsCount; // 24
    ui32 PointsCount; // 28
    ui8 TimePrecision; // 29
    ui8 CompressionAlg; // 30
    ui8 CodingScheme; // 31
    ui8 Reserved[NSlogData::SKIP_BYTES]; // 32
};

static_assert(
    sizeof(NSlogDataHeader) == NSlogData::HEADER_SIZE,
    "size of the resulting log header != HEADER_SIZE"
);

/// C++ implementation of https://nda.ya.ru/t/qkl8Jt8D3VtuRy
class TLogDataHeader {
public:
    TLogDataHeader(
            ui32 numId,
            EDataCodingScheme codingScheme,
            ETimePrecision timePrecision,
            ECompression compressionAlg,
            TInstant commonTs,
            TDuration step,
            ui32 metricsCount,
            ui32 pointsCount)
    {
        Content = {
            .Magic = NSlogData::VALID_MAGIC,
            .Version = NSlogData::CURRENT_VERSION,
            .NumId = numId,
            .CommonTsMillis = commonTs.MilliSeconds(),
            .StepMillis = static_cast<decltype(NSlogDataHeader::StepMillis)>(step.MilliSeconds()),
            .MetricsCount = metricsCount,
            .PointsCount = pointsCount,
            .TimePrecision = NMonitoring::EncodeTimePrecision(timePrecision),
            .CompressionAlg = NMonitoring::EncodeCompression(compressionAlg),
            .CodingScheme = static_cast<ui8>(codingScheme),
        };
    }

    TLogDataHeader(IInputStream* in) {
        Content = ReadFixed<NSlogDataHeader>(in);

        if (Content.Magic != NSlogData::VALID_MAGIC) {
            ythrow TDecodeError()
                << "invalid data magic: expected " << NSlogData::VALID_MAGIC << ", "
                << "got " << Content.Magic;
        }

        if (Content.Version > NSlogData::CURRENT_VERSION) {
            ythrow TDecodeError()
                << "unsupported version of metadata: " << Content.Version;
        }

        if (!NMonitoring::TryDecodeTimePrecision(Content.TimePrecision, nullptr)) {
            ythrow TDecodeError()
                << "unknown data time precision: " << Content.TimePrecision;
        }

        if (!NMonitoring::TryDecodeCompression(Content.CompressionAlg, nullptr)) {
            ythrow TDecodeError()
                << "unknown data compression algorithm: " << Content.CompressionAlg;
        }

        if (Content.Version < NSlogData::VERSION_2_0) {
            Content.CodingScheme = static_cast<ui8>(EDataCodingScheme::Slog);
        }

        if (Content.CodingScheme > static_cast<ui8>(EDataCodingScheme::Last)) {
            ythrow TDecodeError()
                << "unknown coding scheme: " << Content.CodingScheme;
        }
        if (Content.CodingScheme == static_cast<ui8>(EDataCodingScheme::TsModel)) {
            ythrow TDecodeError()
                << "unable to decode ts model data";
        }
    }

    void WriteTo(IOutputStream* out) {
        WriteFixed(out, Content);
    }

public:
    NSlogDataHeader Content;
};

class TLogDataBuilder: public ILogDataBuilder {
public:
    TLogDataBuilder(
            ui32 numId,
            EDataCodingScheme codingScheme,
            ETimePrecision timePrecision,
            ECompression compressionAlg,
            IOutputStream* out)
        : NumId_{numId}
        , CodingScheme_{codingScheme}
        , TimePrecision_{timePrecision}
        , CompressionAlg_{compressionAlg}
        , Out_{out}
    {
        CompressedOut_ = CompressedOutput(TempOut_, CompressionAlg_);
        if (CompressedOut_) {
            TempOut_ = CompressedOut_.Get();
        }
    }

private:
    void Close() override {
        if (IsClosed_) {
            return;
        }

        IsClosed_ = true;

        TLogDataHeader logHeader{
            NumId_,
            CodingScheme_, TimePrecision_, CompressionAlg_,
            CommonTime_, Step_,
            MetricsCount_, PointsCount_
        };
        logHeader.WriteTo(Out_);

        TempOut_->Finish();

        // TODO: a place for an optimization. Pass an arg stream that supports position changing
        const auto& tempBuffer = TempData_.Buffer();
        Out_->Write(tempBuffer.Data(), tempBuffer.Size());
    }

    void OnCommonTime(TInstant time) override {
        if (Y_UNLIKELY(!CommonTime_)) {
            Y_ENSURE(time >= MIN_TIME && time <= MAX_TIME,
                     "time should be in range: [" << MIN_TIME << ", " << MAX_TIME << "] but specified " << time);
            CommonTime_ = time;
        } else {
            Y_ENSURE(CommonTime_ == time,
                     "passed different common time value: prev (" << CommonTime_ << ") != new (" << time << ')');
        }
    }

    void OnStep(TDuration step) override {
        if (Y_UNLIKELY(!Step_)) {
            Step_ = step;
        } else {
            Y_ENSURE(Step_ == step, "passed different step value: prev (" << Step_ << ") != new (" << step << ')');
        }
    }

    TBytes OnPoint(NTsModel::EPointType kind, ELogFlagsComb flags, TAggrPointWithType&& point) override {
        Y_ENSURE(
            CodingScheme_ == EDataCodingScheme::Slog,
            "OnPoint is not allowed in coding scheme " << CodingScheme_);

        ++MetricsCount_;
        ++PointsCount_;

        auto bytesWritten = WritePoint(kind, flags, point, point.GetType());
        point.ClearValue();

        return bytesWritten;
    }

    TBytes OnTimeSeries(NTsModel::EPointType kind, ELogFlagsComb flags, const TAggrTimeSeries& timeseries) override {
        Y_ENSURE(
            CodingScheme_ == EDataCodingScheme::Slog,
            "OnTimeSeries is not allowed in coding scheme " << CodingScheme_);

        ++MetricsCount_;
        PointsCount_ += timeseries.Size();

        Y_ENSURE(!timeseries.Empty(), "cannot log empty timeseries");

        if (1 == timeseries.Size()) {
            return WritePoint(kind, flags, timeseries[0], timeseries.GetValueType());
        } else {
            return WriteMany(kind, flags, timeseries);
        }
    }

    TBytes OnTimeSeriesNativeEncoded(NTsModel::EPointType kind, ui16 columns, NTs::TBitSpan encoded, size_t numPoints) override {
        Y_ENSURE(
            CodingScheme_ == EDataCodingScheme::TsModel,
            "OnTimeSeriesNativeEncoded is not allowed in coding scheme " << CodingScheme_);

        ++MetricsCount_;
        PointsCount_ += numPoints;

        Y_ENSURE(encoded.Size() != 0, "cannot log empty time series");

        TBytes written = WriteFixed(TempOut_, PackTypes(kind, NMonitoring::EValueType::NONE));
        written += WriteFixed(TempOut_, columns);
        written += WriteFixed(TempOut_, static_cast<ui64>(encoded.Size()));
        TempOut_->Write({reinterpret_cast<const char*>(encoded.Data()), encoded.SizeBytes()});
        written += encoded.SizeBytes();

        return written;
    }

private:
    TBytes WriteHistogram(const NMonitoring::IHistogramSnapshot& histogram) {
        ValidateHistogram(histogram);
        ui32 count = histogram.Count();
        TBytes written = NMonitoring::WriteVarUInt32(TempOut_, count);

        for (ui32 i = 0; i < count; i++) {
            TBucketBound bound = histogram.UpperBound(i);
            written += WriteFixed(TempOut_, bound);
        }
        for (ui32 i = 0; i < count; i++) {
            TBucketValue value = histogram.Value(i);
            written += WriteFixed(TempOut_, value);
        }

        return written;
    }

    static void ValidateHistogram(const NMonitoring::IHistogramSnapshot& histogram) {
        ui32 bucketCount = histogram.Count();
        Y_ENSURE(bucketCount >= 0 && bucketCount <= NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, "invalid bucket count: " << bucketCount);
        for (ui32 i = 1; i < bucketCount; i++) {
            TBucketBound bound = histogram.UpperBound(i);
            TBucketBound prev = histogram.UpperBound(i - 1u);
            Y_ENSURE(prev < bound, "histogram bucket unsorted or not unique: " << prev << " >= " << bound);
        }
    }

    TBytes WriteSummaryDouble(const NMonitoring::ISummaryDoubleSnapshot& summary) {
        TBytes written = 0;
        written += WriteFixed(TempOut_, summary.GetCount());
        written += WriteFixed(TempOut_, summary.GetSum());
        written += WriteFixed(TempOut_, summary.GetMin());
        written += WriteFixed(TempOut_, summary.GetMax());
        written += WriteFixed(TempOut_, summary.GetLast());
        return written;
    }

    TBytes WriteLogHistogram(const NMonitoring::TLogHistogramSnapshot& logHist) {
        TBytes written = 0;
        written += WriteFixed(TempOut_, logHist.Base());
        written += WriteFixed(TempOut_, logHist.ZerosCount());
        written += NMonitoring::WriteVarUInt32(TempOut_, static_cast<ui32>(logHist.StartPower()));
        written += NMonitoring::WriteVarUInt32(TempOut_, logHist.Count());
        for (ui32 i = 0; i < logHist.Count(); ++i) {
            written += WriteFixed(TempOut_, logHist.Bucket(i));
        }
        return written;
    }

    TBytes WriteTime(TInstant instant) {
        Y_ENSURE(instant >= MIN_TIME && instant <= MAX_TIME,
                "Time should be in range: [" << MIN_TIME << ", " << MAX_TIME << "] but specified " << instant);
        switch (TimePrecision_) {
            case ETimePrecision::SECONDS: {
                ui32 time = static_cast<ui32>(instant.Seconds());
                return WriteFixed(TempOut_, time);
            }
            case ETimePrecision::MILLIS: {
                ui64 time = static_cast<ui64>(instant.MilliSeconds());
                return WriteFixed(TempOut_, time);
            }
        }
    }

    template <typename TPoint>
    TBytes WriteOptional(ELogFlagsComb flags, const TPoint& point) {
        if (static_cast<ELogFlagsComb>(ELogFlags::None) == flags) {
            return 0;
        }

        TBytes written = 0;

        if (flags & static_cast<ELogFlagsComb>(ELogFlags::Count)) {
            auto count = point.GetCount();
            Y_ENSURE(count >= 0 && count < MAX_COUNT_VALUE,
                    "Invalid count " << count << " expected [0; " << MAX_COUNT_VALUE << ")");
            written += WriteFixed(TempOut_, count);
        }

        if (flags & static_cast<ELogFlagsComb>(ELogFlags::Denom)) {
            auto denom = point.GetDenom();
            Y_ENSURE(denom >= 0 && denom < MAX_DENOM_VALUE,
                     "Invalid denom " << denom << " expected [0; " << MAX_DENOM_VALUE << ")");
            written += WriteFixed(TempOut_, denom);
        }

        return written;
    }

    TBytes WriteValue(NTsModel::EPointType kind, const NMonitoring::TMetricValueWithType& value) {
        switch (kind) {
            case NTsModel::EPointType::DGauge:
                Y_ENSURE(!IsNan(value.AsDouble()), "NaN value");
                return WriteFixed(TempOut_, value.AsDouble());
            case NTsModel::EPointType::IGauge:
                return WriteFixed(TempOut_, value.AsInt64());
            case NTsModel::EPointType::Rate:
            case NTsModel::EPointType::Counter:
                return WriteFixed(TempOut_, value.AsUint64());
            case NTsModel::EPointType::Hist:
            case NTsModel::EPointType::HistRate:
                return WriteHistogram(*value.AsHistogram());
            case NTsModel::EPointType::LogHist:
                return WriteLogHistogram(*value.AsLogHistogram());
            case NTsModel::EPointType::DSummary:
                return WriteSummaryDouble(*value.AsSummaryDouble());
                // TODO(ivanzhukov@):
//            case NTsModel::EPointType::Isummary:
//                return writeISummary(value.AsSummaryInt64(type));
            default:
                ythrow yexception() << "unsupported kind: " << kind;
                // TODO:
//                throw new UnsupportedOperationException("Unsupported kind: " + kind);
        }
    }

    TBytes WriteValue(NTsModel::EPointType kind, const NMonitoring::TMetricValue& value, EMetricValueType type) {
        switch (kind) {
            case NTsModel::EPointType::DGauge:
                Y_ENSURE(!IsNan(value.AsDouble()), "NaN value");
                return WriteFixed(TempOut_, value.AsDouble(type));
            case NTsModel::EPointType::IGauge:
                return WriteFixed(TempOut_, value.AsInt64(type));
            case NTsModel::EPointType::Rate:
            case NTsModel::EPointType::Counter:
                return WriteFixed(TempOut_, value.AsUint64(type));
            case NTsModel::EPointType::Hist:
            case NTsModel::EPointType::HistRate:
                return WriteHistogram(*value.AsHistogram());
            case NTsModel::EPointType::LogHist:
                return WriteLogHistogram(*value.AsLogHistogram(type));
            case NTsModel::EPointType::DSummary:
                return WriteSummaryDouble(*value.AsSummaryDouble(type));
                // TODO(ivanzhukov@):
//            case NTsModel::EPointType::Isummary:
//                return writeISummary(value.AsSummaryInt64(type));
            default:
                ythrow yexception() << "unsupported kind: " << kind;
                // TODO:
//                throw new UnsupportedOperationException("Unsupported kind: " + kind);
        }
    }

    template <typename TPoint>
    TBytes WritePoint(NTsModel::EPointType kind, ELogFlagsComb flags, const TPoint& aggrPoint, EMetricValueType type) {
        TBytes written = 0;
        const auto ts = aggrPoint.GetTime();
        Y_ENSURE(TInstant::Zero() != CommonTime_ || TInstant::Zero() != ts);

        if (TInstant::Zero() == ts || ts == CommonTime_) {
            written += WriteFixed(TempOut_, PackTypes(kind, EValueType::ONE_WITHOUT_TS));
            written += WriteFixed(TempOut_, flags);
        } else {
            written += WriteFixed(TempOut_, PackTypes(kind, EValueType::ONE_WITH_TS));
            written += WriteFixed(TempOut_, flags);
            written += WriteTime(ts);
        }

        if constexpr (std::is_same_v<TPoint, TAggrPointWithType>) {
            written += WriteValue(kind, aggrPoint.GetValue());
        } else if constexpr (std::is_same_v<TPoint, TAggrPoint>) {
            written += WriteValue(kind, aggrPoint.GetValue(), type);
        } else {
            static_assert(TDependentFalse<TPoint>, "unknown point type");
        }

        written += WriteOptional(flags, aggrPoint);

        return written;
    }

    TBytes WriteMany(NTsModel::EPointType kind, ELogFlagsComb flags, const TAggrTimeSeries& timeSeries) {
        TBytes written = 0;

        written += WriteFixed(TempOut_, PackTypes(kind, EValueType::MANY_WITH_TS));
        written += WriteFixed(TempOut_, flags);
        written += NMonitoring::WriteVarUInt32(TempOut_, timeSeries.Size());

        auto type = timeSeries.GetValueType();

        for (size_t i = 0; i < timeSeries.Size(); i++) {
            written += WriteTime(timeSeries[i].GetTime());
            written += WriteValue(kind, timeSeries[i].GetValue(), type);
            written += WriteOptional(flags, timeSeries[i]);
        }

        return written;
    }

private:
    ui32 NumId_;
    EDataCodingScheme CodingScheme_;
    ETimePrecision TimePrecision_;
    ECompression CompressionAlg_;
    IOutputStream* Out_;
    TBufferStream TempData_;
    IOutputStream* TempOut_{&TempData_};
    THolder<IOutputStream> CompressedOut_;
    bool IsClosed_{false};

    TInstant CommonTime_;
    TDuration Step_;
    ui32 MetricsCount_{0};
    ui32 PointsCount_{0};
};

class TLogDataIterator: public ILogDataIterator {
public:
    TLogDataIterator(IInputStream* in)
        : In_{in}
        , DataHeader_{in}
    {
        auto codingScheme = static_cast<EDataCodingScheme>(DataHeader_.Content.CodingScheme);
        if (codingScheme != EDataCodingScheme::Slog) {
            ythrow TDecodeError()
                << "coding scheme " << codingScheme << " is not supported by this parser";
        }

        auto compression = NMonitoring::DecodeCompression(DataHeader_.Content.CompressionAlg);
        if (compression == ECompression::UNKNOWN) {
            ythrow TDecodeError()
                << "data compression algorithm is set to 'UNKNOWN'";
        }

        CompressedIn_ = CompressedInput(In_, compression);
        if (CompressedIn_) {
            In_ = CompressedIn_.Get();
        }
    }

    ui32 NumId() override {
        return DataHeader_.Content.NumId;
    }

    TInstant CommonTs() override {
        return TInstant::MilliSeconds(DataHeader_.Content.CommonTsMillis);
    }

    TDuration Step() override {
        return TDuration::MilliSeconds(DataHeader_.Content.StepMillis);
    }

    NMonitoring::ETimePrecision TimePrecision() override {
        return static_cast<ETimePrecision>(DataHeader_.Content.TimePrecision);
    }

private:
    /// ILogDataIterator::*

    bool HasNext() override {
        return MetricIdx_ < DataHeader_.Content.MetricsCount;
    }

    TLogDataRecord Next() override {
        if (MetricIdx_ >= DataHeader_.Content.MetricsCount) {
            ythrow yexception() << "no metrics left in data";
        }

        TLogDataRecord record;
        record.NumId = DataHeader_.Content.NumId;
        record.StepMillis = DataHeader_.Content.StepMillis;

        auto typesByte = ReadFixed<ui8>(In_);

        record.Kind = ReadPointType(typesByte);
        ELogFlagsComb flags = ReadFixed<ELogFlagsComb>(In_);
        record.Flags = flags;
        // TODO: check this?

        EValueType type;
        if (!TryDecodeValueType(typesByte & 0x03, &type)) {
            ythrow TDecodeError()
                << "unknown metric value type: " << (typesByte & 0x03) << " at metric #" << MetricIdx_;
        }

        switch (type) {
            case EValueType::ONE_WITHOUT_TS: {
                TInstant commonTs = TInstant::MilliSeconds(DataHeader_.Content.CommonTsMillis);
                ProcessValueOne(record, record.Kind, flags, commonTs);
                break;
            }
            case EValueType::ONE_WITH_TS: {
                TInstant ts = ReadTsMillisOrUseCommon();
                ProcessValueOne(record, record.Kind, flags, ts);
                break;
            }
            case EValueType::MANY_WITH_TS:
                ProcessValueMany(record, record.Kind, flags);
                break;
            case EValueType::NONE:
                break;
            default:
                ythrow yexception()
                    << "invalid metric value type that was not reported by 'TryDecodeValueType': "
                    << static_cast<ui16>(type);
        }

        ++MetricIdx_;
        return record;
    }

private:
    ui64 ReadTsMillis() {
        return DataHeader_.Content.TimePrecision == NMonitoring::EncodeTimePrecision(ETimePrecision::SECONDS)
               ? static_cast<ui64>(ReadFixed<ui32>(In_)) * 1000u
               : ReadFixed<ui64>(In_);
    }

    TInstant ReadTsMillisOrUseCommon() {
        ui64 tsMillis = ReadTsMillis();
        if (tsMillis == 0) {
            tsMillis = DataHeader_.Content.CommonTsMillis;
        }

        return TInstant::MilliSeconds(tsMillis);
    }

    IHistogramSnapshotPtr ReadHistogram() {
        ui32 bucketsCount = ReadVariadic(In_);
        if (bucketsCount > NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT) {
            ythrow TDecodeError()
                << "too many buckets in a histogram: " << bucketsCount;
        }

        auto s = TExplicitHistogramSnapshot::New(bucketsCount);

        for (ui32 i = 0; i < bucketsCount; i++) {
            double doubleBound = ReadFixed<double>(In_);
            (*s)[i].first = doubleBound;
        }

        // values
        for (ui32 i = 0; i < bucketsCount; i++) {
            (*s)[i].second = ReadFixed<ui64>(In_);
        }

        return s;
    }

    ISummaryDoubleSnapshotPtr ReadSummaryDouble() {
        ui64 count = ReadFixed<ui64>(In_);
        double sum = ReadFixed<double>(In_);
        double min = ReadFixed<double>(In_);
        double max = ReadFixed<double>(In_);
        double last = ReadFixed<double>(In_);
        return MakeIntrusive<NMonitoring::TSummaryDoubleSnapshot>(sum, min, max, last, count);
    }

    TLogHistogramSnapshotPtr ReadLogHistogram() {
        double base = ReadFixed<double>(In_);
        ui64 zerosCount = ReadFixed<ui64>(In_);
        int startPower = static_cast<int>(ReadVariadic(In_));
        ui32 bucketsCount = ReadVariadic(In_);
        if (bucketsCount > NMonitoring::LOG_HIST_MAX_BUCKETS) {
            ythrow TDecodeError()
                << "too many buckets in a log histogram: " << bucketsCount;
        }

        TVector<double> buckets;
        buckets.reserve(bucketsCount);
        for (ui32 i = 0; i < bucketsCount; ++i) {
            buckets.emplace_back(ReadFixed<double>(In_));
        }
        return MakeIntrusive<NMonitoring::TLogHistogramSnapshot>(base, zerosCount, startPower, std::move(buckets));
    }

    struct TOptionalInfo {
        TCount Count;
        TDenom Denom;
    };

    TOptionalInfo ReadOptional(ELogFlagsComb flags) {
        TCount count = 0;
        TDenom denom = 0;

        if (static_cast<ELogFlagsComb>(ELogFlags::None) != flags) {
            if (static_cast<ELogFlagsComb>(ELogFlags::Count) & flags) {
                count = ReadFixed<TCount>(In_);
                Y_ENSURE(count >= 0 && count < MAX_COUNT_VALUE,
                        "Invalid count " << count << " expected [0; " << MAX_COUNT_VALUE << ")");
            }

            if (static_cast<ELogFlagsComb>(ELogFlags::Denom) & flags) {
                denom = ReadFixed<TDenom>(In_);
                Y_ENSURE(denom >= 0 && denom < MAX_DENOM_VALUE,
                        "Invalid denom " << denom << " expected [0; " << MAX_DENOM_VALUE << ")");
            }
        }

        return {count, denom};
    }

    TAggrPoint ReadPointValue(TAggrTimeSeries& series, NTsModel::EPointType kind, TInstant ts, ELogFlagsComb flags) {
        TAggrPoint point;

        switch (kind) {
            case NTsModel::EPointType::DGauge: {
                auto value = ReadFixed<double>(In_);
                auto [count, denom] = ReadOptional(flags);
                series.Add(ts, value, denom, count);
                break;
            }
            case NTsModel::EPointType::IGauge: {
                auto value = ReadFixed<i64>(In_);
                auto [count, denom] = ReadOptional(flags);
                series.Add(ts, value, denom, count);
                break;
            }
            case NTsModel::EPointType::Rate:
            case NTsModel::EPointType::Counter: {
                auto value = ReadFixed<ui64>(In_);
                auto [count, denom] = ReadOptional(flags);
                series.Add(ts, value, denom, count);
                break;
            }
            case NTsModel::EPointType::HistRate:
            case NTsModel::EPointType::Hist: {
                auto hs = ReadHistogram();
                auto [count, denom] = ReadOptional(flags);
                series.Add(ts, hs.Get(), denom, count);
                break;
            }
            case NTsModel::EPointType::LogHist: {
                auto lhs = ReadLogHistogram();
                auto[count, denom] = ReadOptional(flags);
                series.Add(ts, lhs.Get(), denom, count);
                break;
            }
            case NTsModel::EPointType::DSummary: {
                auto smry = ReadSummaryDouble();
                auto[count, denom] = ReadOptional(flags);
                series.Add(ts, smry.Get(), denom, count);
                break;
            }
                // TODO:
//            case NTsModel::EPointType::Isummary:
//                readSummaryInt64(tmp);
//                break;
            default:
                ythrow TDecodeError() << "unsupported metric kind: " << kind;
        }

        return point;
    }

    void ProcessValueOne(TLogDataRecord& record, NTsModel::EPointType kind, ELogFlagsComb flags, TInstant ts) {
        ReadPointValue(record.TimeSeries, kind, ts, flags);
    }

    void ProcessValueMany(TLogDataRecord& record, NTsModel::EPointType kind, ELogFlagsComb flags) {
        auto count = ReadVariadic(In_);

        for (size_t i = 0; i != count; ++i) {
            auto ts = TInstant::MilliSeconds(ReadTsMillis());
            ProcessValueOne(record, kind, flags, ts);
        }
    }

private:
    IInputStream* In_;
    THolder<IInputStream> CompressedIn_;

    TLogDataHeader DataHeader_;
    size_t MetricIdx_{0};
};

} // namespace

ILogDataIteratorPtr CreateLogDataIterator(IInputStream* in) {
    return MakeHolder<TLogDataIterator>(in);
}

ILogDataBuilderPtr CreateLogDataBuilder(
        ui32 numId,
        EDataCodingScheme codingScheme,
        ETimePrecision timePrecision,
        ECompression compressionAlg,
        IOutputStream* out)
{
    return MakeHolder<TLogDataBuilder>(numId, codingScheme, timePrecision, compressionAlg, out);
}

} // namespace NSolomon::NSlog
