#include "header.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H

#include <util/generic/size_literals.h>

namespace NSolomon::NSlog::NResolvedMeta {

inline void ReadMagic(IInputStream* in) {
    auto magic = ReadFixed<ui16>(in);
    if (magic != VALID_MAGIC) {
        ythrow TDecodeError{}
            << "invalid unresolved meta magic: expected " << VALID_MAGIC << ", "
            << "got " << magic;
    }
}

inline EVersion ReadVersion(IInputStream* in) {
    auto version = ReadFixed<ui8>(in);
    if (version < MIN_VERSION || version > MAX_VERSION) {
        ythrow TDecodeError()
            << "unsupported version of unresolved meta: " << version;
    }
    return static_cast<EVersion>(version);
}

inline ui32 ReadNumId(IInputStream* in) {
    auto numId = ReadFixed<ui32>(in);
    if (numId == 0) {
        ythrow TDecodeError()
            << "invalid numId of unresolved meta: " << numId;
    }
    return numId;
}

inline NMonitoring::ECompression ReadCompression(IInputStream* in) {
    auto compression = ReadFixed<ui8>(in);
    auto result = NMonitoring::ECompression::UNKNOWN;

    if (!NMonitoring::TryDecodeCompression(compression, &result)) {
        ythrow TDecodeError{}
            << "unknown unresolved meta compression algorithm: " << compression;
    }

    if (result == NMonitoring::ECompression::UNKNOWN) {
        ythrow TDecodeError{}
            << "unresolved meta compression algorithm is set to 'UNKNOWN'";
    }

    return result;
}

inline yandex::solomon::stockpile::EDecimPolicy ReadDecimPolicy(IInputStream* in) {
    int policyId = static_cast<int>(ReadFixed<ui8>(in));
    if (!yandex::solomon::stockpile::EDecimPolicy_IsValid(policyId)) {
        ythrow TDecodeError{}
            << "resolved meta unknown decim policy " << policyId;
    }

    return static_cast<yandex::solomon::stockpile::EDecimPolicy>(policyId);
}

THeader ReadHeader(IInputStream* in) {
    ReadMagic(in);
    THeader result;
    Skip(in, 1); // minor version not used anymore
    result.Version = ReadVersion(in);
    result.NumId = ReadNumId(in);
    result.ProducerId = ReadFixed<ui32>(in);
    result.ProducerSeqNo = ReadFixed<ui64>(in);
    result.MetricsCount = ReadFixed<ui32>(in);
    result.PointsCount = ReadFixed<ui32>(in);
    result.CompressionAlg = ReadCompression(in);
    result.DecimPolicy = ReadDecimPolicy(in);
    Skip(in, 2);
    return result;
}

TBytes WriteHeader(IOutputStream* out, THeader header) {
    TBytes bytes = 0;
    bytes += WriteFixed<ui16>(out, VALID_MAGIC); // 2
    bytes += WriteZeros(out, 1); // 3 (reserved)
    bytes += WriteFixed<ui8>(out, header.Version); // 4
    bytes += WriteFixed<ui32>(out, header.NumId); // 8
    bytes += WriteFixed<ui32>(out, header.ProducerId); // 12
    bytes += WriteFixed<ui64>(out, header.ProducerSeqNo); // 20
    bytes += WriteFixed<ui32>(out, header.MetricsCount); // 24
    bytes += WriteFixed<ui32>(out, header.PointsCount); // 28
    bytes += WriteFixed<ui8>(out, NMonitoring::EncodeCompression(header.CompressionAlg)); // 29
    bytes += WriteFixed<ui8>(out, static_cast<ui8>(header.DecimPolicy)); // 30
    bytes += WriteZeros(out, 2); // 32 (reserved)
    return bytes;
}

} // namespace NSolomon::NSlog::NResolvedMeta
