#include "log_data.h"
#include "slog.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H

#include "slog_encoding_state.h"

#include <library/cpp/monlib/encode/buffered/buffered_encoder_base.h>
#include <library/cpp/monlib/metrics/labels.h>

#ifndef _little_endian_
#error Unsupported platform
#endif


using NMonitoring::ECompression;
using NMonitoring::EMetricValueType;
using NMonitoring::ETimePrecision;
using NMonitoring::EValueType;
using NMonitoring::EncodeMetricType;
using NMonitoring::IMetricConsumer;
using NMonitoring::IMetricEncoder;
using NMonitoring::IMetricEncoderPtr;
using NMonitoring::TBucketBound;
using NMonitoring::TBucketValue;

namespace NSolomon::NSlog {
namespace {

class TSlogEncoder: public ILogWriter {
public:
    using TEncoderState = NMonitoring::TEncoderStateImpl<ESlogEncodingState>;

    TSlogEncoder(
            ui32 numId,
            ETimePrecision timePrecision,
            ECompression compressionAlg,
            IOutputStream* dataOut,
            IOutputStream* metaOut)
        : DataOut_{dataOut}
        , MetaOut_{metaOut}
        , LogDataBuilder_{CreateLogDataBuilder(numId, EDataCodingScheme::Slog, timePrecision, compressionAlg, DataOut_)}
        , LogMetaBuilder_{NUnresolvedMeta::CreateUnresolvedMetaBuilder(numId, compressionAlg, MetaOut_)}
    {}

    ~TSlogEncoder() override {
        Close();
    }

private:
    void Close() override {
        if (IsClosed_) {
            return;
        }

        IsClosed_ = true;

        LogDataBuilder_->Close();
        LogMetaBuilder_->Close();
    }

    void OnStreamBegin() override {
        State_.Expect(TEncoderState::EState::ROOT);
    }

    void OnStreamEnd() override {
        State_.Expect(TEncoderState::EState::ROOT);
    }

    void OnStep(TDuration step) override {
        LogDataBuilder_->OnStep(step);
    }

    void OnCommonTime(TInstant time) override {
        LogDataBuilder_->OnCommonTime(time);
    }

    void OnMetricBegin(NMonitoring::EMetricType kind) override {
        State_.Switch(TEncoderState::EState::ROOT, TEncoderState::EState::METRIC);
        if (auto type = FromMonlibType(kind)) {
            CurrentMetric_.Kind = *type;
        } else {
            ythrow TEncodeError{} << "metric type is set to Unknown";
        }
    }

    void OnMetricEnd() override {
        if (State_ == TEncoderState::EState::TIMESERIES || State_ == TEncoderState::EState::POINT) {
            State_ = TEncoderState::EState::ROOT;
        } else {
            State_.ThrowInvalid("expected TIMESERIES or POINT");
        }

        ResetState();
    }

    void OnLabelsBegin() override {
        if (State_ == TEncoderState::EState::METRIC) {
            State_ = TEncoderState::EState::METRIC_LABELS;
        } else if (State_ == TEncoderState::EState::ROOT) {
            State_ = TEncoderState::EState::COMMON_LABELS;
        } else {
            State_.ThrowInvalid("expected METRIC or ROOT");
        }
    }

    void OnLabelsEnd() override {
        if (State_ == TEncoderState::EState::COMMON_LABELS) {
            State_ = TEncoderState::EState::ROOT;
            LogMetaBuilder_->OnCommonLabels(std::move(CommonLabels_));
            return;
        } else if (State_ != TEncoderState::EState::METRIC_LABELS) {
            State_.ThrowInvalid("expected METRIC_LABELS or COMMON_LABELS");
        }

        State_.Switch(TEncoderState::EState::METRIC_LABELS, TEncoderState::EState::METRIC);
    }

    void OnLabel(TStringBuf name, TStringBuf value) override {
        if (State_ == TEncoderState::EState::METRIC_LABELS) {
            CurrentMetric_.Labels.Add(name, value);
        } else if (State_ == TEncoderState::EState::COMMON_LABELS) {
            CommonLabels_.Add(name, value);
        } else {
            State_.ThrowInvalid("expected LABELS or COMMON_LABELS");
        }
    }

    void OnPoint(ELogFlagsComb flags, TAggrPointWithType&& point) override {
        State_.Switch(TEncoderState::EState::METRIC, TEncoderState::EState::POINT);

        TBytes bytes = LogDataBuilder_->OnPoint(CurrentMetric_.Kind, flags, std::move(point));
        auto pointsCnt = 1;
        LogMetaBuilder_->OnMetric(CurrentMetric_.Kind, std::move(CurrentMetric_.Labels), pointsCnt, bytes);
    }

    void OnTimeSeries(ELogFlagsComb flags, const TAggrTimeSeries& timeseries) override {
        State_.Switch(TEncoderState::EState::METRIC, TEncoderState::EState::TIMESERIES);

        ui64 tsSize = timeseries.Size();
        ui64 bytes = LogDataBuilder_->OnTimeSeries(CurrentMetric_.Kind, flags, timeseries);
        LogMetaBuilder_->OnMetric(CurrentMetric_.Kind, std::move(CurrentMetric_.Labels), tsSize, bytes);
    }

private:
    void ResetState() {
        CurrentMetric_.Kind = NTsModel::EPointType::DGauge;
        CurrentMetric_.Labels.clear();
    }

private:
    IOutputStream* DataOut_;
    IOutputStream* MetaOut_;

    ILogDataBuilderPtr LogDataBuilder_;
    NUnresolvedMeta::IUnresolvedMetaBuilderPtr LogMetaBuilder_;

    NMonitoring::TLabels CommonLabels_;

    struct TMetricState {
        NMonitoring::TLabels Labels;
        NTsModel::EPointType Kind;
    };

    TMetricState CurrentMetric_;
    bool IsClosed_{false};
    TEncoderState State_;
};

} // namespace

ILogWriterPtr CreateSlogEncoder(
        ui32 numId,
        NMonitoring::ETimePrecision timePrecision,
        NMonitoring::ECompression compressionAlg,
        IOutputStream* dataOut,
        IOutputStream* metaOut)
{
    return MakeHolder<TSlogEncoder>(numId, timePrecision, compressionAlg, dataOut, metaOut);
}

} // namespace NSolomon::NSlog
