#include "iterator.h"
#include "header.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/ts_codec/bits.h>

#include <library/cpp/monlib/encode/spack/compression.h>

using namespace NMonitoring;

namespace NSolomon::NSlog::NSnapshotData {

class TSnapshotLogDataIterator: public ISnapshotLogDataIterator {
public:
    TSnapshotLogDataIterator(IInputStream* in, NSnapshotData::THeader header)
        : In_{in}
        , DataHeader_{header}
    {
        auto compression = NMonitoring::DecodeCompression(DataHeader_.CompressionAlg);
        if (compression == ECompression::UNKNOWN) {
            ythrow TDecodeError()
                << "data compression algorithm is set to 'UNKNOWN'";
        }

        CompressedIn_ = CompressedInput(In_, compression);
        if (CompressedIn_) {
            In_ = CompressedIn_.Get();
        }
    }

    ui32 NumId() override {
        return DataHeader_.NumId;
    }

    ui32 TotalMetricCount() override {
        return DataHeader_.MetricsCount;
    }

    ui32 TotalPointCount() override {
        return DataHeader_.PointsCount;
    }

    bool HasNext() override {
        return MetricIdx_ < DataHeader_.MetricsCount;
    }

    TSnapshotLogDataRecord Next() override {
        if (MetricIdx_ >= DataHeader_.MetricsCount) {
            ythrow yexception() << "no metrics left in data";
        }

        NTsModel::EPointType type = ReadPointType(ReadFixed<ui8>(In_));
        ui16 columnSetMask = ReadFixed<ui16>(In_);
        ui64 bitsSize = ReadFixed<ui64>(In_);

        NTs::TBitBuffer data;
        data.Resize(bitsSize);
        In_->Read(data.Data(), data.SizeBytes());

        ++MetricIdx_;
        return TSnapshotLogDataRecord{type, columnSetMask, std::move(data)};
    }

private:
    IInputStream* In_;
    THolder<IInputStream> CompressedIn_;

    NSnapshotData::THeader DataHeader_;
    size_t MetricIdx_{0};
};

ISnapshotLogDataIteratorPtr CreateSnapshotLogDataIterator(IInputStream* in) {
    return std::make_unique<TSnapshotLogDataIterator>(in, NSnapshotData::ReadHeader(in));
}

} // namespace NSolomon::NSlog::NSnapshotData
