#include <solomon/libs/cpp/slog/snapshot_data/builder.h>
#include <solomon/libs/cpp/slog/snapshot_data/iterator.h>
#include <solomon/libs/cpp/ts_model/points.h>
#include <solomon/libs/cpp/ts_model/point_traits.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>
#include <solomon/libs/cpp/ts_codec/columns.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/buffer.h>

using namespace NMonitoring;
using namespace NSolomon::NTsModel;
using namespace NSolomon::NTs;
using namespace NSolomon::NSlog;
using namespace NSolomon::NSlog::NSnapshotData;

template <typename TPoint>
TBytes Add(ISnapshotLogDataBuilder* builder, NSolomon::NTsModel::EPointType type, TColumnSet columnSet, std::vector<TPoint> points) {
    TBitBuffer content;
    TBitWriter writer{&content};
    NSolomon::NTsModel::TEncoder<TPoint> encoder(columnSet, &writer);
    for (auto& point: points) {
        encoder.EncodePoint(point);
    }
    encoder.Flush();
    return builder->OnTimeSeries(type, columnSet.Mask(), content, points.size());
}

template <typename TPoint>
void AssertDataEquals(TSnapshotLogDataRecord record, NSolomon::NTsModel::EPointType expectType, TColumnSet expectColumnSet, std::vector<TPoint> expectPoints) {
    ASSERT_EQ(record.Type, expectType);
    ASSERT_EQ(record.ColumnSetMask, expectColumnSet.Mask());

    NSolomon::NTsModel::TDecoder<TPoint> decoder(expectColumnSet, record.Encoded);
    TPoint point;
    for (size_t index = 0; index < expectPoints.size(); index++) {
        TPoint expectPoint = expectPoints[index];
        ASSERT_TRUE(decoder.NextPoint(&point));
        ASSERT_EQ(point, expectPoint);
    }
    ASSERT_FALSE(decoder.NextPoint(&point));
}

TEST(TSlogSnapshotDataTest, GaugeType) {
    auto expectType = NSolomon::NTsModel::EPointType::DGauge;
    auto expectColumnSet = TGaugePoint::AggrColumns;
    auto expectPoints = std::vector<TGaugePoint>{{
        Gauge("2020-01-01T00:00:00Z", 1, 0),
        Gauge("2020-01-01T00:00:10Z", 2, 0),
        Gauge("2020-01-01T00:00:20Z", 15, 0),
        Gauge("2020-01-01T00:00:25Z", 15, 0)
    }};

    Y_UNUSED(expectType);
    Y_UNUSED(expectColumnSet);
    Y_UNUSED(expectPoints);

    TBufferStream data;
    {
        auto builder = CreateSnapshotLogDataBuilder(1, ECompression::IDENTITY, &data);
        Add(builder.Get(), expectType, expectColumnSet, expectPoints);
        builder->Close();
    }

    auto it = CreateSnapshotLogDataIterator(&data);
    ASSERT_TRUE(it->HasNext());
    AssertDataEquals(it->Next(), expectType, expectColumnSet, expectPoints);
    ASSERT_FALSE(it->HasNext());
}

TEST(TSlogSnapshotDataTest, CounterType) {
    auto expectType = NSolomon::NTsModel::EPointType::Counter;
    auto expectColumnSet = TCounterPoint::AggrColumns;
    auto expectPoints = std::vector<TCounterPoint>{{
        Counter("2020-01-01T00:00:00Z", 1),
        Counter("2020-01-01T00:00:10Z", 2),
        Counter("2020-01-01T00:00:20Z", 15),
        Counter("2020-01-01T00:00:25Z", 15)
    }};

    TBufferStream data;
    {
        auto builder = CreateSnapshotLogDataBuilder(42, ECompression::IDENTITY, &data);
        Add(builder.Get(), expectType, expectColumnSet, expectPoints);
        builder->Close();
    }

    auto it = CreateSnapshotLogDataIterator(&data);
    ASSERT_TRUE(it->HasNext());
    AssertDataEquals(it->Next(), expectType, expectColumnSet, expectPoints);
    ASSERT_FALSE(it->HasNext());
}

TEST(TSlogSnapshotDataTest, ManyMetrics) {
    auto expectOneType = NSolomon::NTsModel::EPointType::DGauge;
    auto expectOneColumnSet = TGaugePoint::AggrColumns;
    auto expectOnePoints = std::vector<TGaugePoint>{{
        Gauge("2020-01-01T00:00:00Z", 1, 0),
        Gauge("2020-01-01T00:00:10Z", 2, 0),
        Gauge("2020-01-01T00:00:20Z", 15, 0),
        Gauge("2020-01-01T00:00:25Z", 15, 0)
    }};

    auto expectTwoType = NSolomon::NTsModel::EPointType::Counter;
    auto expectTwoColumnSet = TCounterPoint::AggrColumns;
    auto expectTwoPoints = std::vector<TCounterPoint>{{
        Counter("2020-01-01T00:00:00Z", 1),
        Counter("2020-01-01T00:00:10Z", 2),
        Counter("2020-01-01T00:00:20Z", 15),
        Counter("2020-01-01T00:00:25Z", 15)
    }};

    TBufferStream data;
    {
        auto builder = CreateSnapshotLogDataBuilder(33, ECompression::ZLIB, &data);
        Add(builder.Get(), expectOneType, expectOneColumnSet, expectOnePoints);
        Add(builder.Get(), expectTwoType, expectTwoColumnSet, expectTwoPoints);
        builder->Close();
    }

    auto it = CreateSnapshotLogDataIterator(&data);
    ASSERT_TRUE(it->HasNext());
    AssertDataEquals(it->Next(), expectOneType, expectOneColumnSet, expectOnePoints);
    ASSERT_TRUE(it->HasNext());
    AssertDataEquals(it->Next(), expectTwoType, expectTwoColumnSet, expectTwoPoints);
    ASSERT_FALSE(it->HasNext());
}
