#include "builder.h"
#include "header.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H

#include <library/cpp/monlib/encode/spack/compression.h>

#include <util/generic/buffer.h>
#include <util/generic/size_literals.h>
#include <util/stream/buffer.h>

using namespace NMonitoring;

namespace NSolomon::NSlog::NUnresolvedMeta {

class TStringPoolEncoder {
public:
    explicit TStringPoolEncoder(ECompression alg)
        : Out_{&DataStream_}
    {
        if ((CompressedOut_ = CompressedOutput(Out_, alg))) {
            Out_ = CompressedOut_.Get();
        }
    }

    size_t WriteString(TStringBuf str) {
        auto it = Pool_.find(str);
        if (it != Pool_.end()) {
            return it->second;
        }

        size_t index = Pool_.size();
        Pool_.emplace(str, index);

        Out_->Write(str.data(), str.size());
        Out_->Write('\0');

        BytesSize_ += str.size() + sizeof('\0');

        return index;
    }

    void Close() {
        if (CompressedOut_) { // let's skip an empty frame to support concatenation of several compressed blocks
            dynamic_cast<NMonitoring::IFramedCompressStream*>(CompressedOut_.Get())->FlushWithoutEmptyFrame();
        }
    }

    size_t BytesSize() const {
        return BytesSize_;
    }

    const auto& Buffer() const {
        return DataStream_.Buffer();
    }

private:
    THashMap<TString, size_t> Pool_;
    TBufferStream DataStream_;
    IOutputStream* Out_{&DataStream_};
    THolder<IOutputStream> CompressedOut_;
    size_t BytesSize_{0};
};

class TUnresolvedMetaBuilder: public IUnresolvedMetaBuilder {
public:
    TUnresolvedMetaBuilder(ui32 numId, ECompression compressionAlg, IOutputStream* out)
        : Out_{out}
        , LabelNamesPool_{compressionAlg}
        , LabelValuesPool_{compressionAlg}
    {
        Header_.NumId = numId;
        Header_.CompressionAlg = compressionAlg;
        CompressedOut_ = CompressedOutput(TempOut_, Header_.CompressionAlg);
        if (CompressedOut_) {
            TempOut_ = CompressedOut_.Get();
        }
    }

    void Close() override {
        if (IsClosed_) {
            return;
        }

        IsClosed_ = true;

        LabelNamesPool_.Close();
        LabelValuesPool_.Close();

        Header_.LabelNamesPoolBytesSize = static_cast<ui32>(LabelNamesPool_.BytesSize());
        Header_.LabelValuesPoolBytesSize = static_cast<ui32>(LabelValuesPool_.BytesSize());
        NUnresolvedMeta::WriteHeader(Out_, Header_);

        Out_->Write(LabelNamesPool_.Buffer().Data(), LabelNamesPool_.Buffer().Size());
        Out_->Write(LabelValuesPool_.Buffer().Data(), LabelValuesPool_.Buffer().Size());

        TempOut_->Finish();

        // TODO: a place for an optimization. Pass an arg stream that supports position changing
        const auto& tempBuffer = TempData_.Buffer();
        Out_->Write(tempBuffer.Data(), tempBuffer.Size());
    }

    void OnCommonLabels(NMonitoring::TLabels&& commonLabels) override {
        Y_ENSURE(!AreCommonLabelsSet_, "common labels are already set");

        WriteLabels(std::move(commonLabels));
        AreCommonLabelsSet_ = true;
    }

    void OnMetric(NTsModel::EPointType kind, NMonitoring::TLabels&& labels, size_t pointsCnt, TBytes bytes) override {
        if (!AreCommonLabelsSet_) {
            AreCommonLabelsSet_ = true;
            WriteEmptyLabels();
        }

        Header_.MetricsCount++;
        Header_.PointsCount += pointsCnt;

        WriteFixed(TempOut_, PackTypes(kind, NMonitoring::EValueType::NONE));
        WriteLabels(std::move(labels));
        NMonitoring::WriteVarUInt32(TempOut_, pointsCnt);
        NMonitoring::WriteVarUInt32(TempOut_, bytes);
    }

private:
    void WriteEmptyLabels() {
        // labels size
        NMonitoring::WriteVarUInt32(TempOut_, 0);
    }

    void WriteLabels(NMonitoring::TLabels&& labels) {
        NMonitoring::WriteVarUInt32(TempOut_, labels.Size());

        for (auto&& label: labels) {
            NMonitoring::WriteVarUInt32(TempOut_, LabelNamesPool_.WriteString(label.Name()));
            NMonitoring::WriteVarUInt32(TempOut_, LabelValuesPool_.WriteString(label.Value()));
        }
    }

private:
    THeader Header_;
    TBufferStream TempData_;
    IOutputStream* TempOut_{&TempData_};
    THolder<IOutputStream> CompressedOut_;
    IOutputStream* Out_;
    bool IsClosed_{false};

    bool AreCommonLabelsSet_ = false;
    NMonitoring::TLabels CommonLabels_;
    TStringPoolEncoder LabelNamesPool_;
    TStringPoolEncoder LabelValuesPool_;
};

IUnresolvedMetaBuilderPtr CreateUnresolvedMetaBuilder(ui32 numId, ECompression compression, IOutputStream* out) {
    return MakeHolder<TUnresolvedMetaBuilder>(numId, compression, out);
}

} // namespace NSolomon::NSlog::NUnresolvedMeta
