#include "header.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H

#include <util/generic/size_literals.h>

namespace NSolomon::NSlog::NUnresolvedMeta {

inline void ReadMagic(IInputStream* in) {
    auto magic = ReadFixed<ui16>(in);
    if (magic != VALID_MAGIC) {
        ythrow TDecodeError{}
            << "invalid unresolved meta magic: expected " << VALID_MAGIC << ", "
            << "got " << magic;
    }
}

inline EVersion ReadVersion(IInputStream* in) {
    auto version = ReadFixed<ui8>(in);
    if (version < MIN_VERSION || version > MAX_VERSION) {
        ythrow TDecodeError()
            << "unsupported version of unresolved meta: " << version;
    }
    return static_cast<EVersion>(version);
}

inline ui32 ReadNumId(IInputStream* in) {
    auto numId = ReadFixed<ui32>(in);
    if (numId == 0) {
        ythrow TDecodeError()
            << "invalid numId of unresolved meta: " << numId;
    }
    return numId;
}

inline TBytes ReadLabelNamesPoolBytesSize(IInputStream* in) {
    auto size = ReadFixed<ui32>(in);
    if (size > 50_MB) {
        ythrow TDecodeError{}
            << "metadata label names pool is too big: expected " << 50_MB << " bytes or less, "
            << "got " << size;
    }
    return size;
}

inline TBytes ReadLabelValuesPoolBytesSize(IInputStream* in) {
    auto size = ReadFixed<ui32>(in);
    if (size > 50_MB) {
        ythrow TDecodeError{}
            << "metadata values pool is too big: expected " << 50_MB << " bytes or less, "
            << "got " << size;
    }
    return size;
}

inline NMonitoring::ECompression ReadCompression(IInputStream* in) {
    auto compression = ReadFixed<ui8>(in);
    auto result = NMonitoring::ECompression::UNKNOWN;

    if (!NMonitoring::TryDecodeCompression(compression, &result)) {
        ythrow TDecodeError{}
            << "unknown unresolved meta compression algorithm: " << compression;
    }

    if (result == NMonitoring::ECompression::UNKNOWN) {
        ythrow TDecodeError{}
            << "unresolved meta compression algorithm is set to 'UNKNOWN'";
    }

    return result;
}

THeader ReadHeader(IInputStream* in) {
    ReadMagic(in);
    THeader result;
    Skip(in, 1); // minor version not used anymore
    result.Version = ReadVersion(in);
    result.NumId = ReadNumId(in);
    result.LabelNamesPoolBytesSize = ReadLabelNamesPoolBytesSize(in);
    result.LabelValuesPoolBytesSize = ReadLabelValuesPoolBytesSize(in);
    result.MetricsCount = ReadFixed<ui32>(in);
    result.PointsCount = ReadFixed<ui32>(in);
    result.CompressionAlg = ReadCompression(in);
    Skip(in, 3);
    return result;
}

TBytes WriteHeader(IOutputStream* out, THeader header) {
    TBytes bytes = 0;
    bytes += WriteFixed<ui16>(out, VALID_MAGIC); // 2
    bytes += WriteZeros(out, 1); // 3 (reserved)
    bytes += WriteFixed<ui8>(out, header.Version); // 4
    bytes += WriteFixed<ui32>(out, header.NumId); // 8
    bytes += WriteFixed<ui32>(out, header.LabelNamesPoolBytesSize); // 12
    bytes += WriteFixed<ui32>(out, header.LabelValuesPoolBytesSize); // 16
    bytes += WriteFixed<ui32>(out, header.MetricsCount); // 20
    bytes += WriteFixed<ui32>(out, header.PointsCount); // 24
    bytes += WriteFixed<ui8>(out, NMonitoring::EncodeCompression(header.CompressionAlg)); // 25
    bytes += WriteZeros(out, 3); // 28 (reserved)
    return bytes;
}

} // namespace NSolomon::NSlog::NUnresolvedMeta
