#include "iterator.h"
#include "header.h"

#define INCLUDE_CODEC_IMPL_H
#include <solomon/libs/cpp/slog/codec/codec_impl.h>
#undef INCLUDE_CODEC_IMPL_H

#include <library/cpp/monlib/encode/spack/compression.h>
#include <library/cpp/monlib/encode/buffered/string_pool.h>
#include <util/stream/mem.h>
#include <util/generic/algorithm.h>

#include <memory>

using namespace NMonitoring;

namespace NPrivate {

}
namespace NSolomon::NSlog::NUnresolvedMeta {

class TUnresolvedMetaIterator: public IUnresolvedMetaIterator {
public:
    explicit TUnresolvedMetaIterator(IInputStream* in)
        : Header_{NUnresolvedMeta::ReadHeader(in)}
    {
        // TODO: not working with compression from java, when hex represent data the same
        auto compressedIn = NMonitoring::CompressedInput(in, Header_.CompressionAlg);
        if (compressedIn) {
            Meta_ = compressedIn->ReadAll();
        } else {
            Meta_ = in->ReadAll();
        }
        Memory_ = TMemoryInput{Meta_};
        Init();
    }

    explicit TUnresolvedMetaIterator(TString meta) {
        Memory_ = TMemoryInput(meta);
        Header_ = NUnresolvedMeta::ReadHeader(&Memory_);
        auto compressedIn = NMonitoring::CompressedInput(&Memory_, Header_.CompressionAlg);
        if (compressedIn) {
            Meta_ = compressedIn->ReadAll();
            Memory_ = TMemoryInput{Meta_};
        } else {
            Meta_ = std::move(meta);
        }
        Init();
    }

    ui32 NumId() override {
        return Header_.NumId;
    }

    ui32 TotalMetricCount() override {
        return Header_.MetricsCount;
    }

    ui32 TotalPointCount() override {
        return Header_.PointsCount;
    }

    bool HasNext() override {
        return MetricIdx_ < Header_.MetricsCount;
    }

    TUnresolvedMetaRecord Next() override {
        Y_VERIFY(HasNext());

        auto type = ReadPointType(ReadFixed<ui8>(&Memory_));

        Labels_.clear();
        ReadLabels(Labels_);
        if (!CommonLabels_.empty()) {
            Labels_.insert(Labels_.end(), CommonLabels_.begin(), CommonLabels_.end());
        }

        ui32 pointsCount = ReadVariadic(&Memory_);
        ui32 dataSize = ReadVariadic(&Memory_);
        ++MetricIdx_;
        return {type, NLabels::TLabels::OwnedStorage(Labels_.begin(), Labels_.end()), pointsCount, dataSize};
    }

    size_t SizeBytes() const override {
        size_t size =
                sizeof(TUnresolvedMetaIterator)
                + Meta_.capacity()
                + LabelNames_.SizeBytes()
                + LabelValues_.SizeBytes()
                + CommonLabels_.capacity() * sizeof(std::pair<const char*, const char*>)
                + Labels_.capacity() * sizeof(std::pair<const char*, const char*>);

        return size;
    }

private:
    void Init() {
        LabelNames_ = ReadStringPool(Header_.LabelNamesPoolBytesSize);
        LabelValues_ = ReadStringPool(Header_.LabelValuesPoolBytesSize);
        ReadLabels(CommonLabels_);
    }

    TStringPool ReadStringPool(ui32 bytes) {
        auto namesBuffer = Memory_.Buf();
        Skip(&Memory_, bytes);
        return TStringPool{namesBuffer, bytes};
    }

    void ReadLabels(TVector<std::pair<const char*, const char*>>& out) {
        size_t size = ReadVariadic(&Memory_);
        out.reserve(out.size() + size);

        for (size_t i = 0; i < size; ++i) {
            ui32 nameIdx = ReadVariadic(&Memory_);
            if (nameIdx >= LabelNames_.Size()) {
                ythrow TDecodeError{}
                    << "invalid index for label names pool: " << nameIdx;
            }

            ui32 valueIdx = ReadVariadic(&Memory_);
            if (valueIdx >= LabelValues_.Size()) {
                ythrow TDecodeError{}
                    << "invalid index for label values pool: " << valueIdx;
            }

            out.emplace_back(LabelNames_.Get(nameIdx).data(), LabelValues_.Get(valueIdx).data());
        }
    }

private:
    THeader Header_;
    TString Meta_;
    TMemoryInput Memory_;
    TStringPool LabelNames_{nullptr, 0};
    TStringPool LabelValues_{nullptr, 0};
    TVector<std::pair<const char*, const char*>> CommonLabels_;

    size_t MetricIdx_{0};
    TVector<std::pair<const char*, const char*>> Labels_;
};

IUnresolvedMetaIteratorPtr CreateUnresolvedMetaIterator(IInputStream* in) {
    return std::make_unique<TUnresolvedMetaIterator>(in);
}

IUnresolvedMetaIteratorPtr CreateUnresolvedMetaIterator(TString meta) {
    return std::make_unique<TUnresolvedMetaIterator>(std::move(meta));
}

} // namespace NSolomon::NSlog::NUnresolvedMeta
