#include <solomon/libs/cpp/slog/unresolved_meta/builder.h>
#include <solomon/libs/cpp/slog/unresolved_meta/iterator.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/buffer.h>

using namespace NMonitoring;
using namespace NSolomon::NSlog::NUnresolvedMeta;

TLabels MonLabels(std::initializer_list<TLabel> labels) {
    TLabels result(labels);
    result.SortByName();
    return result;
}

NSolomon::NLabels::TLabels Labels(std::initializer_list<NSolomon::NLabels::TLabels::TPair> labels) {
    return NSolomon::NLabels::TLabels::OwnedStorage(labels.begin(), labels.end());
}

inline void PrintTo(EMetricType type, std::ostream* stream) {
    *stream << MetricTypeToStr(type);
}

inline void PrintTo(const TLabels& labels, std::ostream* stream) {
    TStringBuilder s;
    s << labels;
    stream->write(s.data(), s.size());
}

TEST(TSlogUnresolvedMetaTest, NoCommonLabels) {
    TBuffer buffer;
    {
        TBufferOutput out(buffer);
        auto builder = CreateUnresolvedMetaBuilder(1, ECompression::IDENTITY, &out);
        builder->OnCommonLabels({});
        auto labels = MonLabels({{"name", "alice"}, {"type", "human"}});
        builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, std::move(labels), 42, 128);
        builder->Close();
    }

    {
        TBufferInput in(buffer);
        auto it = CreateUnresolvedMetaIterator(&in);
        EXPECT_EQ(1u, it->NumId());
        EXPECT_EQ(1u, it->TotalMetricCount());
        EXPECT_EQ(42u, it->TotalPointCount());
        ASSERT_TRUE(it->HasNext());

        auto record = it->Next();
        EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
        EXPECT_EQ(42u, record.PointCounts);
        EXPECT_EQ(128u, record.DataSize);
        ASSERT_EQ(2u, record.Labels.Size());
        auto labels = Labels({{"name", "alice"}, {"type", "human"}});
        EXPECT_EQ(labels, record.Labels);
        ASSERT_FALSE(it->HasNext());
    }
}

TEST(TSlogUnresolvedMetaTest, CommonLabels) {
    TBuffer buffer;
    {
        TBufferOutput out(buffer);
        auto builder = CreateUnresolvedMetaBuilder(42, ECompression::IDENTITY, &out);
        builder->OnCommonLabels({{"host", "monitoring-test-000"}});
        auto labels = MonLabels({{"name", "cpu.usage"}, {"cpu", "1"}});
        builder->OnMetric(NSolomon::NTsModel::EPointType::IGauge, std::move(labels), 2000, 982);
        builder->Close();
    }

    {
        TBufferInput in(buffer);
        auto it = CreateUnresolvedMetaIterator(&in);
        EXPECT_EQ(42u, it->NumId());
        EXPECT_EQ(1u, it->TotalMetricCount());
        EXPECT_EQ(2000u, it->TotalPointCount());
        ASSERT_TRUE(it->HasNext());

        auto record = it->Next();
        EXPECT_EQ(NSolomon::NTsModel::EPointType::IGauge, record.Type);
        EXPECT_EQ(2000u, record.PointCounts);
        EXPECT_EQ(982u, record.DataSize);
        auto labels = Labels({{"name", "cpu.usage"}, {"cpu", "1"}, {"host", "monitoring-test-000"}});
        EXPECT_EQ(labels, record.Labels);
        ASSERT_FALSE(it->HasNext());
    }
}

TEST(TSlogUnresolvedMetaTest, OverrideCommonLabelsJava) {
    TBuffer buffer;
    {
        TBufferOutput out(buffer);
        auto builder = CreateUnresolvedMetaBuilder(42, ECompression::IDENTITY, &out);
        builder->OnCommonLabels({{"host", "prod"}});
        {
            auto labels = MonLabels({{"name", "alice"}, {"host", "test"}});
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, std::move(labels), 1, 2);
        }
        builder->Close();
    }

    {
        TBufferInput in(buffer);
        auto it = CreateUnresolvedMetaIterator(&in);
        EXPECT_EQ(42u, it->NumId());
        EXPECT_EQ(1u, it->TotalMetricCount());
        EXPECT_EQ(1u, it->TotalPointCount());
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(1u, record.PointCounts);
            EXPECT_EQ(2u, record.DataSize);
            auto labels = Labels({{"name", "alice"}, {"host", "test"}});
            EXPECT_EQ(labels, record.Labels);
        }
        ASSERT_FALSE(it->HasNext());
    }
}

TEST(TSlogUnresolvedMetaTest, OverrideCommonLabels) {
    TBuffer buffer;
    {
        TBufferOutput out(buffer);
        auto builder = CreateUnresolvedMetaBuilder(42, ECompression::IDENTITY, &out);
        builder->OnCommonLabels({{"host", "monitoring-test-000"}});
        {
            auto labels = MonLabels({{"name", "cpu.usage"}, {"cpu", "1"}});
            builder->OnMetric(NSolomon::NTsModel::EPointType::IGauge, std::move(labels), 2000, 1000);
        }
        {
            auto labels = MonLabels({{"name", "active_users"}, {"host", "cluster"}});
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, std::move(labels), 2, 64);
        }
        builder->Close();
    }

    {
        TBufferInput in(buffer);
        auto it = CreateUnresolvedMetaIterator(&in);
        EXPECT_EQ(42u, it->NumId());
        EXPECT_EQ(2u, it->TotalMetricCount());
        EXPECT_EQ(2002u, it->TotalPointCount());
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::IGauge, record.Type);
            EXPECT_EQ(2000u, record.PointCounts);
            EXPECT_EQ(1000u, record.DataSize);
            auto labels = Labels({{"name", "cpu.usage"}, {"cpu", "1"}, {"host", "monitoring-test-000"}});
            EXPECT_EQ(labels, record.Labels);
        }
        {
            ASSERT_TRUE(it->HasNext());

            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(2u, record.PointCounts);
            EXPECT_EQ(64u, record.DataSize);
            auto labels = Labels({{"name", "active_users"}, {"host", "cluster"}});
            EXPECT_EQ(labels, record.Labels);
        }
        ASSERT_FALSE(it->HasNext());
    }
}

TEST(TSlogUnresolvedMetaTest, MultipleMetrics) {
    TBuffer buffer;
    {
        TBufferOutput out(buffer);
        auto builder = CreateUnresolvedMetaBuilder(55, ECompression::IDENTITY, &out);
        builder->OnCommonLabels({{"host", "test-000"}});
        {
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, MonLabels({{"name", "cpu.usage"}, {"cpu", "1"}}), 1, 8);
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, MonLabels({{"name", "cpu.usage"}, {"cpu", "2"}}), 1, 7);
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, MonLabels({{"name", "cpu.usage"}, {"cpu", "3"}}), 1, 6);
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, MonLabels({{"name", "cpu.usage"}, {"cpu", "4"}}), 1, 5);
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, MonLabels({{"name", "cpu.usage"}, {"cpu", "5"}}), 1, 4);
        }
        builder->Close();
    }

    {
        TBufferInput in(buffer);
        auto it = CreateUnresolvedMetaIterator(&in);
        EXPECT_EQ(55u, it->NumId());
        EXPECT_EQ(5u, it->TotalMetricCount());
        EXPECT_EQ(5u, it->TotalPointCount());
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(1u, record.PointCounts);
            EXPECT_EQ(8u, record.DataSize);
            auto labels = Labels({{"host", "test-000"}, {"name", "cpu.usage"}, {"cpu", "1"}});
            EXPECT_EQ(labels, record.Labels);
        }
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(1u, record.PointCounts);
            EXPECT_EQ(7u, record.DataSize);
            auto labels = Labels({{"host", "test-000"}, {"name", "cpu.usage"}, {"cpu", "2"}});
            EXPECT_EQ(labels, record.Labels);
        }
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(1u, record.PointCounts);
            EXPECT_EQ(6u, record.DataSize);
            auto labels = Labels({{"host", "test-000"}, {"name", "cpu.usage"}, {"cpu", "3"}});
            EXPECT_EQ(labels, record.Labels);
        }
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(1u, record.PointCounts);
            EXPECT_EQ(5u, record.DataSize);
            auto labels = Labels({{"host", "test-000"}, {"name", "cpu.usage"}, {"cpu", "4"}});
            EXPECT_EQ(labels, record.Labels);
        }
        {
            ASSERT_TRUE(it->HasNext());
            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(1u, record.PointCounts);
            EXPECT_EQ(4u, record.DataSize);
            auto labels = Labels({{"host", "test-000"}, {"name", "cpu.usage"}, {"cpu", "5"}});
            EXPECT_EQ(labels, record.Labels);
        }
        ASSERT_FALSE(it->HasNext());
    }
}

TEST(TSlogUnresolvedMetaTest, Compression) {
    for (ECompression compression: {ECompression::IDENTITY, ECompression::ZLIB, ECompression::ZSTD, ECompression::LZ4}) {
        TBuffer buffer;
        {
            TBufferOutput out(buffer);
            auto builder = CreateUnresolvedMetaBuilder(1, compression, &out);
            builder->OnCommonLabels({});
            auto labels = MonLabels({{"name", "alice"}, {"type", "human"}});
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, std::move(labels), 42, 128);
            builder->Close();
        }
        {
            TBufferInput in(buffer);
            auto it = CreateUnresolvedMetaIterator(&in);
            EXPECT_EQ(1u, it->NumId());
            EXPECT_EQ(1u, it->TotalMetricCount());
            EXPECT_EQ(42u, it->TotalPointCount());
            ASSERT_TRUE(it->HasNext());

            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(42u, record.PointCounts);
            EXPECT_EQ(128u, record.DataSize);
            ASSERT_EQ(2u, record.Labels.Size());
            auto labels = Labels({{"name", "alice"}, {"type", "human"}});
            EXPECT_EQ(labels, record.Labels);
            ASSERT_FALSE(it->HasNext());
        }
    }
}

TEST(TSlogUnresolvedMetaTest, CompressionTString) {
    for (ECompression compression: {ECompression::IDENTITY, ECompression::ZLIB, ECompression::ZSTD, ECompression::LZ4}) {
        TString buffer;
        {
            TStringOutput out(buffer);
            auto builder = CreateUnresolvedMetaBuilder(1, compression, &out);
            builder->OnCommonLabels({});
            auto labels = MonLabels({{"name", "alice"}, {"type", "human"}});
            builder->OnMetric(NSolomon::NTsModel::EPointType::DGauge, std::move(labels), 42, 128);
            builder->Close();
        }
        {
            auto it = CreateUnresolvedMetaIterator(std::move(buffer));
            EXPECT_EQ(1u, it->NumId());
            EXPECT_EQ(1u, it->TotalMetricCount());
            EXPECT_EQ(42u, it->TotalPointCount());
            ASSERT_TRUE(it->HasNext());

            auto record = it->Next();
            EXPECT_EQ(NSolomon::NTsModel::EPointType::DGauge, record.Type);
            EXPECT_EQ(42u, record.PointCounts);
            EXPECT_EQ(128u, record.DataSize);
            ASSERT_EQ(2u, record.Labels.Size());
            auto labels = Labels({{"name", "alice"}, {"type", "human"}});
            EXPECT_EQ(labels, record.Labels);
            ASSERT_FALSE(it->HasNext());
        }
    }
}

