#include <solomon/libs/cpp/slog/log_data.h>
#include <solomon/libs/cpp/slog/log_index.h>
#include <solomon/libs/cpp/slog/slog.h>

#include <solomon/libs/cpp/log_writer/log_writer.h>
#include <solomon/libs/cpp/slog/unresolved_meta/iterator.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/buffer.h>
#include <util/stream/buffer.h>
#include <util/stream/file.h>
#include <util/string/hex.h>

using namespace NMonitoring;

namespace NSolomon::NSlog::NSlogDataTest {
namespace {

MATCHER_P2(BinaryEq, expected, size, std::string{"binary values "} + (negation ? "aren't equal" : "are equal")) {
    if (Y_UNLIKELY(::memcmp(arg, expected, size) != 0)) {
        *result_listener << "     actual: " << HexEncode(arg, size) << '\n';
        *result_listener << "   expected: " << HexEncode(expected, size);
        return false;
    }
    return true;
}

constexpr ui32 NUM_ID = 123;
const TInstant NOW = TInstant::ParseIso8601("2000-01-01T00:00:10Z");

// {1: 1, 2: 1, 4: 2, 8: 4, 16: 8, inf: 83}
IHistogramSnapshotPtr TestHistogram() {
    auto h = ExponentialHistogram(6, 2);
    for (i64 i = 1; i < 100; i++) {
        h->Collect(i);
    }
    return h->Snapshot();
}

TLogHistogramSnapshotPtr TestLogHistogram() {
    TVector<double> buckets{0.5, 0.25, 0.25, 0.5};
    return MakeIntrusive<TLogHistogramSnapshot>(1.5, 1u, 0, std::move(buckets));
}

ISummaryDoubleSnapshotPtr TestSummaryDouble() {
    return MakeIntrusive<NMonitoring::TSummaryDoubleSnapshot>(10.1, -0.45, 0.478, 0.3, 30u);
}

ui8 expectedDataHeader_v2_0[] = {
    0x4C, 0x44,                                     // magic "LD"                     (fixed ui16)
    // minor, major
    0x00, 0x02,                                     // version                        (fixed ui16)
    0x7B, 0x00, 0x00, 0x00,                         // NumId (123)                    (fixed ui32)
    0x00, 0xAC, 0xCF, 0x6A, 0xDC, 0x00, 0x00, 0x00, // common ts millis               (fixed ui64)
    0x00, 0x00, 0x00, 0x00,                         // step millis                    (fixed ui32)
    0x0C, 0x00, 0x00, 0x00,                         // metrics count                  (fixed ui32)
    0x10, 0x00, 0x00, 0x00,                         // points count                   (fixed ui32)
    0x00,                                           // time precision                 (fixed ui8)
    0x00,                                           // compression algorithm          (fixed ui8)
    0x00,                                           // coding scheme                  (fixed ui8)
    0x00,                                           // reserved                       (fixed ui8)
};

ui8 expectedDataHeader_Ts_v2_0[] = {
    0x4C, 0x44,                                     // magic "LD"                     (fixed ui16)
    // minor, major
    0x00, 0x02,                                     // version                        (fixed ui16)
    0x7B, 0x00, 0x00, 0x00,                         // NumId (123)                    (fixed ui32)
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // common ts millis               (fixed ui64)
    0x00, 0x00, 0x00, 0x00,                         // step millis                    (fixed ui32)
    0x02, 0x00, 0x00, 0x00,                         // metrics count                  (fixed ui32)
    0x14, 0x00, 0x00, 0x00,                         // points count                   (fixed ui32)
    0x00,                                           // time precision                 (fixed ui8)
    0x00,                                           // compression algorithm          (fixed ui8)
    0x01,                                           // coding scheme                  (fixed ui8)
    0x00,                                           // reserved                       (fixed ui8)
};

/////// DATA ///////
ui8 expectedMetric1[] = {
    0x09,                                           // packed type                    (fixed ui8)
    0x00,                                           // flags (None)                   (fixed ui8)
    0x0D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (ui64)
};

ui8 expectedMetric2[] = {
    0x0B,                   // packed type                    (fixed ui8)
    0x00,                   // flags (None)                   (fixed ui8)
    0x03,                   // timeseries size                (varuint32)
    0x8A, 0x43, 0x6D, 0x38, // ts in seconds                  (ui32)
    0x11, 0x00, 0x00, 0x00, // value                          (ui64)
    0x00, 0x00, 0x00, 0x00,
    0x8A, 0x43, 0x6D, 0x38, // ts in seconds                  (ui32)
    0x0A, 0x00, 0x00, 0x00, // value                          (ui64)
    0x00, 0x00, 0x00, 0x00,
    0x8A, 0x43, 0x6D, 0x38, // ts in seconds                  (ui32)
    0x0D, 0x00, 0x00, 0x00, // value                          (ui64)
    0x00, 0x00, 0x00, 0x00,
};

ui8 expectedMetric3[] = {
    0x07,                                           // packed type                    (fixed ui8)
    0x00,                                           // flags (None)                   (fixed ui8)
    0x02,                                           // timeseries size                (varuint32)
    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x9A, 0x99, 0x99, 0x99, 0x99, 0x19, 0x45, 0x40, // value                          (double)
    0x99, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x33, 0x33, 0x33, 0x33, 0x33, 0xF3, 0x4D, 0x40, // value                          (double)
};

ui8 expectedMetric4[] = {
    0x16,                                           // packed type                    (fixed ui8)
    0x00,                                           // flags (None)                   (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x06,                                           // histogram buckets count        (varint)
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x3f, // histogram bucket bounds        (array of doubles)
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x40,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x40,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x40,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0x7f,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // histogram bucket values
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

ui8 expectedMetric5[] = {
    0x12,                                           // packed type                    (fixed ui8)
    0x00,                                           // flags (None)                   (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x39, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (i64)
};

ui8 expectedMetric6[] = {
    0x12,                                           // packed type                    (fixed ui8)
    0x04,                                           // flags (Count)                  (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x39, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (i64)
    0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // count                          (fixed ui64)
};

ui8 expectedMetric7[] = {
    0x0E,                                           // packed type                    (fixed ui8)
    0x01,                                           // flags (Denom)                  (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x39, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (ui64)
    0xE8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // denom                          (fixed ui64)
};

ui8 expectedMetric8[] = {
    0x0E,                                           // packed type                    (fixed ui8)
    0x05,                                           // flags (Count | Denom)          (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x39, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (ui64)
    0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // count                          (fixed ui64)
    0xE8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // denom                          (fixed ui64)
};

ui8 expectedMetric9[] = {
    0x0F,                                           // packed type                    (fixed ui8)
    0x05,                                           // flags (Count | Denom)          (fixed ui8)
    0x02,                                           // timeseries size                (varuint32)

    0x8A, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x39, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (ui64)
    0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // count                          (fixed ui64)
    0xE8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // denom                          (fixed ui64)

    0x99, 0x43, 0x6D, 0x38,                         // ts in seconds                  (ui32)
    0x39, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (ui64)
    0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // count                          (fixed ui64)
    0xE8, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // denom                          (fixed ui64)
};

ui8 expectedMetric10[] = {
    0x09,                                           // packed type                    (fixed ui8)
    0x00,                                           // flags (None)                   (fixed ui8)
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // value                          (ui64)
};

ui8 expectedMetric11[] = {
    0x1e,                                           // types (DSUMMARY | ONE_WITH_TS)          (fixed ui8)
    0x00,                                           // flags                                   (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // time in seconds                         (fixed ui32)
    0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // count                                   (fixed ui64)
    0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x24, 0x40, // sum                                     (fixed double)
    0xcd, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xdc, 0xbf, // min                                     (fixed double)
    0x64, 0x3b, 0xdf, 0x4f, 0x8d, 0x97, 0xde, 0x3f, // max                                     (fixed double)
    0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0xd3, 0x3f, // last                                    (fixed double)
};

ui8 expectedMetric12[] = {
    0x26,                                           // types (LOGHIST | ONE_WITH_TS)           (fixed ui8)
    0x00,                                           // flags                                   (fixed ui8)
    0x8A, 0x43, 0x6D, 0x38,                         // time in seconds                         (fixed ui32)
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x3F, // base                                    (fixed double)
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // zerosCount                              (fixed ui64)
    0x00,                                           // startPower                              (fixed ui32)
    0x04,                                           // buckets count                           (variant)
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x3F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD0, 0x3F, // bucket values
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD0, 0x3F,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE0, 0x3F,
};

constexpr size_t expectedDataSize =
    Y_ARRAY_SIZE(expectedDataHeader_v2_0)
    + Y_ARRAY_SIZE(expectedMetric1)
    + Y_ARRAY_SIZE(expectedMetric2)
    + Y_ARRAY_SIZE(expectedMetric3)
    + Y_ARRAY_SIZE(expectedMetric4)
    + Y_ARRAY_SIZE(expectedMetric5)
    + Y_ARRAY_SIZE(expectedMetric6)
    + Y_ARRAY_SIZE(expectedMetric7)
    + Y_ARRAY_SIZE(expectedMetric8)
    + Y_ARRAY_SIZE(expectedMetric9)
    + Y_ARRAY_SIZE(expectedMetric10)
    + Y_ARRAY_SIZE(expectedMetric11)
    + Y_ARRAY_SIZE(expectedMetric12)
    ;

ui8 expectedTsMetric1[] = {
    0x04,                                           // packed type                    (fixed ui8)
    0x03, 0x02,                                     // columns                        (fixed ui16)
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // bits                           (fixed ui64)
    0x64, 0x61, 0x74, 0x61                          // data
};

ui8 expectedTsMetric2[] = {
    0x08,                                           // packed type                    (fixed ui8)
    0x05, 0x04,                                     // columns                        (fixed ui16)
    0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // bits                           (fixed ui64)
    0x6d, 0x6f, 0x72, 0x65,                         // data
    0x20,
    0x64, 0x61, 0x74, 0x61
};

constexpr size_t expectedTsDataSize =
    Y_ARRAY_SIZE(expectedDataHeader_Ts_v2_0)
    + Y_ARRAY_SIZE(expectedTsMetric1)
    + Y_ARRAY_SIZE(expectedTsMetric2)
    ;

/////// META ///////
ui8 expectedMetaHeader_v1_0[] = {
    0x55, 0x4D,             // magic "UM"                     (fixed ui16)
    // minor, major
    0x00, 0x01,             // version                        (fixed ui16)
    0x7B, 0x00, 0x00, 0x00, // NumId (123)                    (fixed ui32)
    0x1A, 0x00, 0x00, 0x00, // label names pool size          (fixed ui32)
    0xA2, 0x00, 0x00, 0x00, // label values pool size         (fixed ui32)
    0x0C, 0x00, 0x00, 0x00, // metrics count                  (fixed ui32)
    0x10, 0x00, 0x00, 0x00, // points count                   (fixed ui32)
    0x00,                   // compression algorithm          (fixed ui8)
    0x00, 0x00, 0x00        // reserved
};

ui8 expectedStringPools[] = {
    0x70, 0x72, 0x6f, 0x6a, 0x65, 0x63, 0x74, 0x00, // "project\0"
    0x6d, 0x65, 0x74, 0x72, 0x69, 0x63, 0x00,       // "metric\0"
    0x61, 0x64, 0x64, 0x69, 0x74, 0x69, 0x6f, 0x6e, // "additional\0"
    0x61, 0x6c, 0x0,
    0x73, 0x6f, 0x6c, 0x6f, 0x6d, 0x6f, 0x6e, 0x00, // "solomon\0"
    0x71, 0x32, 0x00,                               // "q2\0"
    0x71, 0x33, 0x00,                               // "q3\0"
    0x61, 0x6e, 0x73, 0x77, 0x65, 0x72, 0x00,       // "answer\0"
    0x72, 0x65, 0x73, 0x70, 0x6f, 0x6e, 0x73, 0x65, // "responseTimeMillis\0"
    0x54, 0x69, 0x6d, 0x65, 0x4d, 0x69, 0x6c, 0x6c,
    0x69, 0x73, 0x00,
    0x62, 0x79, 0x74, 0x65, 0x73, 0x00,             // "bytes\0"
    0x69, 0x67, 0x61, 0x75, 0x67, 0x65, 0x57, 0x69, // "igaugeWithCount\0"
    0x74, 0x68, 0x43, 0x6f, 0x75, 0x6e, 0x74, 0x00,
    0x72, 0x61, 0x74, 0x65, 0x57, 0x69, 0x74, 0x68, // "rateWithDenom\0"
    0x44, 0x65, 0x6e, 0x6f, 0x6d, 0x00,
    0x72, 0x61, 0x74, 0x65, 0x57, 0x69, 0x74, 0x68, // "rateWithDenomAndCount\0"
    0x44, 0x65, 0x6e, 0x6f, 0x6d, 0x41, 0x6e, 0x64,
    0x43, 0x6f, 0x75, 0x6e, 0x74, 0x00,
    0x72, 0x61, 0x74, 0x65, 0x53, 0x65, 0x72, 0x69, // "rateSeriesWithDenomAndCount\0"
    0x65, 0x73, 0x57, 0x69, 0x74, 0x68, 0x44, 0x65,
    0x6e, 0x6f, 0x6d, 0x41, 0x6e, 0x64, 0x43, 0x6f,
    0x75, 0x6e, 0x74, 0x00,
    0x77, 0x69, 0x74, 0x68, 0x43, 0x6f, 0x6d, 0x6d, // "withCommonTime\0"
    0x6f, 0x6e, 0x54, 0x69, 0x6d, 0x65, 0x00,
    0x6c, 0x61, 0x62, 0x65, 0x6c, 0x00,             // "label\0"
    0x74, 0x65, 0x6D, 0x70, 0x65, 0x72, 0x61, 0x74, // temperature
    0x75, 0x72, 0x65, 0x00,
    0x6d, 0x73, 0x00,                               // ms
};

ui8 expectedCommonLabels[] = {
    0x01, // common labels size (varuint32)
    0x00, // label name index   (varuint32)
    0x00, // label value index  (varuint32)
};

ui8 expectedMetricMeta1[] = {
    0x08,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x01,                          // label value index              (varuint32)
    0x01,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric1), // data size                      (varuint32)
};

ui8 expectedMetricMeta2[] = {
    0x08,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x02,                          // label value index              (varuint32)
    0x03,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric2), // data size                      (varuint32)
};

ui8 expectedMetricMeta3[] = {
    0x04,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x03,                          // label value index              (varuint32)
    0x02,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric3), // data size                      (varuint32)
};

ui8 expectedMetricMeta4[] = {
    0x14,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x04,                          // label value index              (varuint32)
    0x01,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric4), // data size                      (varuint32)
};

ui8 expectedMetricMeta5[] = {
    0x10,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x05,                          // label value index              (varuint32)
    0x01,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric5), // data size                      (varuint32)
};

ui8 expectedMetricMeta6[] = {
    0x10,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x06,                          // label value index              (varuint32)
    0x01,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric6), // data size                      (varuint32)
};

ui8 expectedMetricMeta7[] = {
    0x0C,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x07,                          // label value index              (varuint32)
    0x01,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric7), // data size                      (varuint32)
};

ui8 expectedMetricMeta8[] = {
    0x0C,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x08,                          // label value index              (varuint32)
    0x01,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric8), // data size                      (varuint32)
};

ui8 expectedMetricMeta9[] = {
    0x0C,                          // packed type                    (fixed ui8)
    0x01,                          // labels size                    (varuint32)
    0x01,                          // label name index               (varuint32)
    0x09,                          // label value index              (varuint32)
    0x02,                          // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric9), // data size                      (varuint32)
};

ui8 expectedMetricMeta10[] = {
    0x08,                           // packed type                    (fixed ui8)
    0x02,                           // labels size                    (varuint32)
    0x01,                           // label name index               (varuint32)
    0x0A,                           // label value index              (varuint32)
    0x02,                           // label name index               (varuint32)
    0x0B,                           // label value index              (varuint32)
    0x01,                           // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric10), // data size                      (varuint32)
};

ui8 expectedMetricMeta11[] = {
    0x1c,                           // packed type                    (fixed ui8)
    0x01,                           // labels size                    (varuint32)
    0x01,                           // label name index               (varuint32)
    0x0C,                           // label value index              (varuint32)
    0x01,                           // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric11), // data size                      (varuint32)
};

ui8 expectedMetricMeta12[] = {
    0x24,                           // packed type                    (fixed ui8)
    0x01,                           // labels size                    (varuint32)
    0x01,                           // label name index               (varuint32)
    0x0D,                           // label value index              (varuint32)
    0x01,                           // points count                   (varuint32)
    Y_ARRAY_SIZE(expectedMetric12), // data size                      (varuint32)
};

constexpr size_t expectedMetaSize =
    Y_ARRAY_SIZE(expectedMetaHeader_v1_0)
    + Y_ARRAY_SIZE(expectedStringPools)
    + Y_ARRAY_SIZE(expectedCommonLabels)
    + Y_ARRAY_SIZE(expectedMetricMeta1)
    + Y_ARRAY_SIZE(expectedMetricMeta2)
    + Y_ARRAY_SIZE(expectedMetricMeta3)
    + Y_ARRAY_SIZE(expectedMetricMeta4)
    + Y_ARRAY_SIZE(expectedMetricMeta5)
    + Y_ARRAY_SIZE(expectedMetricMeta6)
    + Y_ARRAY_SIZE(expectedMetricMeta7)
    + Y_ARRAY_SIZE(expectedMetricMeta8)
    + Y_ARRAY_SIZE(expectedMetricMeta9)
    + Y_ARRAY_SIZE(expectedMetricMeta10)
    + Y_ARRAY_SIZE(expectedMetricMeta11)
    + Y_ARRAY_SIZE(expectedMetricMeta12)
    ;

template <typename T>
static T ReadFixed(TBufferStream* buf) {
    T value{};
    buf->Read(&value, sizeof(value));
    return value;
}

TEST(TEncodeTest, NoMetrics) {
    ui32 numIdWritten = 123;
    TBufferStream dataBuf{NSlogData::HEADER_SIZE * 2};
    TBufferStream metaBuf{};

    auto slog = CreateSlogEncoder(numIdWritten, ETimePrecision::SECONDS, ECompression::IDENTITY, &dataBuf, &metaBuf);

    slog->OnStreamBegin();
    slog->OnStreamEnd();
    slog->Close();

    const auto& buffer = dataBuf.Buffer();
    ASSERT_EQ(buffer.Size(), NSlogData::HEADER_SIZE);

    ui16 magic = ReadFixed<ui16>(&dataBuf);
    ui16 version = ReadFixed<ui16>(&dataBuf);
    ui32 numIdRead = ReadFixed<ui32>(&dataBuf);
    ui64 commonTsMillis = ReadFixed<ui64>(&dataBuf);
    ui32 stepMillis = ReadFixed<ui32>(&dataBuf);
    ui32 metricsCount = ReadFixed<ui32>(&dataBuf);
    ui32 pointsCount = ReadFixed<ui32>(&dataBuf);
    ui8 timePrecision = ReadFixed<ui8>(&dataBuf);
    ui8 compressionAlg = ReadFixed<ui8>(&dataBuf);
    ui8 codingScheme = ReadFixed<ui8>(&dataBuf);
    ui8 zero = ReadFixed<ui8>(&dataBuf);

    ASSERT_EQ(magic, NSlogData::VALID_MAGIC);
    ASSERT_EQ(version, NSlogData::CURRENT_VERSION);
    ASSERT_EQ(numIdRead, numIdWritten);
    ASSERT_EQ(commonTsMillis, 0u);
    ASSERT_EQ(stepMillis, 0u);
    ASSERT_EQ(metricsCount, 0u);
    ASSERT_EQ(pointsCount, 0u);
    ASSERT_EQ(timePrecision, static_cast<ui8>(ETimePrecision::SECONDS));
    ASSERT_EQ(compressionAlg, EncodeCompression(ECompression::IDENTITY));
    ASSERT_EQ(codingScheme, static_cast<ui8>(EDataCodingScheme::Slog));
    ASSERT_EQ(zero, 0u);
}

TEST(TEncodeTest, MetricWithNoPoints) {
    TBufferStream dataOut;
    TBufferStream metaOut;

    auto e = CreateSlogEncoder(NUM_ID, ETimePrecision::SECONDS, ECompression::IDENTITY, &dataOut, &metaOut);
    e->OnStreamBegin();

    {
        e->OnMetricBegin(EMetricType::GAUGE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "WithNoPoints");
            e->OnLabelsEnd();
        }
        ASSERT_THROW(e->OnMetricEnd(), yexception);
    }
}

TEST(TEncodeTest, General) {
    TBufferStream dataOut;
    TBufferStream metaOut;

    auto e = CreateSlogEncoder(NUM_ID, ETimePrecision::SECONDS, ECompression::IDENTITY, &dataOut, &metaOut);
    e->OnStreamBegin();

    auto commonTime = TInstant::ParseIso8601("2000-01-01T00:00:00Z");
    e->OnCommonTime(commonTime);

    { // Common Labels
        e->OnLabelsBegin();
        e->OnLabel("project", "solomon");
        e->OnLabelsEnd();
    }

    { // metric #1
        e->OnMetricBegin(EMetricType::COUNTER);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "q2");
            e->OnLabelsEnd();
        }

        e->OnPoint(static_cast<ELogFlagsComb>(ELogFlags::None), {TInstant::Zero(), ui64(13)});
        e->OnMetricEnd();
    }
    { // metric #2
        e->OnMetricBegin(EMetricType::COUNTER);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "q3");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.Add(NOW, ui64(17));
        s.Add(NOW, ui64(10));
        s.Add(NOW, ui64(13));

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #3
        e->OnMetricBegin(EMetricType::GAUGE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "answer");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.Add(NOW, 42.2);
        s.Add(NOW + TDuration::Seconds(15), 59.9);

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #4
        e->OnMetricBegin(EMetricType::HIST);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "responseTimeMillis");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        auto histogram = TestHistogram();
        s.Add(NOW, histogram.Get());

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #5
        e->OnMetricBegin(EMetricType::IGAUGE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "bytes");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.Add(NOW, i64(1337));

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #6
        e->OnMetricBegin(EMetricType::IGAUGE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "igaugeWithCount");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.AddWithCount(NOW, i64(1337), 12);

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::Count);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #7
        e->OnMetricBegin(EMetricType::RATE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "rateWithDenom");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.AddWithDenom(NOW, ui64(1337), 1000);

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::Denom);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #8
        e->OnMetricBegin(EMetricType::RATE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "rateWithDenomAndCount");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.Add(NOW, ui64(1337), 1000, 12);

        auto flags = CombineLogFlags(ELogFlags::Denom, ELogFlags::Count);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #9
        e->OnMetricBegin(EMetricType::RATE);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "rateSeriesWithDenomAndCount");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        s.Add(NOW, ui64(1337), 1000, 12);
        s.Add(NOW + TDuration::Seconds(15), ui64(1337), 1000, 12);

        auto flags = CombineLogFlags(ELogFlags::Denom, ELogFlags::Count);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #10
        e->OnMetricBegin(EMetricType::COUNTER);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "withCommonTime");
            e->OnLabel("additional", "label");
            e->OnLabelsEnd();
        }

        e->OnPoint(static_cast<ELogFlagsComb>(ELogFlags::None), {commonTime, ui64(1)});

        e->OnMetricEnd();
    }
    { // metric #11
        e->OnMetricBegin(EMetricType::DSUMMARY);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "temperature");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        auto summary = TestSummaryDouble();
        s.Add(NOW, summary.Get());

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        e->OnTimeSeries(flags, s);

        e->OnMetricEnd();
    }
    { // metric #12
        e->OnMetricBegin(EMetricType::LOGHIST);
        {
            e->OnLabelsBegin();
            e->OnLabel("metric", "ms");
            e->OnLabelsEnd();
        }

        TAggrTimeSeries s;
        auto logHist = TestLogHistogram();
        s.Add(NOW, logHist.Get());

        auto flags = static_cast<ELogFlagsComb>(ELogFlags::None);
        e->OnTimeSeries(flags, std::move(s));

        e->OnMetricEnd();
    }

    e->OnStreamEnd();
    e->Close();

    { // data
        auto dataBuf = dataOut.Buffer();
        ASSERT_EQ(dataBuf.Size(), expectedDataSize);

        ui8* p = reinterpret_cast<ui8*>(dataBuf.Data());

        ASSERT_THAT(p, BinaryEq(expectedDataHeader_v2_0, std::size(expectedDataHeader_v2_0)));
        p += Y_ARRAY_SIZE(expectedDataHeader_v2_0);

        ASSERT_THAT(p, BinaryEq(expectedMetric1, std::size(expectedMetric1)));
        p += Y_ARRAY_SIZE(expectedMetric1);

        ASSERT_THAT(p, BinaryEq(expectedMetric2, std::size(expectedMetric2)));
        p += Y_ARRAY_SIZE(expectedMetric2);

        ASSERT_THAT(p, BinaryEq(expectedMetric3, std::size(expectedMetric3)));
        p += Y_ARRAY_SIZE(expectedMetric3);

        ASSERT_THAT(p, BinaryEq(expectedMetric4, std::size(expectedMetric4)));
        p += Y_ARRAY_SIZE(expectedMetric4);

        ASSERT_THAT(p, BinaryEq(expectedMetric5, std::size(expectedMetric5)));
        p += Y_ARRAY_SIZE(expectedMetric5);

        ASSERT_THAT(p, BinaryEq(expectedMetric6, std::size(expectedMetric6)));
        p += Y_ARRAY_SIZE(expectedMetric6);

        ASSERT_THAT(p, BinaryEq(expectedMetric7, std::size(expectedMetric7)));
        p += Y_ARRAY_SIZE(expectedMetric7);

        ASSERT_THAT(p, BinaryEq(expectedMetric8, std::size(expectedMetric8)));
        p += Y_ARRAY_SIZE(expectedMetric8);

        ASSERT_THAT(p, BinaryEq(expectedMetric9, std::size(expectedMetric9)));
        p += Y_ARRAY_SIZE(expectedMetric9);

        ASSERT_THAT(p, BinaryEq(expectedMetric10, std::size(expectedMetric10)));
        p += Y_ARRAY_SIZE(expectedMetric10);

        ASSERT_THAT(p, BinaryEq(expectedMetric11, std::size(expectedMetric11)));
        p += Y_ARRAY_SIZE(expectedMetric11);

        ASSERT_THAT(p, BinaryEq(expectedMetric12, std::size(expectedMetric12)));
        p += Y_ARRAY_SIZE(expectedMetric12);
    }

    { // metadata
        auto metaBuf = metaOut.Buffer();
        ASSERT_EQ(metaBuf.Size(), expectedMetaSize);

        ui8* mp = reinterpret_cast<ui8*>(metaBuf.Data());

        ASSERT_THAT(mp, BinaryEq(expectedMetaHeader_v1_0, std::size(expectedMetaHeader_v1_0)));
        mp += Y_ARRAY_SIZE(expectedMetaHeader_v1_0);

        ASSERT_THAT(mp, BinaryEq(expectedStringPools, std::size(expectedStringPools)));
        mp += Y_ARRAY_SIZE(expectedStringPools);

        ASSERT_THAT(mp, BinaryEq(expectedCommonLabels, std::size(expectedCommonLabels)));
        mp += Y_ARRAY_SIZE(expectedCommonLabels);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta1, std::size(expectedMetricMeta1)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta1);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta2, std::size(expectedMetricMeta2)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta2);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta3, std::size(expectedMetricMeta3)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta3);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta4, std::size(expectedMetricMeta4)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta4);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta5, std::size(expectedMetricMeta5)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta5);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta6, std::size(expectedMetricMeta6)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta6);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta7, std::size(expectedMetricMeta7)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta7);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta8, std::size(expectedMetricMeta8)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta8);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta9, std::size(expectedMetricMeta9)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta9);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta10, std::size(expectedMetricMeta10)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta10);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta11, std::size(expectedMetricMeta11)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta11);

        ASSERT_THAT(mp, BinaryEq(expectedMetricMeta12, std::size(expectedMetricMeta12)));
        mp += Y_ARRAY_SIZE(expectedMetricMeta12);
    }
}

TEST(TEncodeTest, TsModel) {
    TBufferStream dataOut;

    auto encoder = CreateLogDataBuilder(
        123,
        EDataCodingScheme::TsModel,
        NMonitoring::ETimePrecision::SECONDS,
        NMonitoring::ECompression::IDENTITY,
        &dataOut);

    ASSERT_THROW_MESSAGE_HAS_SUBSTR(
        std::invoke([&]() { encoder->OnPoint(NTsModel::EPointType::DGauge, 0, {NOW, (double)0}); }),
        yexception,
        "OnPoint is not allowed in coding scheme TsModel");

    encoder->OnTimeSeriesNativeEncoded(NTsModel::EPointType::DGauge, 0x0203, NTs::TBitSpan{"data"}, 15);
    encoder->OnTimeSeriesNativeEncoded(NTsModel::EPointType::Counter, 0x0405, NTs::TBitSpan{"more data"}, 5);
    encoder->Close();

    { // data
        auto dataBuf = dataOut.Buffer();
        ASSERT_EQ(dataBuf.Size(), expectedTsDataSize);

        ui8* p = reinterpret_cast<ui8*>(dataBuf.Data());

        ASSERT_THAT(p, BinaryEq(expectedDataHeader_Ts_v2_0, std::size(expectedDataHeader_Ts_v2_0)));
        p += Y_ARRAY_SIZE(expectedDataHeader_Ts_v2_0);

        ASSERT_THAT(p, BinaryEq(expectedTsMetric1, std::size(expectedTsMetric1)));
        p += Y_ARRAY_SIZE(expectedTsMetric1);

        ASSERT_THAT(p, BinaryEq(expectedTsMetric2, std::size(expectedTsMetric2)));
        p += Y_ARRAY_SIZE(expectedTsMetric2);
    }
}

TEST(TDecodeTest, Iterators) {
    TBufferStream dataOut;
    TBufferStream metaOut;

    auto e = CreateSlogEncoder(NUM_ID, ETimePrecision::SECONDS, ECompression::IDENTITY, &dataOut, &metaOut);
    auto step = TDuration::Seconds(15);

    e->OnStreamBegin();
    e->OnCommonTime(NOW);

    e->OnStep(step);

    e->OnMetricBegin(EMetricType::RATE);
    e->OnLabelsBegin();
    e->OnLabel("test", "metric");
    e->OnLabelsEnd();

    auto ts = NOW + TDuration::Seconds(123);
    ui64 value = 123;
    ui64 count = 0;
    ui64 denom = 0;
    e->OnPoint(static_cast<ELogFlagsComb>(ELogFlags::None), {ts, value, 0, 0});

    e->OnMetricEnd();

    e->OnStreamEnd();
    e->Close();

    { // data
        auto dataIt = CreateLogDataIterator(&dataOut);
        ASSERT_TRUE(dataIt->HasNext());
        ASSERT_TRUE(dataIt->HasNext());

        auto record = dataIt->Next();
        ASSERT_EQ(record.Kind, NTsModel::EPointType::Rate);
        ASSERT_EQ(record.NumId, NUM_ID);
        ASSERT_EQ(record.StepMillis, step.MilliSeconds());

        ASSERT_EQ(record.TimeSeries.Size(), 1u);
        auto m = record.TimeSeries[0];
        ASSERT_EQ(m.GetTime(), ts);
        ASSERT_EQ(m.GetValue().AsUint64(), 123u);
        ASSERT_EQ(m.GetDenom(), denom);
        ASSERT_EQ(m.GetCount(), count);

        ASSERT_FALSE(dataIt->HasNext());
        ASSERT_THROW(dataIt->Next(), yexception);
    }

    { // meta
        auto metaIt = NUnresolvedMeta::CreateUnresolvedMetaIterator(&metaOut);
        ASSERT_TRUE(metaIt->HasNext());
        ASSERT_TRUE(metaIt->HasNext());

        auto record = metaIt->Next();
        ASSERT_EQ(record.Type, NTsModel::EPointType::Rate);
        ASSERT_EQ(record.PointCounts, 1u);
        ASSERT_EQ(record.DataSize, dataOut.Buffer().Size() - NSlogData::HEADER_SIZE);
        ASSERT_EQ(record.Labels.Size(), 1u);
        ASSERT_STREQ(record.Labels[0].first, "test");
        ASSERT_STREQ(record.Labels[0].second, "metric");

        ASSERT_FALSE(metaIt->HasNext());
    }
}

ui8 expectedIndexBinary[] = {
    0x4c, 0x49,             // magic "LI"         (fixed ui16)
    0x00, 0x01,             // version            (fixed ui16)
    0x03, 0x00, 0x00, 0x00, // size               (fixed ui32)

    0x01, 0x00, 0x00, 0x00, // numId              (fixed ui32)
    0x0A, 0x00, 0x00, 0x00, // meta size in bytes (fixed ui32)
    0x64, 0x00, 0x00, 0x00, // data size in bytes (fixed ui32)

    0x02, 0x00, 0x00, 0x00, // numId              (fixed ui32)
    0x14, 0x00, 0x00, 0x00, // meta size in bytes (fixed ui32)
    0xC8, 0x00, 0x00, 0x00, // data size in bytes (fixed ui32)

    0x03, 0x00, 0x00, 0x00, // numId              (fixed ui32)
    0x1E, 0x00, 0x00, 0x00, // meta size in bytes (fixed ui32)
    0x2C, 0x01, 0x00, 0x00, // data size in bytes (fixed ui32)
};

TSlogIndex ConstructIndex() {
    TSlogIndex index{3};

    index.Add(1, 10, 100);
    index.Add(2, 20, 200);
    index.Add(3, 30, 300);

    return index;
}

TEST(TIndexTest, General) {
    auto index = ConstructIndex();

    ASSERT_EQ(index.Size(), 3u);

    for (size_t i = 0; i != 3; ++i) {
        auto j = i + 1;

        ASSERT_EQ(index.GetNumId(i), j);
        ASSERT_EQ(index.GetMetaSizeBytes(i), j * 10);
        ASSERT_EQ(index.GetDataSizeBytes(i), j * 100);
    }
}

TEST(TIndexTest, Encode) {
    auto index = ConstructIndex();

    TBufferStream buf;
    EncodeIndex(&buf, index);

    ui8* p = reinterpret_cast<ui8*>(buf.Buffer().Data());
    ASSERT_THAT(p, BinaryEq(expectedIndexBinary, std::size(expectedIndexBinary)));
}

TEST(TIndexTest, Decode) {
    auto index = ConstructIndex();

    TBufferStream buf;
    EncodeIndex(&buf, index);
    auto decodedIndex = DecodeIndex(&buf);

    ASSERT_EQ(index.Size(), decodedIndex.Size());
    for (size_t i = 0; i != index.Size(); ++i) {
        ASSERT_EQ(index.GetNumId(i), decodedIndex.GetNumId(i));
        ASSERT_EQ(index.GetMetaSizeBytes(i), decodedIndex.GetMetaSizeBytes(i));
        ASSERT_EQ(index.GetDataSizeBytes(i), decodedIndex.GetDataSizeBytes(i));
    }
}

void TestCompression(ECompression alg) {
    TBufferStream dataStream;
    TBufferStream metaStream;

    double value = 42;

    {
        auto e = CreateSlogEncoder(NUM_ID, ETimePrecision::SECONDS, alg, &dataStream, &metaStream);
        e->OnStreamBegin();
        {
            auto commonTime = TInstant::Now();
            e->OnCommonTime(commonTime);
            e->OnMetricBegin(EMetricType::GAUGE);
            {
                e->OnLabelsBegin();
                e->OnLabel("sensor", "answer");
                e->OnLabelsEnd();
            }

            e->OnPoint(static_cast<ELogFlagsComb>(ELogFlags::None), {NOW, value});
            e->OnMetricEnd();
        }
        e->OnStreamEnd();
        e->Close();
    }

    auto dataIt = CreateLogDataIterator(&dataStream);
    auto metaIt = NUnresolvedMeta::CreateUnresolvedMetaIterator(&metaStream);

    { // data
        ASSERT_TRUE(dataIt->HasNext());

        auto record = dataIt->Next();
        ASSERT_EQ(record.Kind, NTsModel::EPointType::DGauge);
        ASSERT_EQ(record.NumId, NUM_ID);

        ASSERT_EQ(record.TimeSeries.Size(), 1u);
        auto m = record.TimeSeries[0];
        ASSERT_EQ(m.GetTime(), NOW);
        ASSERT_EQ(m.GetValue().AsDouble(), value);
        ASSERT_EQ(m.GetDenom(), 0u);
        ASSERT_EQ(m.GetCount(), 0u);

        ASSERT_FALSE(dataIt->HasNext());
    }

    { // meta
        ASSERT_TRUE(metaIt->HasNext());

        auto record = metaIt->Next();
        ASSERT_EQ(record.Type, NTsModel::EPointType::DGauge);
        ASSERT_EQ(record.PointCounts, 1u);
        ASSERT_EQ(record.Labels.Size(), 1u);
        ASSERT_STREQ(record.Labels[0].first, "sensor");
        ASSERT_STREQ(record.Labels[0].second, "answer");

        ASSERT_FALSE(metaIt->HasNext());
    }
}

TEST(TCompressionTest, Identity) {
    TestCompression(ECompression::IDENTITY);
}

TEST(TCompressionTest, Zlib) {
    TestCompression(ECompression::ZLIB);
}

TEST(TCompressionTest, Zstd) {
    TestCompression(ECompression::ZSTD);
}

TEST(TCompressionTest, Lz4) {
    TestCompression(ECompression::LZ4);
}

} // namespace
} // namespace NSolomon::NSlog
