#include "ru_yandex_solomon_SlogCodecNative.h"

#include "java_objects.h"
#include <solomon/libs/cpp/slog/slog.h>
#include <solomon/libs/cpp/slog/codec/fwd.h>
#include <solomon/libs/cpp/slog/log_data.h>
#include <solomon/libs/cpp/slog/snapshot_data/builder.h>
#include <solomon/libs/cpp/slog/snapshot_data/iterator.h>
#include <solomon/libs/cpp/slog/unresolved_meta/iterator.h>
#include <solomon/libs/cpp/slog/resolved_meta/builder.h>
#include <solomon/libs/cpp/slog/resolved_meta/iterator.h>
#include <solomon/libs/cpp/ts_model/points.h>
#include <solomon/libs/cpp/ts_model/point_traits.h>
#include <solomon/libs/cpp/ts_codec/base_ts_codec.h>

#include <util/generic/buffer.h>
#include <util/generic/yexception.h>
#include <util/stream/buffer.h>
#include <util/string/hex.h>

using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NTs;
using namespace NSolomon::NTsModel;

namespace {

TBuffer RepackMeta(const TBuffer& buffer) {
    try {
        TBufferInput in(buffer);
        auto it = NSlog::NUnresolvedMeta::CreateUnresolvedMetaIterator(&in);
        auto numId = it->NumId();

        TBufferStream resultBuf{};
        {
            auto builder = NSlog::NUnresolvedMeta::CreateUnresolvedMetaBuilder(numId, NMonitoring::ECompression::ZLIB, &resultBuf);
            builder->OnCommonLabels({});
            while (it->HasNext()) {
                auto record = it->Next();
                NMonitoring::TLabels labels;
                for (auto label: record.Labels) {
                    labels.Add(label.first, label.second);
                }
                builder->OnMetric(record.Type, std::move(labels), record.PointCounts, record.DataSize);
            }
            builder->Close();
        }
        return resultBuf.Buffer();
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        TString meta(buffer.Data(), buffer.size());
        ythrow NSlog::TDecodeError() << msg << " Hex: " << HexEncode(meta);
    }
}

TBuffer RepackResolvedMeta(const TBuffer& buffer) {
    try {
        TBufferInput in(buffer);
        TString meta(buffer.Data(), buffer.size());
        auto it = NSlog::NResolvedMeta::CreateResolvedMetaIterator(std::move(meta));
        auto numId = it->NumId();
        auto decimPolicy = it->DecimPolicy();
        auto producerId = it->ProducerId();
        auto producerSeqNo = it->ProducerSeqNo();

        TBufferStream resultBuf{};
        {
            auto builder = NSlog::NResolvedMeta::CreateResolvedMetaBuilder(numId, decimPolicy, producerId, producerSeqNo, NMonitoring::ECompression::LZ4, &resultBuf);
            while (it->HasNext()) {
                auto record = it->Next();
                builder->OnMetric(record.Type, record.LocalId, record.PointCounts, record.DataSize);
            }
            builder->Close();
        }
        return resultBuf.Buffer();
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        TString meta(buffer.Data(), buffer.size());
        ythrow NSlog::TDecodeError() << msg << " Hex: " << HexEncode(meta);
    }
}

TBuffer RepackData(const TBuffer& buffer) {
    TBufferInput in(buffer);
    auto it = NSlog::CreateLogDataIterator(&in);
    auto numId = it->NumId();
    auto commonTs = it->CommonTs();
    auto step = it->Step();
    auto timePrecision = it->TimePrecision();

    TBufferStream resultBuf{};
    auto builder = NSlog::CreateLogDataBuilder(
        numId,
        NSlog::EDataCodingScheme::Slog,
        timePrecision,
        NMonitoring::ECompression::ZLIB, &resultBuf);
    builder->OnCommonTime(commonTs);
    builder->OnStep(step);
    while (it->HasNext()) {
        auto record = it->Next();
        builder->OnTimeSeries(record.Kind, record.Flags, std::move(record.TimeSeries));
    }
    return buffer;
}

struct TRepackSnapshotLogDataRecord: NSlog::NSnapshotData::TSnapshotLogDataRecord {
    size_t NumPoints{0};
};

template <typename TPoint>
TRepackSnapshotLogDataRecord RepackSnapshotRecord(NSlog::NSnapshotData::TSnapshotLogDataRecord record) {
    auto columnSet = TColumnSet{static_cast<TColumnsMask>(record.ColumnSetMask)};
    NTsModel::TDecoder<TPoint> decoder(columnSet, record.Encoded);

    TRepackSnapshotLogDataRecord result;
    result.Type = record.Type;
    result.ColumnSetMask = columnSet.Mask();
    NTs::TBitBuffer buffer;
    NTs::TBitWriter writer{&buffer};
    NTsModel::TEncoder<TPoint> encoder(columnSet, &writer);
    TPoint point;
    while (decoder.NextPoint(&point)) {
        encoder.EncodePoint(point);
        result.NumPoints++;
    }
    result.Encoded = std::move(buffer);
    return result;
}

TRepackSnapshotLogDataRecord RepackSnapshotLogDataRecord(NSlog::NSnapshotData::TSnapshotLogDataRecord record) {
    switch (record.Type) {
        case NTsModel::EPointType::DGauge:
            return RepackSnapshotRecord<TGaugePoint>(std::move(record));
        case NTsModel::EPointType::Counter:
            return RepackSnapshotRecord<TCounterPoint>(std::move(record));
        case NTsModel::EPointType::Rate:
            return RepackSnapshotRecord<TRatePoint>(std::move(record));
        case NTsModel::EPointType::IGauge:
            return RepackSnapshotRecord<TIGaugePoint>(std::move(record));
        case NTsModel::EPointType::Hist:
            return RepackSnapshotRecord<THistPoint>(std::move(record));
        case NTsModel::EPointType::HistRate:
            return RepackSnapshotRecord<THistRatePoint>(std::move(record));
        case NTsModel::EPointType::DSummary:
            return RepackSnapshotRecord<TDSummaryPoint>(std::move(record));
        case NTsModel::EPointType::LogHist:
            return RepackSnapshotRecord<TLogHistPoint>(std::move(record));
        default:
            ythrow yexception() << "Unsupported metric type " << record.Type;
    }
}

TBuffer RepackSnapshotData(const TBuffer& buffer) {
    TBufferInput in(buffer);
    auto it = NSlog::NSnapshotData::CreateSnapshotLogDataIterator(&in);
    auto numId = it->NumId();

    TBufferStream resultBuf{};
    auto builder = NSlog::NSnapshotData::CreateSnapshotLogDataBuilder(numId,ECompression::ZLIB, &resultBuf);
    while (it->HasNext()) {
        auto record = RepackSnapshotLogDataRecord(it->Next());
        builder->OnTimeSeries(record.Type, record.ColumnSetMask, record.Encoded, record.NumPoints);
    }
    builder->Close();
    resultBuf.Flush();
    return resultBuf.Buffer();
}

} // namespace

jbyteArray Java_ru_yandex_solomon_SlogCodecNative_repackData(JNIEnv* jenv, jclass, jbyteArray buffer) {
    try {
        auto result = RepackData(NJava::ToBuffer(jenv, buffer));
        return NJava::ToByteArray(jenv, result);
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        jenv->ThrowNew(jenv->FindClass("java/lang/RuntimeException"), msg.c_str());
        return nullptr;
    }
}

jbyteArray Java_ru_yandex_solomon_SlogCodecNative_repackSnapshotData(JNIEnv* jenv, jclass, jbyteArray buffer) {
    try {
        auto result = RepackSnapshotData(NJava::ToBuffer(jenv, buffer));
        return NJava::ToByteArray(jenv, result);
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        jenv->ThrowNew(jenv->FindClass("java/lang/RuntimeException"), msg.c_str());
        return nullptr;
    }
}

jbyteArray Java_ru_yandex_solomon_SlogCodecNative_repackMeta(JNIEnv* jenv, jclass, jbyteArray buffer) {
    try {
        auto result = RepackMeta(NJava::ToBuffer(jenv, buffer));
        return NJava::ToByteArray(jenv, result);
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        jenv->ThrowNew(jenv->FindClass("java/lang/RuntimeException"), msg.c_str());
        return nullptr;
    }
}

jbyteArray Java_ru_yandex_solomon_SlogCodecNative_repackResolvedMeta(JNIEnv* jenv, jclass, jbyteArray buffer) {
    try {
        auto result = RepackResolvedMeta(NJava::ToBuffer(jenv, buffer));
        return NJava::ToByteArray(jenv, result);
    } catch (...) {
        TString msg = CurrentExceptionMessage();
        jenv->ThrowNew(jenv->FindClass("java/lang/RuntimeException"), msg.c_str());
        return nullptr;
    }
}
