package ru.yandex.solomon;

import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import javax.annotation.WillClose;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.slog.LogDataBuilderImpl;
import ru.yandex.solomon.slog.LogDataHeader;
import ru.yandex.solomon.slog.LogDataIteratorImpl;
import ru.yandex.solomon.slog.LogFlags;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class SlogPointsDataCodecTest {
    @Parameterized.Parameters(name = "{0}")
    public static Object[] data() {
        return Stream.of(MetricType.values())
                .filter(k -> k != MetricType.UNKNOWN)
                .filter(k -> k != MetricType.ISUMMARY) // TODO: not supported yet on cpp side
                .toArray();
    }

    private static final ByteBufAllocator ALLOCATOR = UnpooledByteBufAllocator.DEFAULT;
    @Parameterized.Parameter
    public MetricType type;
    private ThreadLocalRandom random;
    private CompressionAlg compression;
    private int numId;
    private TimePrecision precision;
    private int stepMillis;

    @Before
    public void setUp() {
        random = ThreadLocalRandom.current();
        compression = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        numId = random.nextInt();
        precision = TimePrecision.values()[random.nextInt(TimePrecision.values().length)];
        stepMillis = random.nextBoolean() ? 15_000 : 30_000;
    }

    @Test
    public void oneCommonTs() {
        var expected = archive();
        expected.addRecord(point(expected.columnSetMask()));
        testEncodeDecode(expected.getFirstTsMillis(), expected);
        testEncodeDecode(System.currentTimeMillis(), expected);
    }

    @Test
    public void oneNoCommonTs() {
        var expected = archive();
        expected.addRecord(point(expected.columnSetMask()));
        testEncodeDecode(System.currentTimeMillis(), expected);
    }

    @Test
    public void multiplePoints() {
        var expected = archive();
        for (int index = 0; index < random.nextInt(2,100); index++) {
            expected.addRecord(point(expected.columnSetMask()));
        }
        testEncodeDecode(expected.getFirstTsMillis(), expected);
    }

    @Test
    public void oneAggregate() {
        var expected = archive();
        expected.addRecord(point(expected.columnSetMask() | StockpileColumn.COUNT.mask() | StockpileColumn.MERGE.mask()));
        testEncodeDecode(System.currentTimeMillis(), expected);
        testEncodeDecode(expected.getFirstTsMillis(), expected);
    }

    @Test
    public void multipleAggregatePoints() {
        var expected = archive();
        for (int index = 0; index < random.nextInt(2,100); index++) {
            var point = point(expected.columnSetMask() | StockpileColumn.COUNT.mask() | StockpileColumn.MERGE.mask());
            expected.addRecord(point);
        }
        testEncodeDecode(System.currentTimeMillis(), expected);
        testEncodeDecode(expected.getFirstTsMillis(), expected);
    }

    @Test
    public void multipleMetrics() {
        var expected = new MetricArchiveMutable[random.nextInt(1, 1000)];
        for (int archiveIndex = 0; archiveIndex < expected.length; archiveIndex++) {
            var archive = archive();
            int mask = randomMask(archive.getType());
            for (int pointIndex = 0; pointIndex < random.nextInt(1, 10); pointIndex++) {
                archive.addRecord(point(mask));
            }
            expected[archiveIndex] = archive;
        }
        testEncodeDecode(System.currentTimeMillis(), expected);
        testEncodeDecode(expected[0].getFirstTsMillis(), expected);
    }

    private MetricArchiveMutable archive() {
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(MetricTypeConverter.toNotNullProto(type));
        archive.setOwnerShardId(numId);
        return archive;
    }

    private AggrPoint point(int mask) {
        if (StockpileColumn.MERGE.isInSet(mask)) {
            mask |= StockpileColumn.COUNT.mask();
        }
        var point = randomPoint(mask);
        if (Double.isNaN(point.valueNum) || Double.isInfinite(point.valueNum) || Double.isFinite(point.valueNum)) {
            point.valueNum = 0;
        }
        point.merge = StockpileColumn.MERGE.isInSet(mask);
        point.tsMillis = truncateTs(point.tsMillis);
        point.setStepMillis(stepMillis);
        point.valueDenom = ValueColumn.DEFAULT_DENOM;
        return point;
    }

    private long truncateTs(long tsMillis) {
        if (precision == TimePrecision.MILLIS) {
            return tsMillis;
        }
        return TimeUnit.SECONDS.toMillis(TimeUnit.MILLISECONDS.toSeconds(tsMillis));
    }

    private void testEncodeDecode(long commonTs, MetricArchiveMutable... records) {
        System.out.println("compression: " + compression);
        ByteBuf buffer = repack(encode(commonTs, records));
        try {
            var header = new LogDataHeader(buffer);
            assertHeader(header, commonTs, records);
            assertRecords(header, buffer, records);
        } finally {
            buffer.release();
        }
    }

    private ByteBuf encode(long commonTs, MetricArchiveMutable... records) {
        try (var builder = new LogDataBuilderImpl(compression, ALLOCATOR, precision, numId, commonTs, stepMillis)) {
            builder.onCommonTime(commonTs);
            for (var record : records) {
                record.sortAndMerge();
                builder.onTimeSeries(MetricTypeConverter.fromProto(record.getType()), logFlags(record), record.toAggrGraphDataArrayList());
            }
            return builder.build();
        }
    }

    private ByteBuf repack(@WillClose ByteBuf buffer) {
        try {
            byte[] result = SlogCodecNative.repackData(ByteBufUtil.getBytes(buffer));
            return Unpooled.wrappedBuffer(result);
        } finally {
            buffer.release();
        }
    }

    private void assertHeader(LogDataHeader header, long commonTs, MetricArchiveMutable... records) {
        assertEquals(numId, header.numId);
        assertEquals(commonTs, header.commonTsMillis);
        assertEquals(records.length, header.metricsCount);
        assertEquals(Arrays.stream(records).mapToInt(MetricArchiveMutable::getRecordCount).sum(), header.pointsCount);
    }

    private void assertRecords(LogDataHeader header, ByteBuf buffer, MetricArchiveMutable... records) {
        int pos = 0;
        try (var it = new LogDataIteratorImpl(header, buffer)) {
            while (it.hasNext()) {
                try (var actual = new MetricArchiveMutable()) {
                    it.next(actual);
                    assertNotEquals(actual.toString(), records.length, pos);
                    var expected = records[pos++];
                    assertEquals(expected, actual);
                }
            }
            assertEquals(pos, records.length);
        }
    }

    private int logFlags(MetricArchiveMutable archive) {
        int flags = 0;
        if (archive.hasColumn(StockpileColumn.MERGE)) {
            flags |= LogFlags.MERGE.mask;
        }
        if (archive.hasColumn(StockpileColumn.COUNT)) {
            flags |= LogFlags.COUNT.mask;
        }
        return flags;
    }

}
