package ru.yandex.solomon;

import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.WillClose;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.slog.ResolvedLogMetaBuilderImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.ResolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaRecord;
import ru.yandex.stockpile.api.EDecimPolicy;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;

/**
 * @author Vladimir Gordiychuk
 */
public class SlogResolvedMetaCodecTest {

    private static final ByteBufAllocator ALLOCATOR = UnpooledByteBufAllocator.DEFAULT;
    private CompressionAlg compression;
    private int numId;
    private EDecimPolicy decimPolicy;
    private int producerId;
    private long producerSeqNo;

    @Before
    public void setUp() {
        var random = ThreadLocalRandom.current();
        compression = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        numId = random.nextInt();
        decimPolicy = EDecimPolicy.values()[random.nextInt(EDecimPolicy.values().length - 1)];
        producerId = random.nextInt();
        producerSeqNo = random.nextLong();
    }

    @Test
    public void one() {
        var expect = record(DGAUGE);
        testEncodeDecode(expect);
    }

    @Test
    public void two() {
        var one = record(DGAUGE);
        var two = record(DGAUGE);
        testEncodeDecode(one, two);
    }

    @Test
    public void many() {
        var records = IntStream.range(0, 10_000)
                .mapToObj(ignore -> record(DGAUGE))
                .toArray(ResolvedLogMetaRecord[]::new);
        testEncodeDecode(records);
    }

    @Test
    public void differentTypes() {
        var records = Stream.of(MetricType.values())
                .filter(type -> type != MetricType.UNKNOWN)
                .filter(type -> type != MetricType.ISUMMARY) // TODO: unsupported yet on cpp side
                .map(SlogResolvedMetaCodecTest::record)
                .toArray(ResolvedLogMetaRecord[]::new);

        testEncodeDecode(records);
    }

    private static ResolvedLogMetaRecord record(MetricType type) {
        int points = ThreadLocalRandom.current().nextInt(1, 10_000);
        int bytes = ThreadLocalRandom.current().nextInt(1, 100_000);
        long localId = nextLocalId();
        return new ResolvedLogMetaRecord(type, localId, points, bytes);
    }

    private static long nextLocalId() {
        long localId;
        do {
            localId = ThreadLocalRandom.current().nextLong();
        } while (localId == 0);
        return localId;
    }

    private void testEncodeDecode(ResolvedLogMetaRecord... records) {
        System.out.println("compression: " + compression);
        ByteBuf buffer = repack(encode(records));
        try {
            var header = new ResolvedLogMetaHeader(buffer);
            assertHeader(header, records);
            assertRecords(header, buffer, records);
        } finally {
            buffer.release();
        }
    }

    private ByteBuf encode(ResolvedLogMetaRecord... records) {
        var header = new ResolvedLogMetaHeader(numId, compression)
                .setDecimPolicy(decimPolicy)
                .setProducerId(producerId)
                .setProducerSeqNo(producerSeqNo);
        try (var builder = new ResolvedLogMetaBuilderImpl(header, ALLOCATOR)) {
            for (var record : records) {
                builder.onMetric(record.type, record.localId, record.points, record.dataSize);
            }
            return builder.build();
        }
    }

    private ByteBuf repack(@WillClose ByteBuf buffer) {
        try {
            byte[] result = SlogCodecNative.repackResolvedMeta(ByteBufUtil.getBytes(buffer));
            return Unpooled.wrappedBuffer(result);
        } finally {
            buffer.release();
        }
    }

    private void assertHeader(ResolvedLogMetaHeader header, ResolvedLogMetaRecord... records) {
        assertEquals(numId, header.numId);
        assertEquals(decimPolicy, header.decimPolicy);
        assertEquals(producerId, header.producerId);
        assertEquals(producerSeqNo, header.producerSeqNo);
        assertEquals(records.length, header.metricsCount);
        assertEquals(Arrays.stream(records).mapToInt(value -> value.points).sum(), header.pointsCount);
    }

    private void assertRecords(ResolvedLogMetaHeader header, ByteBuf buffer, ResolvedLogMetaRecord... records) {
        var actual = new ResolvedLogMetaRecord();
        int pos = 0;
        try (var it = new ResolvedLogMetaIteratorImpl(header, buffer)) {
            while (it.next(actual)) {
                assertNotEquals(actual.toString(), records.length, pos);
                var expected = records[pos++];
                assertEquals(expected, actual);
            }
            assertEquals(actual.toString(), pos, records.length);
        }
    }
}
