package ru.yandex.solomon;

import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.WillClose;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.slog.SnapshotLogDataBuilderImpl;
import ru.yandex.solomon.slog.SnapshotLogDataHeader;
import ru.yandex.solomon.slog.SnapshotLogDataIteratorImpl;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class SlogSnapshotDataCodecTest {
    @Parameterized.Parameters(name = "{0}")
    public static MetricType[] supportedTypes() {
        return Stream.of(MetricType.values())
                .filter(k -> k != MetricType.UNRECOGNIZED)
                .filter(k -> k != MetricType.METRIC_TYPE_UNSPECIFIED)
                .filter(k -> k != MetricType.ISUMMARY) // TODO: not supported yet on cpp side
                .toArray(MetricType[]::new);
    }

    private static final ByteBufAllocator ALLOCATOR = UnpooledByteBufAllocator.DEFAULT;
    @Parameterized.Parameter
    public MetricType type;
    private ThreadLocalRandom random;
    private CompressionAlg compression;
    private int numId;

    @Before
    public void setUp() {
        random = ThreadLocalRandom.current();
        compression = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        numId = random.nextInt();
    }

    @Test
    public void onePointOneArchive() {
        var archive = archive();
        archive.addRecord(randomPoint(type));
        testEncodeDecode(archive);
    }

    @Test
    public void manyPointOneArchive() {
        var archive = archive();
        int mask = randomMask(type);
        for (int index = 0; index < random.nextInt(10, 100); index++) {
            archive.addRecord(randomPoint(mask));
        }
        archive.sortAndMerge();
        testEncodeDecode(archive);
    }

    @Test
    public void onePointManyArchive() {
        var archives = IntStream.range(0, random.nextInt(2, 20))
                .mapToObj(ignore -> {
                    var archive = archive();
                    archive.addRecord(randomPoint(type));
                    return archive;
                })
                .toArray(MetricArchiveMutable[]::new);
        testEncodeDecode(archives);
    }

    @Test
    public void manyPointManyArchive() {
        var archives = IntStream.range(0, random.nextInt(2, 20))
                .mapToObj(ignore -> {
                    var archive = archive();
                    int mask = randomMask(type);
                    for (int index = 0; index < random.nextInt(5, 10); index++) {
                        archive.addRecord(randomPoint(mask));
                    }
                    return archive;
                })
                .toArray(MetricArchiveMutable[]::new);
        testEncodeDecode(archives);
    }

    @Test
    public void differentTypeTogether() {
        var archives = Stream.of(supportedTypes())
                .map(t -> {
                    var archive = archive();
                    archive.setType(t);
                    var mask = randomMask(t);
                    for (int index = 0; index < random.nextInt(1, 10); index++) {
                        archive.addRecord(randomPoint(mask));
                    }
                    return archive;
                })
                .toArray(MetricArchiveMutable[]::new);
        testEncodeDecode(archives);
    }

    private MetricArchiveMutable archive() {
        MetricArchiveMutable archive = new MetricArchiveMutable();
        archive.setType(type);
        archive.setOwnerShardId(numId);
        return archive;
    }

    private void testEncodeDecode(MetricArchiveMutable... records) {
        System.out.println("compression: " + compression);
        System.out.println("mask: " + Stream.of(records)
                .map(HasColumnSet::columnSet)
                .map(StockpileColumnSet::toString)
                .collect(Collectors.joining(", ")));
        ByteBuf buffer = repack(encode(records));
        try {
            var header = new SnapshotLogDataHeader(buffer);
            assertHeader(header, records);
            assertRecords(header, buffer, records);
        } finally {
            buffer.release();
        }
    }

    private ByteBuf encode(MetricArchiveMutable... records) {
        try (var builder = new SnapshotLogDataBuilderImpl(compression, numId, ALLOCATOR)) {
            for (var record : records) {
                builder.onTimeSeries(record.toImmutableNoCopy());
            }
            return builder.build();
        }
    }

    private ByteBuf repack(@WillClose ByteBuf buffer) {
        try {
            byte[] result = SlogCodecNative.repackSnapshotData(ByteBufUtil.getBytes(buffer));
            return Unpooled.wrappedBuffer(result);
        } finally {
            buffer.release();
        }
    }

    private void assertHeader(SnapshotLogDataHeader header, MetricArchiveMutable... records) {
        assertEquals(numId, header.numId);
        assertEquals(records.length, header.metricsCount);
        assertEquals(Arrays.stream(records).mapToInt(MetricArchiveMutable::getRecordCount).sum(), header.pointsCount);
    }

    private void assertRecords(SnapshotLogDataHeader header, ByteBuf buffer, MetricArchiveMutable... records) {
        int pos = 0;
        try (var it = new SnapshotLogDataIteratorImpl(header, buffer)) {
            while (it.hasNext()) {
                try (var actual = new MetricArchiveMutable()) {
                    it.next(actual);
                    assertNotEquals(actual.toString(), records.length, pos);
                    var expected = records[pos++];
                    assertEquals(expected, actual);
                }
            }
            assertEquals(pos, records.length);
        }
    }

}
