package ru.yandex.solomon;

import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Stream;

import javax.annotation.WillClose;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.slog.UnresolvedLogMetaBuilderImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaHeader;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;

/**
 * @author Vladimir Gordiychuk
 */
public class SlogUnresolvedMetaCodecTest {

    private static final ByteBufAllocator ALLOCATOR = UnpooledByteBufAllocator.DEFAULT;
    private CompressionAlg compression;
    private int numId;

    @Before
    public void setUp() {
//        compression = CompressionAlg.values()[ThreadLocalRandom.current().nextInt(CompressionAlg.values().length)];
        compression = CompressionAlg.LZ4;
        numId = ThreadLocalRandom.current().nextInt();
    }

    @Test
    public void one() {
        var expect = record(DGAUGE, Labels.of("name", "alice"));
        testEncodeDecode(Labels.of(), expect);
        testEncodeDecode(Labels.of("host", "custom"), expect);
    }

    @Test
    public void many() {
        var alice = record(DGAUGE, Labels.of("name", "alice"));
        var bob = record(DGAUGE, Labels.of("name", "bob"));
        testEncodeDecode(Labels.of(), alice, bob);
        testEncodeDecode(Labels.of("host", "custom"), alice, bob);
    }

    @Test
    public void overrideCommon() {
        var expect = record(DGAUGE, Labels.of("name", "alice", "host", "test"));
        testEncodeDecode(Labels.of("host", "prod"), expect);
    }

    @Test
    public void differentTypes() {
        var records = Stream.of(MetricType.values())
                .filter(type -> type != MetricType.UNKNOWN)
                .filter(type -> type != MetricType.ISUMMARY) // TODO: unsupported yet on cpp side
                .map(type -> record(type, Labels.of("name", "myMetric", "type", "my"+type)))
                .toArray(UnresolvedLogMetaRecord[]::new);

        testEncodeDecode(Labels.of(), records);
        testEncodeDecode(Labels.of("host", "test"), records);
    }

    private static UnresolvedLogMetaRecord record(MetricType type, Labels labels) {
        int points = ThreadLocalRandom.current().nextInt(1, 10_000);
        int bytes = ThreadLocalRandom.current().nextInt(1, 100_000);
        return new UnresolvedLogMetaRecord(type, labels, points, bytes);
    }

    private void testEncodeDecode(Labels common, UnresolvedLogMetaRecord... records) {
        System.out.println("compression: " + compression);
        ByteBuf buffer = repack(encode(common, records));
        try {
            var header = new UnresolvedLogMetaHeader(buffer);
            assertHeader(header, records);
            assertRecords(header, buffer, common, records);
        } finally {
            buffer.release();
        }
    }

    private ByteBuf encode(Labels commonLabels, UnresolvedLogMetaRecord... records) {
        try (var builder = new UnresolvedLogMetaBuilderImpl(numId, compression, ALLOCATOR)) {
            builder.onCommonLabels(commonLabels);
            for (var record : records) {
                builder.onMetric(record.type, record.labels, record.points, record.dataSize);
            }
            return builder.build();
        }
    }

    private ByteBuf repack(@WillClose ByteBuf buffer) {
        try {
            byte[] result = SlogCodecNative.repackMeta(ByteBufUtil.getBytes(buffer));
            return Unpooled.wrappedBuffer(result);
        } finally {
            buffer.release();
        }
    }

    private void assertHeader(UnresolvedLogMetaHeader header, UnresolvedLogMetaRecord... records) {
        assertEquals(numId, header.numId);
        assertEquals(records.length, header.metricsCount);
        assertEquals(Arrays.stream(records).mapToInt(value -> value.points).sum(), header.pointsCount);
    }

    private void assertRecords(UnresolvedLogMetaHeader header, ByteBuf buffer, Labels common, UnresolvedLogMetaRecord... records) {
        var actual = new UnresolvedLogMetaRecord();
        int pos = 0;
        try (var it = new UnresolvedLogMetaIteratorImpl(header, buffer, Labels.allocator)) {
            while (it.next(actual)) {
                assertNotEquals(actual.toString(), records.length, pos);
                var expected = records[pos++];
                // common labels can be override by metric, and metric win
                expected.labels = common.addAll(expected.labels);
                assertEquals(expected, actual);
            }
            assertEquals(actual.toString(), pos, records.length);
        }
    }
}
