#include "solomon_env.h"

#include <util/charset/utf8.h>
#include <util/string/strip.h>

#include <cstdlib>

namespace NSolomon {
namespace {

template <typename T>
T EnumFromString(TString content) {
    std::transform(
            content.begin(), content.vend(), content.begin(),
            [](char c) {
                return c == '-' ? '_' : AsciiToUpper(c);
            });
    return FromString<T>(content);
}

template <typename T>
T ReadEnum(const TFsPath& path) {
    TString content = TUnbufferedFileInput{path}.ReadAll();
    StripInPlace(content);
    return EnumFromString<T>(std::move(content));
}

EDc GetSolomonDc(const TFsPath& baseDir) {
    try {
        if (const char *content = std::getenv("SOLOMON_DC")) {
            return EnumFromString<EDc>(content);
        }
    } catch (...) {
        Cerr << "Incorrect environment variable SOLOMON_DC: " << CurrentExceptionMessage() << Endl;
    }
    return ReadEnum<EDc>(baseDir / "dc");
}

EEnvironment GetSolomonEnv(const TFsPath& baseDir) {
    try {
        if (const char *content = std::getenv("SOLOMON_ENV")) {
            return EnumFromString<EEnvironment>(content);
        }
    } catch (...) {
        Cerr << "Incorrect environment variable SOLOMON_ENV: " << CurrentExceptionMessage() << Endl;
    }
    return ReadEnum<EEnvironment>(baseDir / "env");
}

TString GetSolomonFqdn() {
    return std::getenv("SOLOMON_FQDN");
}

} // namespace

TErrorOr<TSolomonEnv, TGenericError> LoadSolomonEnv(const TFsPath& baseDir) noexcept {
    try {
        return TSolomonEnv{
            .Dc = GetSolomonDc(baseDir),
            .Env = GetSolomonEnv(baseDir),
            .Fqdn = GetSolomonFqdn(),
        };
    } catch (...) {
        return TGenericError{CurrentExceptionMessage()};
    }
}

} // namespace NSolomon
