#include <solomon/libs/cpp/solomon_env/solomon_env.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>

#include <util/system/tempfile.h>

TTempFile WriteFile(TStringBuf filename, TStringBuf data) {
    TFsPath path = TFsPath{GetWorkPath()} / filename;
    TUnbufferedFileOutput out{path};
    out << data;
    return path.GetPath();
}

TEST(TSolomonEnvTest, NoFiles) {
    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_FALSE(res.Success());
    ASSERT_THAT(res.Error().Message(), testing::MatchesRegex(".*can't open \".*/dc\".*"));
}

TEST(TSolomonEnvTest, OnlyDc) {
    auto dcFile = WriteFile("dc", "sas");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_FALSE(res.Success());
    ASSERT_THAT(res.Error().Message(), testing::MatchesRegex(".*can't open \".*/env\".*"));
}

TEST(TSolomonEnvTest, OnlyEnv) {
    auto dcEnv = WriteFile("env", "PRESTABLE");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_FALSE(res.Success());
    ASSERT_THAT(res.Error().Message(), testing::MatchesRegex(".*can't open \".*/dc\".*"));
}

TEST(TSolomonEnvTest, DcAndEnv) {
    auto dcFile = WriteFile("dc", "SAS");
    auto dcEnv = WriteFile("env", "DEVELOPMENT");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::SAS);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::DEVELOPMENT);
}

TEST(TSolomonEnvTest, DcAndEnv_WithSpaces) {
    auto dcFile = WriteFile("dc", " SAS\t");
    auto dcEnv = WriteFile("env", "\tDEVELOPMENT \n");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::SAS);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::DEVELOPMENT);
}

TEST(TSolomonEnvTest, DcAndEnv_LowerCase) {
    auto dcFile = WriteFile("dc", "myt");
    auto dcEnv = WriteFile("env", "production");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::MYT);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::PRODUCTION);
}

TEST(TSolomonEnvTest, DcAndEnv_MixedCase) {
    auto dcFile = WriteFile("dc", "AsH");
    auto dcEnv = WriteFile("env", "TestiNG");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::ASH);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::TESTING);
}

TEST(TSolomonEnvTest, DcAndEnv_WithDashes) {
    auto dcFile = WriteFile("dc", "Vla");
    auto dcEnv = WriteFile("env", "cloud-prod");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::VLA);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::CLOUD_PROD);
}

TEST(TSolomonEnvTest, DcAndEnv_GPN) {
    auto dcFile = WriteFile("dc", "Man");
    auto dcEnv = WriteFile("env", "cloud-GPN-1");

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::MAN);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::CLOUD_GPN_1);
}

TEST(TSolomonEnvTest, DataFromEnv) {
    setenv("SOLOMON_ENV", "dEvElOpMeNt", 1);
    setenv("SOLOMON_DC", "iva", 1);
    setenv("SOLOMON_FQDN", "apuw4glqjmyhv62w.sas.yp-c.yandex.net", 1);

    auto res = NSolomon::LoadSolomonEnv();
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::IVA);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::DEVELOPMENT);
    ASSERT_EQ(env.Fqdn, "apuw4glqjmyhv62w.sas.yp-c.yandex.net");
}

TEST(TSolomonEnvTest, IncorrectEnv) {
    auto dcFile = WriteFile("dc", "Man");
    auto dcEnv = WriteFile("env", "cloud-GPN-1");
    setenv("SOLOMON_ENV", "some bad env", 1);
    setenv("SOLOMON_DC", "incorrect dc", 1);

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::MAN);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::CLOUD_GPN_1);
}

TEST(TSolomonEnvTest, EnvVariablesAreMoreMeaningful) {
    auto dcFile = WriteFile("dc", "Man");
    auto dcEnv = WriteFile("env", "cloud-GPN-1");
    setenv("SOLOMON_ENV", "preStaBle", 1);
    setenv("SOLOMON_DC", "sas", 1);

    auto res = NSolomon::LoadSolomonEnv(GetWorkPath());
    ASSERT_TRUE(res.Success()) << res.Error().Message();

    auto env = res.Extract();
    ASSERT_EQ(env.Dc, NSolomon::EDc::SAS);
    ASSERT_EQ(env.Env, NSolomon::EEnvironment::PRESTABLE);
}
