#include "codec.h"

namespace NSolomon::NStockpile {
namespace {

constexpr int MaxVarInt64Bytes = 10;
constexpr int MaxVarInt32Bytes = 5;

std::optional<ui32> ReadVarInt32Fast(const ui8* data, size_t& pos) {
    ui32 b = data[pos++];
    ui32 result = b - 0x80;

    b = data[pos++];
    result += b <<  7;
    if ((b & 0x80) == 0) {
        return result;
    }

    result -= 0x80 << 7;
    b = data[pos++];
    result += b << 14;
    if ((b & 0x80) == 0) {
        return result;
    }

    result -= 0x80 << 14;
    b = data[pos++];
    result += b << 21;
    if ((b & 0x80) == 0) {
        return result;
    }

    result -= 0x80 << 21;
    b = data[pos++];
    result += b << 28;
    if ((b & 0x80) == 0) {
        return result;
    }

    return std::nullopt;
}

std::optional<ui64> ReadVarInt64Fast(const ui8* data, size_t& pos) {
    ui32 part0 = 0, part1 = 0, part2 = 0;

    ui32 b = data[pos++];
    part0 = b;
    if (!(b & 0x80)) goto done;

    part0 -= 0x80;
    b = data[pos++];
    part0 += b << 7;
    if (!(b & 0x80)) goto done;

    part0 -= 0x80 << 7;
    b = data[pos++];
    part0 += b << 14;
    if (!(b & 0x80)) goto done;

    part0 -= 0x80 << 14;
    b = data[pos++];
    part0 += b << 21;
    if (!(b & 0x80)) goto done;

    part0 -= 0x80 << 21;
    b = data[pos++];
    part1 = b;
    if (!(b & 0x80)) goto done;

    part1 -= 0x80;
    b = data[pos++];
    part1 += b <<  7;
    if (!(b & 0x80)) goto done;

    part1 -= 0x80 << 7;
    b = data[pos++];
    part1 += b << 14;
    if (!(b & 0x80)) goto done;

    part1 -= 0x80 << 14;
    b = data[pos++];
    part1 += b << 21;
    if (!(b & 0x80)) goto done;

    part1 -= 0x80 << 21;
    b = data[pos++];
    part2  = b;
    if (!(b & 0x80)) goto done;

    part2 -= 0x80;
    b = data[pos++];
    part2 += b << 7;
    if (!(b & 0x80)) goto done;

    // overrun the maximum size of a varint (10 bytes). Assume the data is corrupt.
    return std::nullopt;

    done:
    ui64 value =
            (static_cast<ui64>(part0)) |
            (static_cast<ui64>(part1) << 28) |
            (static_cast<ui64>(part2) << 56);
    return value;
}

std::optional<ui64> ReadVarInt64Slow(const ui8* data, size_t size, size_t& pos) {
    ui64 result = 0;
    int count = 0;
    ui8 b;

    do {
        if (count == MaxVarInt64Bytes || pos == size) {
            return std::nullopt;
        }

        b = data[pos++];
        result |= static_cast<ui64>(b & 0x7F) << (7 * count);
        ++count;
    } while (b & 0x80);

    return result;
}

template <typename T>
void WriteVarInt(T value, TBuffer* out) {
    while (value >= 0x80) {
        out->Append(static_cast<char>(value | 0x80));
        value >>= 7;
    }
    out->Append(static_cast<char>(value));
}

} // namespace


std::optional<ui32> TCodecInput::ReadVarInt32() noexcept {
    if (AtEnd()) {
        return std::nullopt;
    }

    // most common scenario - just one byte
    if (ui32 b = Data_[Pos_]; b < 0x80) {
        ++Pos_;
        return b;
    }

    // have enough bytes in buffer, no need to check border every time
    if (Left() >= MaxVarInt32Bytes) {
        return ReadVarInt32Fast(Data_, Pos_);
    }

    // slow path
    auto result = ReadVarInt64Slow(Data_, Size_, Pos_);
    if (result && *result <= std::numeric_limits<ui32>::max()) {
        return static_cast<ui32>(*result);
    }

    return std::nullopt;
}

std::optional<ui64> TCodecInput::ReadVarInt64() noexcept {
    if (AtEnd()) {
        return std::nullopt;
    }

    // most common scenario - just one byte
    if (ui64 b = Data_[Pos_]; b < 0x80) {
        ++Pos_;
        return b;
    }

    // have enough bytes in buffer, no need to check border every time
    if (Left() >= MaxVarInt64Bytes) {
        return ReadVarInt64Fast(Data_, Pos_);
    }

    // slow path
    return ReadVarInt64Slow(Data_, Size_, Pos_);
}

void TCodecOutput::WriteVarInt32(ui32 value) {
    WriteVarInt(value, &Buf_);
}

void TCodecOutput::WriteVarInt64(ui64 value) {
    WriteVarInt(value, &Buf_);
}

} // namespace NSolomon::NStockpile
