#pragma once

#include <util/generic/buffer.h>
#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>
#include <util/system/unaligned_mem.h>

#include <optional>

namespace NSolomon::NStockpile {

class TCodecException: public yexception {
};

#define STOCKPILE_CODEC_ENSURE(condition, message) \
    Y_ENSURE_EX(condition, ::NSolomon::NStockpile::TCodecException{} << message)

class TCodecInput {
public:
    TCodecInput(const void* data, size_t size) noexcept
        : Data_{reinterpret_cast<const ui8*>(data)}
        , Size_{size}
        , Pos_{0}
    {
    }

    explicit TCodecInput(TStringBuf buf) noexcept
        : TCodecInput(buf.data(), buf.size())
    {
    }

    explicit TCodecInput(const TBuffer& buf) noexcept
        : TCodecInput(buf.data(), buf.size())
    {
    }

    void Read(void* data, size_t size) noexcept {
        Y_ASSERT(Pos_ + size <= Size_);
        memcpy(data, Data_ + Pos_, size);
        Pos_ += size;
    }

    template <typename T, typename = std::enable_if_t<std::is_fundamental_v<T>>>
    T ReadFixed() noexcept {
        Y_ASSERT(Pos_ + sizeof(T) <= Size_);
        auto value = ReadUnaligned<T>(Data_ + Pos_);
        Pos_ += sizeof(T);
        return value;
    }

    std::optional<ui32> ReadVarInt32() noexcept;

    std::optional<ui64> ReadVarInt64() noexcept;

    size_t Left() const noexcept {
        return Size_ - Pos_;
    }

    bool AtEnd() const noexcept {
        return Pos_ == Size_;
    }

private:
    const ui8* Data_;
    const size_t Size_;
    size_t Pos_;
};

class TCodecOutput {
public:
    explicit TCodecOutput(size_t capacity)
        : Buf_{capacity}
    {
    }

    void Write(const void* data, size_t size) {
        Buf_.Append(reinterpret_cast<const char*>(data), size);
    }

    template <typename T, typename = std::enable_if_t<std::is_fundamental_v<T>>>
    void WriteFixed(T value) {
        Write(&value, sizeof(value));
    }

    void WriteVarInt32(ui32 value);

    void WriteVarInt64(ui64 value);

    TBuffer TakeBuffer() {
        return std::move(Buf_);
    }

private:
    TBuffer Buf_;
};

template <typename T>
class ICodec {
public:
    virtual void Encode(const T& value, TCodecOutput* out) = 0;
    virtual T Decode(TCodecInput* in) = 0;
};

} // namespace NSolomon::NStockpile
