#include "metric_archive.h"

#include <solomon/libs/cpp/ts_codec/bits.h>

namespace NSolomon::NStockpile {

void TMetricArchiveCodec::Encode(const TMetricArchive& archive, TCodecOutput* out) {
    // (1) header
    TMetricHeaderCodec{}.Encode(archive.Header(), out);

    // (2) emptiness flag
    out->WriteFixed<ui8>(archive.PointCount() == 0 ? 0 : 1);  // 0 - empty, 1 - non empty
    if (archive.PointCount() == 0) {
        return;
    }

    // (3) point count
    out->WriteVarInt32(archive.PointCount());

    // (4) columns
    TColumnSetCodec{}.Encode(archive.Columns(), out);

    // (5) compressed data
    NTs::TBitSpan data = archive.Data();
    out->WriteFixed(data.Size());
    out->Write(data.Data(), data.SizeBytes());
}

TMetricArchive TMetricArchiveCodec::Decode(TCodecInput* in) {
    // (1) metric header
    TMetricHeader header = TMetricHeaderCodec{}.Decode(in);

    // (2) is empty flag
    STOCKPILE_CODEC_ENSURE(in->Left() >= 1, "cannot read emptiness flag");
    if (in->ReadFixed<ui8>() == 0) {  // 0 - empty, 1 - non empty
        return TMetricArchive{header, Format_};
    }

    // (3) point count
    auto pointCount = in->ReadVarInt32();
    STOCKPILE_CODEC_ENSURE(pointCount, "cannot read point count");

    // (4) columns
    NTs::TColumnSet columns = TColumnSetCodec{}.Decode(in);

    // (5) compressed data
    STOCKPILE_CODEC_ENSURE(in->Left() >= sizeof(ui64), "cannot read data size");
    ui64 dataSizeBits = in->ReadFixed<ui64>();
    size_t dataSize = NTs::ByteCount(dataSizeBits);

    STOCKPILE_CODEC_ENSURE(
            in->Left() >= dataSize,
            "not enough bytes to read: left(" << in->Left() << ") < dataSize(" << dataSize << ')');

    NTs::TBitBuffer data;
    data.Resize(dataSizeBits);
    in->Read(data.Data(), dataSize);

    return TMetricArchive{header, Format_, columns, *pointCount, std::move(data)};
}

} // namespace NSolomon::NStockpile
