#include "metric_header.h"

#include <solomon/libs/cpp/ts_codec/bits.h>

#include <string_view>

namespace NSolomon::NStockpile {
namespace {

constexpr int MinHeaderBytes = 5;
constexpr int DeleteBeforeFlag = 1;

} // namespace

void TMetricHeaderCodec::Encode(const TMetricHeader& header, TCodecOutput* out) {
    // (1) delete before
    ui8 flags = 0;
    if (header.DeleteBefore != TInstant::Zero()) {
        flags = NTs::BitSet<DeleteBeforeFlag>(flags);
        out->WriteFixed<ui8>(flags);
        out->WriteVarInt64(header.DeleteBefore.MilliSeconds());
    } else {
        out->WriteFixed<ui8>(flags);
    }

    // (2) project id
    out->WriteVarInt32(header.Owner.ProjectId);

    // (3) shard id
    out->WriteVarInt32(header.Owner.ShardId);

    // (4) decim policy id
    out->WriteVarInt32(header.DecimPolicyId);

    // (5) metric type
    out->WriteVarInt32(static_cast<ui32>(header.Type));
}

TMetricHeader TMetricHeaderCodec::Decode(TCodecInput* in) {
    STOCKPILE_CODEC_ENSURE(in->Left() >= MinHeaderBytes, "cannot read metric header, left " << in->Left() << " bytes");

    // (1) delete before
    ui8 flags = in->ReadFixed<ui8>();
    TInstant deleteBefore;
    if (NTs::BitTest<DeleteBeforeFlag>(flags)) {
        auto millis = in->ReadVarInt64();
        STOCKPILE_CODEC_ENSURE(millis, "cannot read delete before millis");
        deleteBefore = TInstant::MilliSeconds(*millis);
    }

    // (2) project id
    auto projectId = in->ReadVarInt32();
    STOCKPILE_CODEC_ENSURE(projectId, "cannot read project id");

    // (3) shard id
    auto shardId = in->ReadVarInt32();
    STOCKPILE_CODEC_ENSURE(shardId, "cannot read shard id");

    // (4) decim policy id
    auto decimPolicyId = in->ReadVarInt32();
    STOCKPILE_CODEC_ENSURE(decimPolicyId, "cannot read decim policy id");

    // (5) metric type
    auto type = in->ReadVarInt32();
    STOCKPILE_CODEC_ENSURE(type, "cannot read metric type");
    STOCKPILE_CODEC_ENSURE(yandex::solomon::model::MetricType_IsValid(*type), "invalid metric type: " << *type);

    return TMetricHeader{
        TMetricOwner{*projectId, *shardId},
        deleteBefore,
        *decimPolicyId,
        static_cast<yandex::solomon::model::MetricType>(*type)};
}

} // namespace NSolomon::NStockpile

template <>
void Out<NSolomon::NStockpile::TMetricHeader>(IOutputStream& out, const NSolomon::NStockpile::TMetricHeader& mh) {
    using namespace std::literals;

    out << "{projectId: "sv << mh.Owner.ProjectId;
    out << ", shardId: "sv << mh.Owner.ShardId;
    out << ", deleteBefore: "sv << mh.DeleteBefore;
    out << ", decimPolicyId: "sv << mh.DecimPolicyId;
    out << ", type: "sv << yandex::solomon::model::MetricType_Name(mh.Type);
    out << '}';
}
