#include "java_objects.h"

namespace NSolomon::NJava {

TBuffer ToBuffer(JNIEnv* jenv, jbyteArray array) {
    jbyte* data = jenv->GetByteArrayElements(array, nullptr);
    jsize size = jenv->GetArrayLength(array);
    TBuffer buffer(reinterpret_cast<const char*>(data), static_cast<size_t>(size));
    jenv->ReleaseByteArrayElements(array, data, JNI_ABORT);
    return buffer;
}

jbyteArray ToByteArray(JNIEnv* jenv, const TBuffer& buf) {
    jbyteArray array = jenv->NewByteArray(buf.size());
    jenv->SetByteArrayRegion(array, 0, buf.size(), reinterpret_cast<const jbyte*>(buf.data()));
    return array;
}

jbyteArray ToByteArray(JNIEnv* jenv, const NTs::TBitSpan data) {
    jbyteArray result = jenv->NewByteArray(data.SizeBytes());
    jenv->SetByteArrayRegion(result, 0, data.SizeBytes(), reinterpret_cast<const jbyte*>(data.Data()));
    return result;
}

TByteArray::TByteArray(JNIEnv* jenv, jbyteArray array)
        : Jenv_{jenv}
        , Array_{array}
        , Data_{jenv->GetByteArrayElements(array, nullptr)}
        , Size_{jenv->GetArrayLength(array)}
{
    Y_ENSURE(Data_, "cannot get array elements");
}

TByteArray::~TByteArray() {
    Jenv_->ReleaseByteArrayElements(Array_, Data_, JNI_ABORT);
}

TJavaClass::TJavaClass(JNIEnv* jenv, const char* className)
        : Jenv_{jenv}
        , Class_{jenv->FindClass(className)}
{
    Y_ENSURE(Class_, "cannot find class " << className);
}

TJavaClass::~TJavaClass() {
    Jenv_->DeleteLocalRef(Class_);
}

jmethodID TJavaClass::GetStaticMethod(const char* name, const char* sig) {
    jmethodID method = Jenv_->GetStaticMethodID(Class_, name, sig);
    Y_ENSURE(method, "cannot get static method " << name);
    return method;
}

jmethodID TJavaClass::GetMethod(const char* name, const char* sig) {
    jmethodID method = Jenv_->GetMethodID(Class_, name, sig);
    Y_ENSURE(method, "cannot get method " << name);
    return method;
}

jfieldID TJavaClass::GetField(const char* name, const char* sig) {
    jfieldID field = Jenv_->GetFieldID(Class_, name, sig);
    Y_ENSURE(field, "cannot get field " << name);
    return field;
}

TMetricTypeClass::TMetricTypeClass(JNIEnv* jenv)
        : TJavaClass(jenv, "ru/yandex/solomon/model/protobuf/MetricType")
        , ForNumberMethod_{GetStaticMethod("forNumber", "(I)Lru/yandex/solomon/model/protobuf/MetricType;")}
        , GetNumberMethod_{GetMethod("getNumber", "()I")}
{
}

jobject TMetricTypeClass::ForNumber(jint number) {
    return Jenv_->CallStaticObjectMethod(Class_, ForNumberMethod_, number);
}

jint TMetricTypeClass::GetNumber(jobject type) {
    return Jenv_->CallIntMethod(type, GetNumberMethod_);
}

TMetricHeaderClass::TMetricHeaderClass(JNIEnv *jenv)
        : TJavaClass(jenv, "ru/yandex/solomon/codec/archive/header/MetricHeader")
        , Constructor_{GetMethod("<init>", "(JIIILru/yandex/solomon/model/protobuf/MetricType;)V")}
        , DeleteBeforeField_{GetField("deleteBefore", "J")}
        , ProjectIdField_{GetField("ownerProjectId", "I")}
        , ShardIdField_{GetField("ownerShardId", "I")}
        , DecimPolicyIdField_{GetField("decimPolicyId", "I")}
        , TypeField_{GetField("type", "Lru/yandex/solomon/model/protobuf/MetricType;")}
{
}

jobject TMetricHeaderClass::New(jlong deleteBeforeMillis, jint projectId, jint shardId, jint decimPolicyId, jobject type) {
    return Jenv_->NewObject(Class_, Constructor_, deleteBeforeMillis, projectId, shardId, decimPolicyId, type);
}

TMetricHeaderObject TMetricHeaderClass::Convert(jobject headerObject) {
    TMetricHeaderObject header;
    header.DeleteBefore = Jenv_->GetLongField(headerObject, DeleteBeforeField_);
    header.ProjectId = Jenv_->GetIntField(headerObject, ProjectIdField_);
    header.ShardId = Jenv_->GetIntField(headerObject, ShardIdField_);
    header.DecimPolicyId = Jenv_->GetIntField(headerObject, DecimPolicyIdField_);
    header.Type = Jenv_->GetObjectField(headerObject, TypeField_);
    return header;
}

TMetricArchiveClass::TMetricArchiveClass(JNIEnv* jenv)
        : TJavaClass(jenv, "ru/yandex/solomon/codec/archive/MetricArchiveImmutable")
        , Constructor_{GetMethod("<init>",
                "("
                     "Lru/yandex/solomon/codec/archive/header/MetricHeader;"
                     "Lru/yandex/solomon/codec/serializer/StockpileFormat;"
                     "I"
                     "Lru/yandex/solomon/codec/bits/BitBuf;"
                     "I"
                ")V")}
        , DeleteBeforeField_{GetField("deleteBefore", "J")}
        , ProjectIdField_{GetField("ownerProjectId", "I")}
        , ShardIdField_{GetField("ownerShardId", "I")}
        , DecimPolicyIdField_{GetField("decimPolicyId", "I")}
        , TypeField_{GetField("type", "Lru/yandex/solomon/model/protobuf/MetricType;")}
        , ColumnSetMaskField_{GetField("columnSetMask", "I")}
        , BufferField_{GetField("buffer", "Lru/yandex/solomon/codec/bits/BitBuf;")}
        , RecordCountField_{GetField("recordCount", "I")}
        , FormatField_{GetField("format", "Lru/yandex/solomon/codec/serializer/StockpileFormat;")}
{
}

jobject TMetricArchiveClass::New(jobject header, jobject format, jint columnSetMask, jobject buffer, jint recordCount) {
    return Jenv_->NewObject(Class_, Constructor_, header, format, columnSetMask, buffer, recordCount);
}

TMetricArchiveObject TMetricArchiveClass::Convert(jobject archiveObject) {
    TMetricArchiveObject archive;
    archive.Header.DeleteBefore = Jenv_->GetLongField(archiveObject, DeleteBeforeField_);
    archive.Header.ProjectId = Jenv_->GetIntField(archiveObject, ProjectIdField_);
    archive.Header.ShardId = Jenv_->GetIntField(archiveObject, ShardIdField_);
    archive.Header.DecimPolicyId = Jenv_->GetIntField(archiveObject, DecimPolicyIdField_);
    archive.Header.Type = Jenv_->GetObjectField(archiveObject, TypeField_);

    archive.ColumnSetMask = Jenv_->GetIntField(archiveObject, ColumnSetMaskField_);
    archive.Buffer = Jenv_->GetObjectField(archiveObject, BufferField_);
    archive.RecordCount = Jenv_->GetIntField(archiveObject, RecordCountField_);
    archive.Format = Jenv_->GetObjectField(archiveObject, FormatField_);
    return archive;
}

TStockpileFormatClass::TStockpileFormatClass(JNIEnv* jenv)
        : TJavaClass(jenv, "ru/yandex/solomon/codec/serializer/StockpileFormat")
        , FormatField_{GetField("format", "I")}
{
}

jint TStockpileFormatClass::Convert(jobject format) {
    return Jenv_->GetIntField(format, FormatField_);
}

THeapBitBufClass::THeapBitBufClass(JNIEnv* jenv)
        : TJavaClass(jenv, "ru/yandex/solomon/codec/bits/HeapBitBuf")
        , Constructor_{GetMethod("<init>", "([BJ)V")}
        , ArrayField_{GetField("array", "[B")}
        , WriteIndexField_{GetField("writeIndex", "J")}
        , ReadIndexField_{GetField("readIndex", "J")}
{
}

jobject THeapBitBufClass::New(jbyteArray array, jlong lengthBits) {
    return Jenv_->NewObject(Class_, Constructor_, array, lengthBits);
}

THeapBitBufObject THeapBitBufClass::Convert(jobject bitBufObject) {
    THeapBitBufObject bitBuf;
    jobject arrayObject = Jenv_->GetObjectField(bitBufObject, ArrayField_);
    bitBuf.Array = *reinterpret_cast<jbyteArray*>(&arrayObject);
    bitBuf.WriteIndex = Jenv_->GetLongField(bitBufObject, WriteIndexField_);
    bitBuf.ReadIndex = Jenv_->GetLongField(bitBufObject, ReadIndexField_);
    return bitBuf;
}

} // namespace NSolomon::NJava
